/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.chart.ui;

import jp.sf.orangesignal.chart.util.StringManager;
import jp.sf.orangesignal.ta.MovingAverage;
import jp.sf.orangesignal.ta.TechnicalAnalysis;

/**
 * 移動平均の種類を表す列挙型を提供します。
 * 
 * @author 杉澤 浩二
 */
public enum MAType {

	/**
	 * 単純移動平均 (Simple Moving Average) を表します。
	 */
	SMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.SMA; }
		@Override public String toString() { return StringManager.getString("ma.sma"); }
	},

	/**
	 * 平滑移動平均 (Smoothed Moving Average) を表します。
	 */
	SMMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.SMMA; }
		@Override public String toString() { return StringManager.getString("ma.smma"); }
	},

	/**
	 * 修正移動平均 (Running Moving Average) を表します。
	 */
	RMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.RMA; }
		@Override public String toString() { return StringManager.getString("ma.rma"); }
	},

	/**
	 * 加重移動平均 (Weighted Moving Average) を表します。
	 */
	WMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.WMA; }
		@Override public String toString() { return StringManager.getString("ma.wma"); }
	},

	/**
	 * ハル移動平均 (Hull's Moving Average) を表します。
	 */
	HMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.HMA; }
		@Override public String toString() { return StringManager.getString("ma.hma"); }
	},

	/**
	 * 三角移動平均 (Triangular Moving Average) を表します。
	 */
	TMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.TMA; }
		@Override public String toString() { return StringManager.getString("ma.tma"); }
	},

	/**
	 * 指数平滑移動平均 (Exponential Moving Average) を表します。
	 */
	EMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.EMA; }
		@Override public String toString() { return StringManager.getString("ma.ema"); }
	},

	/**
	 * 二重指数平滑移動平均 (Double Exponential Moving Average) を表します。
	 */
	DEMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.DEMA; }
		@Override public String toString() { return StringManager.getString("ma.dema"); }
	},

	/**
	 * 三重指数平滑移動平均 (Triple Exponential Moving Average) を表します。
	 */
	TEMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.TEMA; }
		@Override public String toString() { return StringManager.getString("ma.tema"); }
	},

	/**
	 * 零ラグ指数平滑移動平均 (Zero Lag Exponential Moving Average) を表します。
	 */
	ZLEMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.ZLEMA; }
		@Override public String toString() { return StringManager.getString("ma.zlema"); }
	},

	/**
	 * エンドポイント移動平均 (Endpoint Moving Average) を表します。
	 */
	EPMA {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.EPMA; }
		@Override public String toString() { return StringManager.getString("ma.epma"); }
	},

	/**
	 * ティルソン T3 移動平均 (Tillson's T3 Moving Average) を表します。
	 */
	T3 {
		@Override public MovingAverage getMovingAverage() { return MovingAverage.T3; }
		@Override public String toString() { return StringManager.getString("ma.t3"); }
	};

	/**
	 * 移動平均の種類を表す列挙型を返します。
	 * 
	 * @return 移動平均の種類を表す列挙型
	 */
	public abstract MovingAverage getMovingAverage();

	/**
	 * 指定された移動平均法で移動平均を計算して返します。
	 * 
	 * @param data データ
	 * @param period 期間(当日を含む)
	 * @return 移動平均
	 */
	public Number[] calculate(final Number[] data, final int period) {
		return TechnicalAnalysis.ma(data, period, this.getMovingAverage());
	}

}
