/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 */

package jp.sourceforge.orangesignal.trading.commission;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 体系化手数料情報を提供します。
 * 
 * @author 杉澤 浩二
 */
public class CommissionSystem implements Commission, Serializable {

	private static final long serialVersionUID = 6836911131954296597L;

	/**
	 * 手数料情報のリストを保持します。
	 */
	private List<CommissionUnit> commissionList = new ArrayList<CommissionUnit>();

	/**
	 * デフォルトコンストラクタです。
	 */
	public CommissionSystem() {}

	/**
	 * コンストラクタです。
	 * 
	 * @param unit 手数料情報
	 */
	public CommissionSystem(final CommissionUnit unit) {
		addCommission(unit);
	}

	/**
	 * 手数料情報のリストからこのクラスを構築するコンストラクタです。
	 * 
	 * @param list 手数料情報のリスト
	 */
	public CommissionSystem(final List<CommissionUnit> list) {
		setCommissionList(list);
	}

	/**
	 * 手数料情報のリストを返します。
	 * 
	 * @return 手数料情報のリスト
	 */
	public List<CommissionUnit> getCommissionList() {
		return commissionList;
	}

	/**
	 * 手数料情報のリストを設定します。
	 * 
	 * @param commissionList 手数料情報のリスト
	 */
	public void setCommissionList(final List<CommissionUnit> commissionList) {
		this.commissionList = new ArrayList<CommissionUnit>();
		this.commissionList.addAll(commissionList);
        Collections.sort(this.commissionList);
	}

	/**
	 * 指定された手数料情報を手数料体系へ追加します。
	 * 
	 * @param unit 手数料情報
	 */
	public void addCommission(final CommissionUnit unit) {
		// FIXME: 同じ基準値のものがあったらエラーとする
		commissionList.add(unit);
        Collections.sort(commissionList);
	}

	@Override
	public double calcCommission(final double price, final int quantity) {
		return getCommission(price, quantity).calcCommission(price, quantity);
	}

	/**
	 * 指定された約定金額に対応する手数料情報を返します。
	 * 
	 * @param amount 約定金額
	 * @return 約定金額に対応する手数料情報
	 */
	@SuppressWarnings("serial")
	private CommissionUnit getCommission(final double price, final int quantity) {
		return getCommission(
				new CommissionUnit(price * quantity) {
					@Override public double calcCommission(final double price, final int quantity) { return 0; }
				});
	}

	private CommissionUnit getCommission(final CommissionUnit commission) {
		int index = Collections.binarySearch(commissionList, commission);
		if (index >= 0)
			return commissionList.get(index);

		index = -(index + 1);
		return commissionList.get(Math.min(index, commissionList.size() - 1));
	}

}
