/*

Copyright (c) 2003-2006, AXE, Inc.  All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
/*
 2007/1 Modefied by BBR Inc.
  Changed for opvp version 1.0
*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<time.h>
#include	<sys/time.h>
#include	<string.h>
#include	"opvp.h"
#include	"opvp_null.h"

/* macros */
#define InitBrush       { \
                                OPVP_CSPACE_STANDARDRGB, \
                                {255,255,255,255}, \
                                0, \
                                0, \
                                NULL \
                        }


/* definition */
typedef	struct {
	opvp_point_t	pos;
	opvp_brush_t	sCol;
	opvp_brush_t	fCol;
	opvp_brush_t	bCol;
	opvp_linestyle_t	lStyle;
	opvp_linecap_t	lCap;
	opvp_linejoin_t	lJoin;
	opvp_fix_t	mLimit;
	opvp_paintmode_t	pMode;
} GS;
typedef	struct	_GS_stack {
	struct	_GS_stack	*prev;
	struct	_GS_stack	*next;
	GS			gs;
} GS_stack;

/* global variables */
opvp_int_t	opvpErrorNo = OPVP_OK;

/* private variables */
static	opvp_int_t	outputStream = -1;
static	opvp_dc_t	pContext = -1;
static	opvp_ctm_t	ctm = {1,0,0,1,0,0};
static	opvp_api_procs_t	apiList;
static	opvp_cspace_t	csList[3] =
			{
				OPVP_CSPACE_STANDARDRGB,
				OPVP_CSPACE_DEVICEGRAY,
				OPVP_CSPACE_BW
			};
static	opvp_cspace_t	colorSpace = OPVP_CSPACE_STANDARDRGB;
static	opvp_fillmode_t	fill = OPVP_FILLMODE_WINDING;
static	opvp_float_t	alphaConst = 0;
static	opvp_fix_t	lineWidth = 0;
static	opvp_int_t	dashNum = 0;
static	opvp_fix_t	*dash = (opvp_fix_t*)NULL;
static	opvp_fix_t	dashOffset = 0;
static	opvp_cliprule_t	clip = OPVP_CLIPRULE_EVENODD;
static	GS		currentGS = {
				     {0,0},InitBrush,InitBrush,InitBrush,
				     OPVP_LINESTYLE_SOLID,OPVP_LINECAP_BUTT,
				     OPVP_LINEJOIN_MITER,0,
				     OPVP_PAINTMODE_OPAQUE
				    };
static	GS_stack	*GSstack = NULL;

static
void SetApiList(void);


#define	BUF_SIZE	1024

static
void Output(char *fmt,...)
{
#ifdef	DEBUG_PRINT
	va_list	ap;
	int	d;
	char	c, *s;
	float	f;
	FILE	*fp = stderr;

	va_start(ap, fmt);

	while (*fmt) {
		switch (*fmt) {
			case '%':
				switch (*(++fmt)) {
					case 'c':
						c = (char)va_arg(ap, int);
						fprintf(fp, "%c", c);
						break;
					case 's':
						s = va_arg(ap, char *);
						fprintf(fp, "%s", s);
						break;
					case 'd':
						d = va_arg(ap, int);
						fprintf(fp, "%d", d);
						break;
					case 'f':
						f = (opvp_float_t)va_arg(ap, double);
						fprintf(fp, "%f", f);
						break;
					case 'F':
						d = va_arg(ap, int);
						f = (opvp_float_t)d/(opvp_float_t)OPVP_FIX_FRACT_DENOM;
						fprintf(fp, "%f", f);
						break;
					case 'x':
						d = va_arg(ap, int);
						fprintf(fp, "%x", d);
						break;
					case 'X':
						d = va_arg(ap, int);
						fprintf(fp, "%X", d);
						break;
					default:
						fprintf(fp, "%%");
						break;
				}
				break;
			default:
				fputc((int)*fmt, fp);
				break;
		}
		fmt++;
	}

	va_end(ap);
#endif
}

/*
 * ------------------------------------------------------------------------
 * Creating and Managing Print Contexts
 * ------------------------------------------------------------------------
 */

/*
 * OpenPrinter
 */
opvp_dc_t	opvpOpenPrinter(
	opvp_int_t	outputFD,
	const opvp_char_t *printerModel,
	const opvp_int_t apiVersion[2],
	struct _opvp_api_procs **apiProcs
	)
{
	opvpErrorNo = OPVP_OK;

	Output("OpenPrinter:outputFD=%d",outputFD);
	Output(":version %d.%d",apiVersion[0],apiVersion[1]);
	if (printerModel == NULL) {
	  Output(":printerModel=NULL");
	} else {
	  Output(":printerModel=%s",printerModel);
	}
	Output(":printerContext=%x\n",pContext);


	/* support opvp Version 1.0 */
	if (apiVersion[0] != 1 || apiVersion[1] != 0) {
		opvpErrorNo = OPVP_VERSIONERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	if (pContext != -1) {
		opvpErrorNo = OPVP_BADREQUEST;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	pContext = 1;
	OPVP_I2FIX(1,lineWidth);

	outputStream = outputFD;

	SetApiList();
	*apiProcs = &apiList;

	return pContext;
}

/*
 * ClosePrinter
 */
static
opvp_result_t	opvpClosePrinter(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("ClosePrinter:printerContext=%d\n",printerContext);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}


	outputStream = -1;
	pContext = -1;

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Job Control Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartJob
 */
static
opvp_result_t	opvpStartJob(
	opvp_dc_t	printerContext,
	const opvp_char_t *jobInfo
	)
{
	opvpErrorNo = OPVP_OK;

	Output("StartJob:printerContext=%d",printerContext);
	Output(":jobInfo=%s\n",jobInfo);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * EndJob
 */
static
opvp_result_t	opvpEndJob(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("EndJob:printerContext=%d\n",printerContext);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * AbortJob
 */
static
opvp_result_t	opvpAbortJob(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("AbortJob:printerContext=%d\n",printerContext);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * StartDoc
 */
static
opvp_result_t	opvpStartDoc(
	opvp_dc_t	printerContext,
	const opvp_char_t *docInfo
	)
{
	opvpErrorNo = OPVP_OK;

	Output("StartDoc:printerContext=%d",printerContext);
	Output(":docInfo=%s\n",docInfo);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * EndDoc
 */
static
opvp_result_t	opvpEndDoc(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("EndDoc:printerContext=%d\n",printerContext);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * StartPage
 */
static
opvp_result_t	opvpStartPage(
	opvp_dc_t	printerContext,
	const opvp_char_t *pageInfo
	)
{
	opvpErrorNo = OPVP_OK;

	Output("StartPage:printerContext=%d",printerContext);
	Output(":pageInfo=%s\n",pageInfo);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * EndPage
 */
static
opvp_result_t	opvpEndPage(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("EndPage:printerContext=%d\n",printerContext);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

static
opvp_result_t     opvpQueryDeviceCapability(
	opvp_dc_t	printerContext,
	opvp_flag_t	queryflag,
	opvp_int_t	*buflen,
	opvp_byte_t	*infoBuf
	)
{
	int rlen;
	opvp_char_t *dummyInfo = "deviceCapability";

	opvpErrorNo = OPVP_OK;

	Output("QueryDeviceCapability:printerContext=%d\n",
	   printerContext);
	Output(":queryflag=%d\n",queryflag);
	Output(":buflen=%d\n",*buflen);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	rlen = strlen(dummyInfo);
	if (buflen == NULL) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (infoBuf == NULL) {
		*buflen = rlen+1;
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (*buflen < rlen+1) {
		*buflen = rlen+1;
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	strncpy(infoBuf,dummyInfo,rlen+1);

	return OPVP_OK;
}

static
opvp_result_t     opvpQueryDeviceInfo(
	opvp_dc_t	printerContext,
	opvp_flag_t	queryflag,
	opvp_int_t	*buflen,
	opvp_char_t	*infoBuf
	)
{
	opvp_char_t *dummyInfo = "deviceInfo";
	int rlen;

	opvpErrorNo = OPVP_OK;

	Output("QueryDeviceInfo:printerContext=%d\n",
	   printerContext);
	Output(":queryflag=%d\n",queryflag);
	Output(":buflen=%d\n",*buflen);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	rlen = strlen(dummyInfo);
	if (buflen == NULL) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (infoBuf == NULL) {
		*buflen = rlen+1;
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (*buflen < rlen+1) {
		*buflen = rlen+1;
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	strncpy(infoBuf,dummyInfo,rlen+1);

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Graphics State Object Operations
 * ------------------------------------------------------------------------
 */

/*
 * ResetCTM
 */
static
opvp_result_t	opvpResetCTM(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("ResetCTM\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	ctm.a = 1;
	ctm.b = 0;
	ctm.c = 0;
	ctm.d = 1;
	ctm.e = 0;
	ctm.f = 0;

	return OPVP_OK;
}

/*
 * SetCTM
 */
static
opvp_result_t	opvpSetCTM(
	opvp_dc_t	printerContext,
	const opvp_ctm_t *pCTM
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetCTM:CTM={%f,",pCTM->a);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!pCTM) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	Output("%f,",pCTM->b);
	Output("%f,",pCTM->c);
	Output("%f,",pCTM->d);
	Output("%f,",pCTM->e);
	Output("%f}\n",pCTM->f);

	ctm = *pCTM;

	return OPVP_OK;
}

/*
 * GetCTM
 */
static
opvp_result_t	opvpGetCTM(
	opvp_dc_t	printerContext,
	opvp_ctm_t	*pCTM
	)
{
	opvpErrorNo = OPVP_OK;

	Output("GetCTM:CTM={%f,",ctm.a);
	Output("%f,",ctm.b);
	Output("%f,",ctm.c);
	Output("%f,",ctm.d);
	Output("%f,",ctm.e);
	Output("%f}\n",ctm.f);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!pCTM) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*pCTM = ctm;

	return OPVP_OK;
}

/*
 * InitGS
 */
static
opvp_result_t opvpInitGS(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("InitGS\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * SaveGS
 */
static
opvp_result_t	opvpSaveGS(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SaveGS\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	if (!GSstack) {
		GSstack = calloc(sizeof(GS_stack),1);
		GSstack->prev = NULL;
		GSstack->next = NULL;
	} else {
		GSstack->next = calloc(sizeof(GS_stack),1);
		GSstack->next->prev = GSstack;
		GSstack->next->next = NULL;
		GSstack = GSstack->next;
	}
	GSstack->gs = currentGS;

	return OPVP_OK;
}

/*
 * RestoreGS
 */
static
opvp_result_t	opvpRestoreGS(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("RestoreGS\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	if (!GSstack) {
		currentGS = GSstack->gs;
		if (GSstack->prev) {
			GSstack = GSstack->prev;
			free(GSstack->next);
			GSstack->next = NULL;
		} else {
			free(GSstack);
			GSstack = NULL;
		}
	}

	return OPVP_OK;
}

/*
 * QueryColorSpace
 */
static
opvp_result_t	opvpQueryColorSpace(
	opvp_dc_t	printerContext,
	opvp_int_t	*pnum,
	opvp_cspace_t	*pcspace
	)
{
	int		i;
	int		n;

	opvpErrorNo = OPVP_OK;

	Output("QueryColorSpace:");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!pnum) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	n = sizeof(csList)/sizeof(opvp_cspace_t);
	*pnum = n;

	if (pcspace == NULL) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (*pnum < n) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	for (i=0;i<*pnum;i++) {
		Output("%d ",csList[i]);
		pcspace[i] = csList[i];
	}
	Output("\n");

	return OPVP_OK;
}

/*
 * SetColorSpace
 */
static
opvp_result_t	opvpSetColorSpace(
	opvp_dc_t	printerContext,
	opvp_cspace_t	cspace
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetColorSpace:cspace=%d\n",cspace);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	colorSpace = cspace;

	return OPVP_OK;
}

/*
 * GetColorSpace
 */
static
opvp_result_t	opvpGetColorSpace(
	opvp_dc_t	printerContext,
	opvp_cspace_t	*pcspace
	)
{
	opvpErrorNo = OPVP_OK;

	Output("GetColorSpace:cspace=%d\n",colorSpace);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!pcspace) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*pcspace = colorSpace;

	return OPVP_OK;
}

/*
 * SetFillMode
 */
static
opvp_result_t	opvpSetFillMode(
	opvp_dc_t	printerContext,
	opvp_fillmode_t	fillmode
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetFillMode:fillmode=%d\n",fillmode);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	fill = fillmode;

	return OPVP_OK;
}

/*
 * GetFillMode
 */
static
opvp_result_t	opvpGetFillMode(
	opvp_dc_t	printerContext,
	opvp_fillmode_t	*pfillmode
	)
{
	opvpErrorNo = OPVP_OK;

	Output("GetFillMode:fillmode=%d\n",fill);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!pfillmode) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*pfillmode = fill;

	return OPVP_OK;
}

/*
 * SetAlphaConstant
 */
static
opvp_result_t	opvpSetAlphaConstant(
	opvp_dc_t	printerContext,
	opvp_float_t		alpha
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetAlphaConstant:alpha=%f\n",alpha);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	alphaConst = alpha;

	return OPVP_OK;
}

/*
 * GetAlphaConstant
 */
static
opvp_result_t	opvpGetAlphaConstant(
	opvp_dc_t	printerContext,
	opvp_float_t		*palpha
	)
{
	opvpErrorNo = OPVP_OK;

	Output("GetAlphaConstant:alpha=%f\n",alphaConst);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!palpha) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*palpha = alphaConst;

	return OPVP_OK;
}

/*
 * SetLineWidth
 */
static
opvp_result_t	opvpSetLineWidth(
	opvp_dc_t	printerContext,
	opvp_fix_t	width
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetLineWidth:width=%F\n",width);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	lineWidth = width;

	return OPVP_OK;
}

/*
 * GetLineWidth
 */
static
opvp_result_t	opvpGetLineWidth(
	opvp_dc_t	printerContext,
	opvp_fix_t	*pwidth
	)
{
	opvpErrorNo = OPVP_OK;

	Output("GetLineWidth:width=%F\n",lineWidth);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!pwidth) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*pwidth = lineWidth;

	return OPVP_OK;
}

/*
 * SetLineDash
 */
static
opvp_result_t	opvpSetLineDash(
	opvp_dc_t	printerContext,
	opvp_int_t	num,
	const opvp_fix_t *pdash
	)
{

	opvpErrorNo = OPVP_OK;
	int		i;

	Output("SetLineDash:");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (num<0) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if ((num>0)&&(!pdash)) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	if (num) {
		if (dash) {
			dash = realloc(dash,(sizeof(opvp_fix_t))*num);
		} else {
			dash = malloc((sizeof(opvp_fix_t))*num);
		}
		if (!dash) {
			opvpErrorNo = OPVP_FATALERROR;
			Output("	ERROR No=%d\n",opvpErrorNo);
			return -1;
		}
	} else {
		if (dash) {
			free(dash);
			dash = (opvp_fix_t*)NULL;
		}
	}

	if (dash) {
		for (i=0;i<num;i++) {
			Output("%F ",pdash[i]);
			dash[i] = pdash[i];
		}
	}
	Output("\n");

	dashNum = num;

	return OPVP_OK;
}

/*
 * GetLineDash
 */
static
opvp_result_t	opvpGetLineDash(
	opvp_dc_t	printerContext,
	opvp_int_t	*pnum,
	opvp_fix_t	*pdash
	)
{
	opvpErrorNo = OPVP_OK;
	int		i;

	Output("GetLineDash:");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!pnum) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	*pnum = dashNum;
	if (!pdash) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	for (i=0;i<*pnum;i++) {
		Output("%F ",dash[i]);
		pdash[i] = dash[i];
	}
	Output("\n");

	return OPVP_OK;
}

/*
 * SetLineDashOffset
 */
static
opvp_result_t	opvpSetLineDashOffset(
	opvp_dc_t	printerContext,
	opvp_fix_t	offset
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetLineDashOffset:offset=%F\n",offset);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	dashOffset = offset;

	return OPVP_OK;
}

/*
 * GetLineDashOffset
 */
static
opvp_result_t	opvpGetLineDashOffset(
	opvp_dc_t	printerContext,
	opvp_fix_t	*poffset
	)
{
	opvpErrorNo = OPVP_OK;

	Output("GetLineDashOffset:offset=%F\n",dashOffset);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!poffset) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*poffset = dashOffset;

	return OPVP_OK;
}

/*
 * SetLineStyle
 */
static
opvp_result_t	opvpSetLineStyle(
	opvp_dc_t	printerContext,
	opvp_linestyle_t	linestyle
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetLineStyle:linestyle=%d\n",linestyle);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	currentGS.lStyle = linestyle;

	return OPVP_OK;
}

/*
 * GetLineStyle
 */
static
opvp_result_t	opvpGetLineStyle(
	opvp_dc_t	printerContext,
	opvp_linestyle_t	*plinestyle
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetLineStyle:linestyle=%d\n",currentGS.lStyle);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!plinestyle) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*plinestyle = currentGS.lStyle;

	return OPVP_OK;
}

/*
 * SetLineCap
 */
static
opvp_result_t	opvpSetLineCap(
	opvp_dc_t	printerContext,
	opvp_linecap_t	linecap
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetLineCap:linecap=%d\n",linecap);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	currentGS.lCap = linecap;

	return OPVP_OK;
}

/*
 * GetLineCap
 */
static
opvp_result_t	opvpGetLineCap(
	opvp_dc_t	printerContext,
	opvp_linecap_t	*plinecap
	)
{
	opvpErrorNo = OPVP_OK;

	Output("GetLineCap:linecap=%d\n",currentGS.lCap);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!plinecap) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*plinecap = currentGS.lCap;

	return OPVP_OK;
}

/*
 * SetLineJoin
 */
static
opvp_result_t	opvpSetLineJoin(
	opvp_dc_t	printerContext,
	opvp_linejoin_t	linejoin
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetLineJoin:linejoin=%d\n",linejoin);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	currentGS.lJoin = linejoin;

	return OPVP_OK;
}

/*
 * GetLineJoin
 */
static
opvp_result_t	opvpGetLineJoin(
	opvp_dc_t	printerContext,
	opvp_linejoin_t	*plinejoin
	)
{
	opvpErrorNo = OPVP_OK;

	Output("GetLineJoin:linejoin=%d\n",currentGS.lJoin);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!plinejoin) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*plinejoin = currentGS.lJoin;

	return OPVP_OK;
}

/*
 * SetMiterLimit
 */
static
opvp_result_t	opvpSetMiterLimit(
	opvp_dc_t	printerContext,
	opvp_fix_t	miterlimit
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetMiterLimit:miterlimit=%F\n",miterlimit);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	currentGS.mLimit = miterlimit;

	return OPVP_OK;
}

/*
 * GetMiterLimit
 */
static
opvp_result_t	opvpGetMiterLimit(
	opvp_dc_t	printerContext,
	opvp_fix_t	*pmiterlimit
	)
{
	opvpErrorNo = OPVP_OK;

	Output("GetMiterLimit:miterlimit=%F\n",currentGS.mLimit);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!pmiterlimit) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*pmiterlimit = currentGS.mLimit;

	return OPVP_OK;
}

/*
 * SetPaintMode
 */
static
opvp_result_t	opvpSetPaintMode(
	opvp_dc_t	printerContext,
	opvp_paintmode_t	paintmode
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetPaintMode:paintmode=%d\n",paintmode);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	currentGS.pMode = paintmode;

	return OPVP_OK;
}

/*
 * GetPaintMode
 */
static
opvp_result_t	opvpGetPaintMode(
	opvp_dc_t	printerContext,
	opvp_paintmode_t	*ppaintmode
	)
{
	opvpErrorNo = OPVP_OK;

	Output("GetPaintMode:paintmode=%d\n",currentGS.pMode);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!ppaintmode) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	*ppaintmode = currentGS.pMode;

	return OPVP_OK;
}

/*
 * SetStrokeColor
 */
static
opvp_result_t	opvpSetStrokeColor(
	opvp_dc_t	printerContext,
	const opvp_brush_t	*brush
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetStrokeColor:");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!brush) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (brush->pbrush) {
		Output("BRUSH:");
	} else {
		Output("SOLID:");
	}
	Output("%X,",brush->color[0]);
	Output("%X,",brush->color[1]);
	Output("%X,",brush->color[2]);
	Output("%X", brush->color[3]);
	if (brush->pbrush) {
		Output(":width=%d",brush->pbrush->width);
		Output(":height=%d",brush->pbrush->height);
		Output(":pitch=%d",brush->pbrush->pitch);
	}
	Output("\n");

	currentGS.sCol = *brush;

	return OPVP_OK;
}

/*
 * SetFillColor
 */
static
opvp_result_t	opvpSetFillColor(
	opvp_dc_t	printerContext,
	const opvp_brush_t	*brush
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetFillColor:");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!brush) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (brush->pbrush) {
		Output("BRUSH:");
	} else {
		Output("SOLID:");
	}
	Output("%X,",brush->color[0]);
	Output("%X,",brush->color[1]);
	Output("%X,",brush->color[2]);
	Output("%X", brush->color[3]);
	if (brush->pbrush) {
		Output(":width=%d",brush->pbrush->width);
		Output(":height=%d",brush->pbrush->height);
		Output(":pitch=%d",brush->pbrush->pitch);
	}
	Output("\n");

	currentGS.fCol = *brush;

	return OPVP_OK;
}

/*
 * SetBgColor
 */
static
opvp_result_t	opvpSetBgColor(
	opvp_dc_t	printerContext,
	const opvp_brush_t	*brush
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetBgColor:");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!brush) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (brush->pbrush) {
		Output("BRUSH:");
	} else {
		Output("SOLID:");
	}
	Output("%X,",brush->color[0]);
	Output("%X,",brush->color[1]);
	Output("%X,",brush->color[2]);
	Output("%X", brush->color[3]);
	if (brush->pbrush) {
		Output(":width=%d",brush->pbrush->width);
		Output(":height=%d",brush->pbrush->height);
		Output(":pitch=%d",brush->pbrush->pitch);
	}
	Output("\n");

	currentGS.bCol = *brush;

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Path Operations
 * ------------------------------------------------------------------------
 */

/*
 * NewPath
 */
static
opvp_result_t	opvpNewPath(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("NewPath\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * EndPath
 */
static
opvp_result_t	opvpEndPath(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("EndPath\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * StrokePath
 */
static
opvp_result_t	opvpStrokePath(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("StrokePath\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * FillPath
 */
static
opvp_result_t	opvpFillPath(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("FillPath\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * StrokeFillPath
 */
static
opvp_result_t	opvpStrokeFillPath(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("StrokeFillPath\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * SetClipPath
 */
static
opvp_result_t	opvpSetClipPath(
	opvp_dc_t	printerContext,
	opvp_cliprule_t	clipRule
	)
{
	opvpErrorNo = OPVP_OK;

	Output("ResetClipPath:clipRule=%d\n",clipRule);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	clip = clipRule;

	return OPVP_OK;
}

static
opvp_result_t	opvpResetClipPath(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("ResetClipPath\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * SetCurrentPoint
 */
static
opvp_result_t	opvpSetCurrentPoint(
	opvp_dc_t	printerContext,
	opvp_fix_t	x,
	opvp_fix_t	y
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SetCurrentPoint:(x,y)=(%F,",x);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	Output("%F)\n",y);

	currentGS.pos.x = x;
	currentGS.pos.y = y;

	return OPVP_OK;
}

/*
 * LinePath
 */
static
opvp_result_t	opvpLinePath(
	opvp_dc_t	printerContext,
	opvp_pathmode_t	flag,
	opvp_int_t	npoints,
	const opvp_point_t *points
	)
{
	opvpErrorNo = OPVP_OK;
	int		i;

	Output("LinePath:flag=%d:",flag);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!points) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	Output("(%F,",currentGS.pos.x);
	Output("%F)",currentGS.pos.y);

	for (i=0;i<npoints;i++) {
		Output("-(%F,",(points[i]).x);
		Output("%F)",(points[i]).y);
		currentGS.pos.x = (points[i]).x;
		currentGS.pos.y = (points[i]).y;
	}
	Output("\n");

	return OPVP_OK;
}

/*
 * PolygonPath
 */
static
opvp_result_t	opvpPolygonPath(
	opvp_dc_t	printerContext,
	opvp_int_t	npolygons,
	const opvp_int_t *nvertexes,
	const opvp_point_t *points
	)
{
	opvpErrorNo = OPVP_OK;
	int		i, j, p;

	Output("PolygonPath");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!nvertexes) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!points) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	for (p=0,i=0;i<npolygons;i++) {
		Output(":(%F,",(points[p]).x);
		Output("%F)",(points[p]).y);
		currentGS.pos.x = (points[p]).x;
		currentGS.pos.y = (points[p]).y;
		for (j=1;j<nvertexes[i];j++) {
			p++;
			Output("-(%F,",(points[p]).x);
			Output("%F)",(points[p]).y);
		}
	}
	Output("\n");

	return OPVP_OK;
}

/*
 * RectanglePath
 */
static
opvp_result_t	opvpRectanglePath(
	opvp_dc_t	printerContext,
	opvp_int_t	nrectangles,
	const opvp_rectangle_t *rectangles
	)
{
	opvpErrorNo = OPVP_OK;
	int		i;

	Output("RectanglePath");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!rectangles) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	for (i=0;i<nrectangles;i++) {
		Output(":(%F,",(rectangles[i]).p0.x);
		Output("%F)-(",(rectangles[i]).p0.y);
		Output("%F,",(rectangles[i]).p1.x);
		Output("%F)",(rectangles[i]).p1.y);
		currentGS.pos.x = (rectangles[i]).p0.x;
		currentGS.pos.y = (rectangles[i]).p0.y;
	}
	Output("\n");

	return OPVP_OK;
}

/*
 * RoundRectanglePath
 */
static
opvp_result_t	opvpRoundRectanglePath(
	opvp_dc_t	printerContext,
	opvp_int_t	nrectangles,
	const opvp_roundrectangle_t *rectangles
	)
{
	opvpErrorNo = OPVP_OK;
	int		i;

	Output("RoundRectanglePath");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!rectangles) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	for (i=0;i<nrectangles;i++) {
		Output(":(%F,",(rectangles[i]).p0.x);
		Output("%F)-(",(rectangles[i]).p0.y);
		Output("%F,",(rectangles[i]).p1.x);
		Output("%F)",(rectangles[i]).p1.y);
		currentGS.pos.x = (rectangles[i]).p0.x;
		currentGS.pos.y = (rectangles[i]).p0.y;
	}
	Output("\n");

	return OPVP_OK;
}

/*
 * BezierPath
 */
static
opvp_result_t	opvpBezierPath(
	opvp_dc_t	printerContext,
	opvp_int_t	npoints,
	const opvp_point_t *points
	)
{
	opvpErrorNo = OPVP_OK;
	int		i;

	Output("BezierPath",NULL);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!points) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	for (i=0;i<npoints;i++) {
		Output("-(%F,",(points[i]).x);
		Output("%F)",(points[i]).y);
		currentGS.pos.x = (points[i]).x;
		currentGS.pos.y = (points[i]).y;
	}
	Output("\n");

	return OPVP_OK;
}

/*
 * ArcPath
 */
static
opvp_result_t	opvpArcPath(
	opvp_dc_t	printerContext,
	opvp_arcmode_t	kind,
	opvp_arcdir_t	dir,
	opvp_fix_t	bbx0,
	opvp_fix_t	bby0,
	opvp_fix_t	bbx1,
	opvp_fix_t	bby1,
	opvp_fix_t	x0,
	opvp_fix_t	y0,
	opvp_fix_t	x1,
	opvp_fix_t	y1
	)
{
	opvpErrorNo = OPVP_OK;

	Output("ArcPath:kind=%d",kind);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	Output(":dir=%d",dir);
	Output(":(%F,",bbx0);
	Output("%F)-(",bby0);
	Output("%F,",bbx1);
	Output("%F)",bby1);
	Output(":(%F,",x0);
	Output("%F)-(",y0);
	Output("%F,",x1);
	Output("%F)\n",y1);

	currentGS.pos.x = (bbx0<bbx1 ? bbx0 : bbx1);
	currentGS.pos.y = (bby0<bby1 ? bby0 : bby1);

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Bitmap Image Operations
 * ------------------------------------------------------------------------
 */

/*
 * DrawImage
 */
static
opvp_result_t	opvpDrawImage(
	opvp_dc_t	printerContext,
	opvp_int_t	sourceWidth,
	opvp_int_t	sourceHeight,
	opvp_int_t	sourcePitch,
	opvp_imageformat_t	imageFormat,
	opvp_int_t	destinationWidth,
	opvp_int_t	destinationHeight,
	const void	*imagedata
	)
{
	opvpErrorNo = OPVP_OK;

	Output("DrawImage:sourceWidth=%d",sourceWidth);
	Output(":sourceHeight=%d",sourceHeight);
	Output(":sourcePitch=%d",sourcePitch);
	Output(":imageFormat=%d",imageFormat);
	Output(":destinationWidth=%d",destinationWidth);
	Output(":destinationHeight=%d\n",destinationHeight);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!imagedata) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * StartDrawImage
 */
static
opvp_result_t	opvpStartDrawImage(
	opvp_dc_t	printerContext,
	opvp_int_t	sourceWidth,
	opvp_int_t	sourceHeight,
	opvp_int_t	sourcePitch,
	opvp_imageformat_t	imageFormat,
	opvp_int_t	destinationWidth,
	opvp_int_t	destinationHeight
	)
{
	opvpErrorNo = OPVP_OK;

	Output("StartDrawImage:sourceWidth=%d",sourceWidth);
	Output(":sourceHeight=%d",sourceHeight);
	Output(":sourcePitch=%d",sourcePitch);
	Output(":imageFormat=%d",imageFormat);
	Output(":destinationWidth=%d",destinationWidth);
	Output(":destinationHeight=%d\n",destinationHeight);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * TransferDrawImage
 */
static
opvp_result_t	opvpTransferDrawImage(
	opvp_dc_t	printerContext,
	opvp_int_t	count,
	const void	*imagedata
	)
{
	opvpErrorNo = OPVP_OK;

	Output("TransferDrawImage:count=%d\n",count);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!imagedata) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * EndDrawImage
 */
static
opvp_result_t	opvpEndDrawImage(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("EndDrawImage\n",printerContext);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Scan Line Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartScanline
 */
static
opvp_result_t	opvpStartScanline(
	opvp_dc_t	printerContext,
	opvp_int_t	yposition
	)
{
	opvpErrorNo = OPVP_OK;

	Output("StartScanline:yposition=%d\n",yposition);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * Scanline
 */
static
opvp_result_t	opvpScanline(
	opvp_dc_t	printerContext,
	opvp_int_t	nscanpairs,
	const opvp_int_t	*scanpairs
	)
{
	opvpErrorNo = OPVP_OK;

	Output("Scanline:nscanpairs=%d\n",nscanpairs);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!scanpairs) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * EndScanline
 */
static
opvp_result_t	opvpEndScanline(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("EndScanline\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Raster Image Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartRaster
 */
static
opvp_result_t	opvpStartRaster(
	opvp_dc_t	printerContext,
	opvp_int_t	rasterWidth
	)
{
	opvpErrorNo = OPVP_OK;

	Output("StartRaster:rasterWidth=%d\n",rasterWidth);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * TransferRasterData
 */
static
opvp_result_t	opvpTransferRasterData(
	opvp_dc_t	printerContext,
	opvp_int_t	count,
	const opvp_byte_t	*data
	)
{
	opvpErrorNo = OPVP_OK;

	Output("TransferRaster:count=%d\n",count);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!data) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * SkipRaster
 */
static
opvp_result_t	opvpSkipRaster(
	opvp_dc_t	printerContext,
	opvp_int_t		count
	)
{
	opvpErrorNo = OPVP_OK;

	Output("SkipRaster:count=%d\n",count);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * EndRaster
 */
static
opvp_result_t	opvpEndRaster(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("EndRaster\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * ------------------------------------------------------------------------
 * Raster Image Operations
 * ------------------------------------------------------------------------
 */

/*
 * StartStream
 */
static
opvp_result_t	opvpStartStream(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("StartStream\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * TransferStreamData
 */
static
opvp_result_t	opvpTransferStreamData(
	opvp_dc_t	printerContext,
	opvp_int_t	count,
	const void		*data
	)
{
	opvpErrorNo = OPVP_OK;

	Output("TransferStreamData:count=%d\n",count);

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}
	if (!data) {
		opvpErrorNo = OPVP_PARAMERROR;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * EndStream
 */
static
opvp_result_t	opvpEndStream(
	opvp_dc_t	printerContext
	)
{
	opvpErrorNo = OPVP_OK;

	Output("EndStream\n");

	if (pContext != printerContext) {
		opvpErrorNo = OPVP_BADCONTEXT;
		Output("	ERROR No=%d\n",opvpErrorNo);
		return -1;
	}

	return OPVP_OK;
}

/*
 * private functions
 */
static
void	SetApiList(
	void
	)
{
	apiList.opvpOpenPrinter	= opvpOpenPrinter;
	apiList.opvpClosePrinter	= opvpClosePrinter;
	apiList.opvpStartJob		= opvpStartJob;
	apiList.opvpEndJob		= opvpEndJob;
	apiList.opvpAbortJob		= opvpAbortJob;
	apiList.opvpStartDoc		= opvpStartDoc;
	apiList.opvpEndDoc		= opvpEndDoc;
	apiList.opvpStartPage		= opvpStartPage;
	apiList.opvpEndPage		= opvpEndPage;
	apiList.opvpResetCTM		= opvpResetCTM;
	apiList.opvpSetCTM		= opvpSetCTM;
	apiList.opvpGetCTM		= opvpGetCTM;
	apiList.opvpInitGS		= opvpInitGS;
	apiList.opvpSaveGS		= opvpSaveGS;
	apiList.opvpRestoreGS		= opvpRestoreGS;
	apiList.opvpQueryColorSpace	= opvpQueryColorSpace;
	apiList.opvpSetColorSpace	= opvpSetColorSpace;
	apiList.opvpGetColorSpace	= opvpGetColorSpace;
	apiList.opvpSetFillMode	= opvpSetFillMode;
	apiList.opvpGetFillMode	= opvpGetFillMode;
	apiList.opvpSetAlphaConstant	= opvpSetAlphaConstant;
	apiList.opvpGetAlphaConstant	= opvpGetAlphaConstant;
	apiList.opvpSetLineWidth	= opvpSetLineWidth;
	apiList.opvpGetLineWidth	= opvpGetLineWidth;
	apiList.opvpSetLineDash	= opvpSetLineDash;
	apiList.opvpGetLineDash	= opvpGetLineDash;
	apiList.opvpSetLineDashOffset	= opvpSetLineDashOffset;
	apiList.opvpGetLineDashOffset	= opvpGetLineDashOffset;
	apiList.opvpSetLineStyle	= opvpSetLineStyle;
	apiList.opvpGetLineStyle	= opvpGetLineStyle;
	apiList.opvpSetLineCap		= opvpSetLineCap;
	apiList.opvpGetLineCap		= opvpGetLineCap;
	apiList.opvpSetLineJoin	= opvpSetLineJoin;
	apiList.opvpGetLineJoin	= opvpGetLineJoin;
	apiList.opvpSetMiterLimit	= opvpSetMiterLimit;
	apiList.opvpGetMiterLimit	= opvpGetMiterLimit;
	apiList.opvpSetPaintMode	= opvpSetPaintMode;
	apiList.opvpGetPaintMode	= opvpGetPaintMode;
	apiList.opvpSetStrokeColor	= opvpSetStrokeColor;
	apiList.opvpSetFillColor	= opvpSetFillColor;
	apiList.opvpSetBgColor		= opvpSetBgColor;
	apiList.opvpNewPath		= opvpNewPath;
	apiList.opvpEndPath		= opvpEndPath;
	apiList.opvpStrokePath		= opvpStrokePath;
	apiList.opvpFillPath		= opvpFillPath;
	apiList.opvpStrokeFillPath	= opvpStrokeFillPath;
	apiList.opvpSetClipPath	= opvpSetClipPath;
	apiList.opvpSetCurrentPoint	= opvpSetCurrentPoint;
	apiList.opvpLinePath		= opvpLinePath;
	apiList.opvpPolygonPath	= opvpPolygonPath;
	apiList.opvpRectanglePath	= opvpRectanglePath;
	apiList.opvpRoundRectanglePath	= opvpRoundRectanglePath;
	apiList.opvpBezierPath		= opvpBezierPath;
	apiList.opvpArcPath		= opvpArcPath;
	apiList.opvpDrawImage		= opvpDrawImage;
	apiList.opvpStartDrawImage	= opvpStartDrawImage;
	apiList.opvpTransferDrawImage	= opvpTransferDrawImage;
	apiList.opvpEndDrawImage	= opvpEndDrawImage;
	apiList.opvpStartScanline	= opvpStartScanline;
	apiList.opvpScanline		= opvpScanline;
	apiList.opvpEndScanline	= opvpEndScanline;
	apiList.opvpStartRaster	= opvpStartRaster;
	apiList.opvpTransferRasterData	= opvpTransferRasterData;
	apiList.opvpSkipRaster		= opvpSkipRaster;
	apiList.opvpEndRaster		= opvpEndRaster;
	apiList.opvpStartStream	= opvpStartStream;
	apiList.opvpTransferStreamData	= opvpTransferStreamData;
	apiList.opvpEndStream		= opvpEndStream;
	apiList.opvpQueryDeviceCapability = opvpQueryDeviceCapability;
	apiList.opvpQueryDeviceInfo	= opvpQueryDeviceInfo;
	apiList.opvpResetClipPath	= opvpResetClipPath;
}

