/***********************************************************************//**
	@file
	$Revision$
	$Author$
	$Date::                           $
***************************************************************************/
#ifdef WIN32
#include <winsock2.h>
#else
#include <iomanip>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#endif
#include "Socket.h"
#include "Command.h"

namespace openmj {

const char Socket::EOL[Socket::EOL_SIZE + 1] = { 0x0d, 0x0a, 0 };
/***********************************************************************//**
	コンストラクタ.
***************************************************************************/
Socket::Socket()
    : fd_(INVALID_FD)
{}
/***********************************************************************//**
	コンストラクタ
	@param	fd	ファイルディクリプタ
***************************************************************************/
Socket::Socket(int fd)
    : fd_(fd)
{}
/***********************************************************************//**
	デストラクタ.
***************************************************************************/
Socket::~Socket() {
    close();
}
/***********************************************************************//**
	ソケットを開く
***************************************************************************/
void Socket::open() {
    close();
    fd_ = socket(AF_INET, SOCK_STREAM, 0);
}
/***********************************************************************//**
	ソケットを閉じる
***************************************************************************/
void Socket::close() {
    if(isOpen()) {
#ifdef WIN32
        ::closesocket(fd_);
#else
        ::close(fd_);
#endif
        fd_ = INVALID_FD;
    }
}
/***********************************************************************//**
	サーバに接続する
	@param	hostname	接続先ホスト名
	@param	port		接続先ポート
	@return			接続できたとき真
***************************************************************************/
bool Socket::connect(const char* hostname, int port) {
    close();
    struct sockaddr_in address;
    address.sin_family = AF_INET;
    struct hostent* entry = gethostbyname(hostname);
    if(!entry) {
        return false;
    }
    memcpy(&address.sin_addr, entry->h_addr_list[0], entry->h_length);
    address.sin_port = htons(port);
    open();
    int result = 
        ::connect(fd_, (struct sockaddr *)&address, sizeof(address));
    if(result < 0) {
        close();
        return false;
    }
    return true;
}
/***********************************************************************//**
	サーバとしてソケットをオープンする
	@param	port	ポート
	@param	backlog
	@return		ソケット
***************************************************************************/
bool Socket::listen(int port, int backlog) {
    close();
    struct sockaddr_in address;
    address.sin_family = AF_INET;
    address.sin_port = htons(port);
    address.sin_addr.s_addr = INADDR_ANY;
    open();
    if(bind(fd_, (struct sockaddr *)&address, sizeof(address)) != 0) {
        close();
        return false;
    }
    ::listen(fd_, backlog);
    return true;
}
/***********************************************************************//**
	サーバに接続してきたクライアントとのソケットを返す
	@return	接続されたソケット
***************************************************************************/
Socket* Socket::accept() {
    if(!isOpen()) {
        return 0;
    }
    struct sockaddr_in address;
    socklen_t len = sizeof(address);
    int fd = ::accept(fd_, (struct sockaddr*)&address, &len);
    if(fd < 0) {
        return 0;
    }
    return new Socket(fd);
}
/***********************************************************************//**
	@return	ソケットが開いているとき真
***************************************************************************/
bool Socket::isOpen() const {
    return (fd_ >= 0);
}
/***********************************************************************//**
	受信.
***************************************************************************/
bool Socket::recv(std::string& msg) {
    if(!isOpen()) {
        return false;
    }
    char buff[EOL_SIZE];
    if(::recv(fd_, buff, EOL_SIZE, 0) < EOL_SIZE) {
        close();
        return false;
    }
    while(strncmp(buff, EOL, EOL_SIZE) != 0) {
        msg.append(1, buff[0]);
        memmove(buff, &buff[1], EOL_SIZE - 1);
        if(::recv(fd_, &buff[EOL_SIZE - 1], 1, 0) < 1) {
            close();
            return false;
        }
    }
    return true;
}
/***********************************************************************//**
	コマンド受信
***************************************************************************/
bool Socket::recvCommand(Command& command) {
    std::string buff;
    if(!recv(buff)) {
        return false;
    }
    command.parse(buff.c_str());
    return true;
}
/***********************************************************************//**
	文字列を送信する
	@param	msg	送信する文字列
	@return		送信に成功したとき真
***************************************************************************/
bool Socket::send(const char* msg) {
    if(!isOpen()) {
        return false;
    }
    const int total = strlen(msg);
    int rest = total;
    while(rest > 0) {
        int size = ::send(fd_, msg, rest, 0);
        if(size < 0) {
            close();
            return false;
        }
        msg += size;
        rest -= size;
    }
    return true;
}
/***********************************************************************//**
	
***************************************************************************/
bool Socket::send(const std::string& msg) {
    return send(msg.c_str());
}
/***********************************************************************//**
	初期化.
***************************************************************************/
void Socket::Initialize() {
#ifdef WIN32
    WSADATA wsadata;
    int result = WSAStartup(MAKEWORD(1,1), &wsadata);
    assert(result != SOCKET_ERROR);
#endif
}
/***********************************************************************//**
	終了処理.
***************************************************************************/
void Socket::Finalize() {
#ifdef WIN32
    int result = WSACleanup();
    assert(result == 0);
#endif
}
/***********************************************************************//**
	$Id$
***************************************************************************/
}	/* namespace openmj */
