/***********************************************************************//**
	@file
	$Revision$
	$Author$
	$Date::                           $
***************************************************************************/
#include <assert.h>
#include "Hai.h"

namespace mahjong {

const char* Hai::COLORS = "mpsz";
bool Hai::isInitialized = false;
Hai Hai::instanceTable[COLOR_MAX][9][2];
/***********************************************************************//**
	
***************************************************************************/
void Hai::set(int color, int number, bool isDora) {
    this->color = color;
    this->number = number;
    flag_.set(FLAG_DORA, (isDora ? 1 : 0));
    string_[0] = number + '0';
    string_[1] = isDora ? toupper(COLORS[color]) : COLORS[color];
    string_[2] = '\0';
}
/***********************************************************************//**
	@return	赤ドラのとき真
***************************************************************************/
bool Hai::isDora() const {
    return flag_.test(FLAG_DORA);
}
/***********************************************************************//**
	@return	字牌のとき真
***************************************************************************/
bool Hai::isZihai() const {
    return (color == COLOR_ZIHAI);
}
/***********************************************************************//**
	@return	ヤオチュウ牌のとき真
***************************************************************************/
bool Hai::isYaochu() const {
    return (isZihai() || number == 1 || number == 9);
}
/***********************************************************************//**
	@return	三元牌のとき真
***************************************************************************/
bool Hai::isSangenpai() const {
    return (isZihai() && number > 4);
}
/***********************************************************************//**
	@return	緑一色に使える牌のとき真
***************************************************************************/
bool Hai::isGreen() const {
    return ((color == COLOR_ZIHAI && number == 6) || 
            (color == COLOR_SOUZU && (number == 2 ||
                                      number == 3 ||
                                      number == 4 ||
                                      number == 6 ||
                                      number == 8)));
}
/***********************************************************************//**
	文字列に変換する.
	@return	変換した文字列.
***************************************************************************/
const char* Hai::toString() const {
    return string_;
}
/***********************************************************************//**
	次の牌を返す.
	@param	add	いくつ進めるか
	@return		牌
***************************************************************************/
const Hai* Hai::succ(int add) const {
    int num = number + add;
    switch(color) {
    case COLOR_MANZU:
    case COLOR_PINZU:
    case COLOR_SOUZU:
        num = (num - 1) % 9 + 1;
        break;
    case COLOR_ZIHAI:
        if(number <= 4) {
            num = (num - 1) % 4 + 1;
        }
        else {
            num = (num - 5) % 3 + 5;
        }
        break;
    }
    return Get(color, num);
}
/***********************************************************************//**
	比較する
	@param	hai	比較する牌
	@return		自身の方が小さいとき負、同じとき0、大きいときは正
***************************************************************************/
int Hai::compare(const Hai* hai) const {
    int cmp = hai->color - color;
    if(cmp == 0) {
        cmp = hai->number - number;
    }
    return cmp;
}
/***********************************************************************//**
	比較
***************************************************************************/
bool Hai::CompareFunc::operator()(const Hai* a, const Hai* b) const {
    return (a->compare(b) > 0);
}
/***********************************************************************//**
	同じ牌か調べる(赤ドラは考慮しない)
	@param	hai	比較する牌
	@return		同じ牌のとき真
***************************************************************************/
bool Hai::isSame(const Hai* hai) const {
    return (compare(hai) == 0);
}
/***********************************************************************//**
	赤ドラも含めて完全に同じ牌か調べる
	@param	hai	比較する牌
	@return		同じ牌のとき真
***************************************************************************/
bool Hai::isEqual(const Hai* hai) const {
    return (this == hai);
}
/***********************************************************************//**
	牌のインスタンスを返す.
	@param	color	種別
	@param	number	数字(1-9)
	@param	isDora	赤ドラ？
	@return	牌のインスタンスのポインタ(エラーのときは0)
***************************************************************************/
const Hai* Hai::Get(int color, int number, bool isDora) {
    if(!isInitialized) {
        Initialize();
    }
    if(number >= 1 &&
       ((color >= COLOR_MANZU && color <= COLOR_SOUZU && number <= 9) ||
        (color == COLOR_ZIHAI && number <= 7))) {
        return &instanceTable[color][number - 1][isDora ? 1 : 0];
    }
    return 0;
}
/***********************************************************************//**
	文字列から牌のインスタンスを返す.
	@param	str	牌を示す文字列.
	@return	牌のインスタンスのポインタ(エラーのときは0)
***************************************************************************/
const Hai* Hai::Get(const char* str) {
    char* index = strchr(COLORS, tolower(str[1]));
    if(!index) {
        return 0;
    }
    return Get(index - COLORS, str[0] - '0', isupper(str[1]));
}
/***********************************************************************//**
	@copydoc	Hai::Get(const char* str)
***************************************************************************/
const Hai* Hai::Get(const std::string& str) {
    return Get(str.c_str());
}
/***********************************************************************//**
	イテレータ
	@param	hai	牌(最初は0)
	@return		haiの次の牌。0で終了
***************************************************************************/
const Hai* Hai::Each(const Hai* hai) {
    if(!hai) {
        return Get(COLOR_MANZU, 1);
    }
    switch(hai->color) {
    case COLOR_MANZU:
    case COLOR_PINZU:
    case COLOR_SOUZU:
        if(hai->number < 9) {
            return Get(hai->color, hai->number + 1);
        }
        return Get(hai->color + 1, 1);
    case COLOR_ZIHAI:
        if(hai->number < 7) {
            return Get(hai->color, hai->number + 1);
        }
        break;
    default:
        break;
    }
    return 0;
}
/***********************************************************************//**
	初期化.
***************************************************************************/
void Hai::Initialize() {
    for(int color = 0; color < COLOR_MAX; color++) {
        for(int number = 0; number < 9; number++) {
            for(int isDora = 0; isDora < 2; isDora++) {
                Hai& hai = instanceTable[color][number][isDora];
                hai.set(color, number + 1, isDora);
            }
        }
    }
    isInitialized = true;
}
/***********************************************************************//**
	$Id$
***************************************************************************/
}	/* namespace mahjong */
