/***********************************************************************//**
	@file
	$Revision: 77 $
	$Author: yatsuhashi $
	$Date:: 2009-03-26 02:38:16 +0900#$
***************************************************************************/
#include <assert.h>
#include "Agari.h"

namespace mahjong {

const Agari::YakuTable Agari::yakuTable[YAKU_MAX] = {
    { 1, 0, "リーチ" }, 
    { 1, 1, "タンヤオ" }, 
    { 1, 0, "平和" }, 
    { 1, 0, "門前自摸" }, 
    { 1, 0, "一盃口" }, 
    { 2, 0, "七対子" }, 
    { 2, 1, "チャンタ" }, 
    { 3, 2, "純チャン" }, 
    { 2, 2, "混老頭" }, 
    { 2, 2, "対々和" }, 
    { 2, 2, "三暗刻" }, 
    { 2, 2, "三槓子" }, 
    { 3, 0, "二盃口" }, 
    { 2, 1, "三色同順" }, 
    { 2, 2, "三色同刻" }, 
    { 2, 1, "一気通貫" }, 
    { 1, 1, "役牌" }, 
    { 2, 2, "小三元" }, 
    { 3, 2, "混一色" }, 
    { 6, 5, "清一色" }, 
    { 0, 0, "役満" }, 
    { 1, 1, "国士無双" }, 
    { 1, 1, "大三元" }, 
    { 1, 1, "清老頭" }, 
    { 1, 1, "四暗刻" }, 
    { 1, 1, "四槓子" }, 
    { 1, 1, "小四喜" }, 
    { 2, 2, "大四喜" }, 
    { 1, 1, "字一色" }, 
    { 1, 1, "緑一色" }, 
    { 1, 0, "九蓮宝燈" }
};
/***********************************************************************//**
	コンストラクタ.
***************************************************************************/
Agari::Agari() {
    clear();
}
/***********************************************************************//**
	和了の内容を全てクリアする
***************************************************************************/
void Agari::clear() {
    mentsu_.clear();
    flag_.reset();
    clearYaku();
}
/***********************************************************************//**
	役をクリアする
***************************************************************************/
void Agari::clearYaku() {
    for(int i = 0; i < YAKU_MAX; i++) {
        yaku_[i] = 0;
    }
    han_ = 0;
    fu_ = 0;
}
/***********************************************************************//**
        @return	翻数
***************************************************************************/
int Agari::getHan() const {
    if(isYakuman()) {
        return 0;
    }
    return han_;
}
/***********************************************************************//**
	役の翻数を求める
	@param	yaku	調べる役
	@return		翻数
***************************************************************************/
int Agari::getHan(int yaku) const {
    return yaku_[yaku];
}
/***********************************************************************//**
        @return	符
***************************************************************************/
int Agari::getFu() const {
    return fu_;
}
/***********************************************************************//**
	@return	役満数(ダブル=2, トリプル=3, ...)
***************************************************************************/
int Agari::getYakuman() const {
    if(!isYakuman()) {
        return 0;
    }
    return han_;
}
/***********************************************************************//**
	役が含まれているか調べる
	@param	yaku	調べる役
	@return	役が含まれているとき真
***************************************************************************/
bool Agari::isInclude(int yaku) const {
    if(isYakuman() && yaku < YAKUMAN) {
        return false;
    }
    return (yaku_[yaku] > 0);
}
/***********************************************************************//**
	@return	役満のとき真
***************************************************************************/
bool Agari::isYakuman() const {
    return flag_.test(FLAG_YAKUMAN);
}
/***********************************************************************//**
	比較する
	@param	other	比較対象
	@return		自身の方が低いとき真
***************************************************************************/
bool Agari::operator<(const Agari& other) {
    return (getYakuman() < other.getYakuman() ||
            getHan() < other.getHan());
}
/***********************************************************************//**
	面子を追加する.
	@param	mentsu	追加する面子
***************************************************************************/
void Agari::pushMentsu(const Mentsu& mentsu) {
    assert(!isCalc());
    mentsu_.push_back(mentsu);
}
/***********************************************************************//**
	最後に追加した面子を取り除く.
***************************************************************************/
void Agari::popMentsu() {
    assert(!isCalc());
    mentsu_.pop_back();
}
/***********************************************************************//**
	和了牌をセットする
	@param	hai	和了牌
	@param	isRon	ロン和了のとき真
	@return		最も翻数の高い和了形
***************************************************************************/
Agari Agari::setAgariHai(const Hai* hai, bool isRon) {
    assert(!isCalc());
    flag_.set(FLAG_CALC);
    flag_.set(FLAG_RON, isRon);
    if(isInclude(YAKUMAN_KOKUSHI)) {
        return *this;
    }
    for(Mentsu::Vector::iterator iter = mentsu_.begin();
        iter != mentsu_.end();
        iter++) {
        if(!iter->isMenzen()) {
            flag_.set(FLAG_NOT_MENZEN);
            break;
        }
    }
    Agari maxAgari;
    for(Mentsu::Vector::iterator iter = mentsu_.begin();
        iter != mentsu_.end();
        iter++) {
        if(iter->isInclude(hai)) {
            iter->setAgariHai(hai, isRon);
            checkYaku();
            if(maxAgari < *this) {
                maxAgari = *this;
            }
            iter->clearAgariHai();
        }
    }
    return maxAgari;
}    
/***********************************************************************//**
	計算したか調べる
	@return	計算していたら真
***************************************************************************/
bool Agari::isCalc() const {
    return flag_.test(FLAG_CALC);
}
/***********************************************************************//**
	面前か調べる
	@return	面前のとき真
***************************************************************************/
bool Agari::isMenzen() const {
    return !flag_.test(FLAG_NOT_MENZEN);
}
/***********************************************************************//**
	役を調べる
***************************************************************************/
void Agari::checkYaku() {
    enum {
        KIND_TANYAO	= 1 << 0, 
        KIND_YAOCHU	= 1 << 1, 
        KIND_ZIHAI	= 1 << 2, 
        KIND_ALL_YAOCHU	= 1 << 3, 
        KIND_ALL_GREEN	= 1 << 4, 
    };
    static const int COLOR_ALL = ((1 << Hai::COLOR_MANZU) |
                                  (1 << Hai::COLOR_PINZU) |
                                  (1 << Hai::COLOR_SOUZU));
    static const int IPEIKOU_FLAG = (1 << 7);
    int shuntsuNum = 0;
    int kotsuNum = 0;
    int ankoNum = 0;
    int kantsuNum = 0;
    int ipeikouNum = 0;
    int shuntsuFlag[9] = { 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    int kotsuFlag[9] = { 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    int toitsuFlag[9] = { 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    int numberCount[9] = { 0, 0, 0, 0, 0, 0, 0, 0, 0 };
    int colorFlag = 0;
    int kindFlag = KIND_ALL_YAOCHU | KIND_ALL_GREEN;
    bool isPinfu = true;

    clearYaku();
    if(isMenzen() && !flag_.test(FLAG_RON)) {
        appendYaku(YAKU_TSUMO);
    }

    for(Mentsu::Vector::iterator mentsu = mentsu_.begin();
        mentsu != mentsu_.end();
        mentsu++) {
        const Hai* hai = mentsu->getHai();
        int index = hai->number - 1;
        int color = (1 << hai->color);
        colorFlag |= color;
        if(mentsu->isShuntsu()) {
            shuntsuNum++;
            if((shuntsuFlag[index] & (IPEIKOU_FLAG | color)) == color) {
                ipeikouNum++;
                shuntsuFlag[index] |= IPEIKOU_FLAG;
            }
            else {
                shuntsuFlag[index] |= color;
                shuntsuFlag[index] &= ~IPEIKOU_FLAG;
            }
        }
        else if(mentsu->isKotsu()) {
            kotsuNum++;
            if(mentsu->isAnko()) {
                ankoNum++;
            }
            if(mentsu->isKantsu()) {
                kantsuNum++;
            }
            kotsuFlag[index] |= color;
        }
        else {
            toitsuFlag[index] |= color;
        }
        if(isPinfu && !mentsu->isPinfu()) {
            isPinfu = false;
        }
        int kind = 0;
        for(int i = 0; i < mentsu->getSize(); i++) {
            const Hai* hai = mentsu->getHai(i);
            if(!hai->isGreen()) {
                kindFlag &= ~KIND_ALL_GREEN;
            }
            if(hai->isZihai()) {
                kind |= KIND_ZIHAI;
            }
            else if(hai->isYaochu()) {
                kind |= KIND_YAOCHU;
            }
            else {
                kind |= KIND_TANYAO;
            }
            numberCount[hai->number - 1]++;
        }
        kindFlag |= kind;
        if((kind & (KIND_YAOCHU | KIND_ZIHAI)) == 0) {
            kindFlag &= ~KIND_ALL_YAOCHU;
        }
    }
    if(kotsuNum == 0) {
        if(shuntsuNum == 0) {
            appendYaku(YAKU_CHITOI);
        }
        else if(isPinfu) {
            appendYaku(YAKU_PINFU);
        }
    }
    else {
        if(shuntsuNum == 0) {
            appendYaku(YAKU_TOITOI);
        }
        if(ankoNum == 3) {
            appendYaku(YAKU_SANANKO);
        }
        else if(ankoNum >= 4) {
            appendYaku(YAKUMAN_SUANKO);
        }
        if(kantsuNum == 3) {
            appendYaku(YAKU_SANKANTSU);
        }
        else if(kantsuNum >= 4) {
            appendYaku(YAKUMAN_SUKANTSU);
        }
    }

    /* 三色同順/三色同刻のチェック */
    for(int i = 0; i < 9; i++) {
        if((shuntsuFlag[i] & COLOR_ALL) == COLOR_ALL) {
            appendYaku(YAKU_SANSHOKU);
        }
        if((kotsuFlag[i] & COLOR_ALL) == COLOR_ALL) {
            appendYaku(YAKU_SANSHOKUDOUKOU);
        }
    }

    /* 一気通貫のチェック */
    if((shuntsuFlag[0] & shuntsuFlag[3] & shuntsuFlag[6]) != 0) {
        appendYaku(YAKU_ITSU);
    }

    /* 一盃口/二盃口のチェック */
    if(ipeikouNum == 1) {
        appendYaku(YAKU_IPEIKOU);
    }
    else if(ipeikouNum >= 2) {
        appendYaku(YAKU_RYANPEIKOU);
    }

    /* 混一色/清一色/字一色/九蓮宝燈のチェック */
    switch(colorFlag) {
    case ((1 << Hai::COLOR_MANZU) | (1 << Hai::COLOR_ZIHAI)):
    case ((1 << Hai::COLOR_PINZU) | (1 << Hai::COLOR_ZIHAI)):
    case ((1 << Hai::COLOR_SOUZU) | (1 << Hai::COLOR_ZIHAI)):
        appendYaku(YAKU_HONITSU);
        break;
    case (1 << Hai::COLOR_MANZU):
    case (1 << Hai::COLOR_PINZU):
    case (1 << Hai::COLOR_SOUZU):
        appendYaku(YAKU_CHINITSU);
        if(numberCount[0] >= 3 && numberCount[8] >= 3) {
            bool isChuren = true;
            for(int i = 1; i < 8; i++) {
                if(numberCount[i] < 1) {
                    isChuren = false;
                    break;
                }
            }
            if(isChuren) {
                appendYaku(YAKUMAN_CHUREN);
            }
        }
        break;
    case (1 << Hai::COLOR_ZIHAI):
        appendYaku(YAKUMAN_TSUISOU);
        break;
    }

    /* 小三元/大三元のチェック */
    int sangenpaiNum = 0;
    for(int i = 4; i < 7; i++) {
        if(kotsuFlag[i] & (1 << Hai::COLOR_ZIHAI)) {
            sangenpaiNum += 3;
            appendYaku(YAKU_YAKUHAI);
        }
        else if(toitsuFlag[i] & (1 << Hai::COLOR_ZIHAI)) {
            sangenpaiNum += 2;
        }
    }
    if(sangenpaiNum == 8) {
        appendYaku(YAKU_SHOSANGEN);
    }
    else if(sangenpaiNum >= 9) {
        appendYaku(YAKUMAN_DAISANGEN);
    }

    /* 四喜和のチェック */
    int kazehaiNum = 0;
    for(int i = 0; i < 4; i++) {
        if(kotsuFlag[i] & (1 << Hai::COLOR_ZIHAI)) {
            kazehaiNum += 3;
        }
        else if(toitsuFlag[i] & (1 << Hai::COLOR_ZIHAI)) {
            kazehaiNum += 2;
        }
    }
    if(kazehaiNum == 11) {
        appendYaku(YAKUMAN_SHOSUSHI);
    }
    else if(kazehaiNum >= 12) {
        appendYaku(YAKUMAN_DAISUSHI);
    }

    /* 緑一色のチェック */
    if(kindFlag & KIND_ALL_GREEN) {
        appendYaku(YAKUMAN_RYUISOU);
    }

    switch(kindFlag) {
    case KIND_TANYAO:
        appendYaku(YAKU_TANYAO);
        break;
    case (KIND_ALL_YAOCHU | KIND_YAOCHU | KIND_ZIHAI | KIND_TANYAO):
        appendYaku(YAKU_CHANTA);
        break;
    case (KIND_ALL_YAOCHU | KIND_YAOCHU | KIND_TANYAO):
        appendYaku(YAKU_JUNCHAN);
        break;
    case (KIND_ALL_YAOCHU | KIND_YAOCHU | KIND_ZIHAI):
        appendYaku(YAKU_HONROUTOU);
        break;
    case (KIND_ALL_YAOCHU | KIND_YAOCHU):
        appendYaku(YAKUMAN_CHINROUTOU);
        break;
    default:
        break;
    }
}
/***********************************************************************//**
	役をセットする.
***************************************************************************/
void Agari::appendYaku(int yaku) {
    if(isYakuman() && yaku < YAKUMAN) {
        return;
    }
    const YakuTable& table = yakuTable[yaku];
    int han = isMenzen() ? table.menzenHan : table.nakiHan;
    if(han > 0) {
        if(!isYakuman() && yaku > YAKUMAN) {
            clearYaku();
            flag_.set(FLAG_YAKUMAN);
        }
        yaku_[yaku] += han;
        han_ += han;
    }
}
/***********************************************************************//**
	文字列に変換する
***************************************************************************/
std::string Agari::toString() const {
    std::string str;
    char buff[256];
    for(Mentsu::Vector::const_iterator mentsu = mentsu_.begin();
        mentsu != mentsu_.end();
        mentsu++) {
        str.append(mentsu->toString());
        sprintf(buff, "[%d]", mentsu->getFu());
        str.append(buff);
    }
    for(int i = 0; i < YAKU_MAX; i++) {
        if(isInclude(i)) {
            const YakuTable& table = yakuTable[i];
            sprintf(buff, " %s %d", table.yakuName, 
                    (isMenzen() ? table.menzenHan : table.nakiHan));
            str.append(buff);
        }
    }
    return str;
}
/***********************************************************************//**
	$Id: Agari.cpp 77 2009-03-25 17:38:16Z yatsuhashi $
***************************************************************************/
}	/* namespace mahjong */
