/*
 * OpenI2CRADIO
 * EUSART Handler
 * Copyright (C) 2013-06-20 K.Ohta <whatisthis.sowhat ai gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2,
 *  or (at your option) any later version.
 *  This library / program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; see the file COPYING. If not, write to the
 *  Free Software Foundation, 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301, USA.
 *
 *  As a special exception, if you link this(includeed from sdcc) library
 *  with other files, some of which are compiled with SDCC,
 *  to produce an executable, this library does not by itself cause
 *  the resulting executable to be covered by the GNU General Public License.
 *  This exception does not however invalidate any other reasons why
 *  the executable file might be covered by the GNU General Public License.
 */

#if defined(__SDCC)
#include <sdcc-lib.h>
#include <pic18fregs.h> /* ONLY FOR PIC18x */
#include <delay.h>
#else
#include <xc.h>
#endif
#include <signal.h>

#include "iodef.h"
#include "idle.h"
#include "ioports.h"
#include "euart.h"

static char uart_rfifo[UART_BUF_SIZE];

static unsigned int uart_rx_rptr;
static unsigned int uart_rx_wptr;
static int uart_rx_bytes;
static unsigned char uart_rx_sts;
static unsigned char uart_rx_xon;
static unsigned char uart_tx_xon;

static unsigned char uart_rx_wakeup;
#define UART_BAUD_FACTOR 34 // 57600bps on 8MHz,BRGH=1,BRG16=1

void uart_sleep(void)
{
    PIR1bits.TX1IF = 0;
    PIR1bits.RC1IF = 0;
    IPR1bits.TX1IP = 0;  // Low
    IPR1bits.RC1IP = 1; // High
    RCSTA = 0b10010000; //SPEN, 8bit, ASYNC, CREN = 0
    BAUDCON = 0b00001010; // IDLE High, DTRXP=0(Positive), BRG16, -ABDEN, WUE
    TXSTA = 0b00100100; //8bit, ASYNC, TXEN, Break, BRGH=1
    PIE1bits.TX1IE = 0;
    PIE1bits.RC1IE = 1;
    uart_rx_wakeup = 0;
    UART_CTSOUT = 0;
    SPBRGH = UART_BAUD_FACTOR / 256;
    SPBRG = UART_BAUD_FACTOR % 256;
}

void uart_wakeup(void)
{
    PIR1bits.TX1IF = 0;
    PIR1bits.RC1IF = 0;
    IPR1bits.TX1IP = 0;  // Low
    IPR1bits.RC1IP = 1; // High
    RCSTA = 0b10010000; //SPEN, 8bit, ASYNC, CREN
    BAUDCON = 0b00001010; // IDLE High,  DTRXP=0(Positive),BRG16, -ABDEN, WUE
    TXSTA = 0b00100100; //8bit, ASYNC, TXEN, Break, BRGH=1
    PIE1bits.TX1IE = 0;
    PIE1bits.RC1IE = 1;
    UART_CTSOUT = 0;
   //   uart_rx_wakeup = 0xff;
   SPBRGH = UART_BAUD_FACTOR / 256;
   SPBRG = UART_BAUD_FACTOR % 256;
}

void uart_init(void)
{
    uart_rx_rptr = 0;
    uart_rx_wptr = 0;
    uart_rx_bytes = 0;
    uart_rx_wakeup = 0;
    uart_rx_xon = 0xff;
    uart_tx_xon = 0xff;
    uart_sleep();
}


void uart_inthdr_rx(void)
{
    unsigned char c;
#if 1
    if(uart_rx_wakeup == 0){
        if(BAUDCONbits.WUE != 1){ // Okay, Wakeup interrupt
            uart_rx_wakeup = 0xff;
        }
        goto _l0; // exit
    }
    if(BAUDCONbits.ABDEN == 1) { // Still configure baudrate
        goto _l0; // Exit
    }
#endif
    c = RCREG;
    if(RCSTAbits.OERR == 0) {
        if(RCSTAbits.FERR == 0) {
            uart_rx_sts = 0; // Clear error
	    uart_rfifo[uart_rx_wptr] = c;
	    uart_rx_wptr++;
	    if(uart_rx_wptr >= UART_BUF_SIZE) uart_rx_wptr = 0;
	    uart_rx_bytes++;
	    if(uart_rx_bytes >= UART_BUF_SIZE) uart_rx_bytes = UART_BUF_SIZE;
	    if((uart_rx_xon != 0) && (uart_rx_bytes >= ((UART_BUF_SIZE * 8) / 10))) { // Buffer will be full
	       UART_CTSOUT = 1; // OFF
	       uart_rx_xon = 0;
	    }
        } else { // Frame Error
         uart_rx_sts != UART_FRAMEERROR;
        }
    } else {
         uart_rx_sts != UART_OVERFLOW;
    }

_l0:  // Exit
 PIR1bits.RC1IF = 0;
 RCSTAbits.CREN = 1;
 return;
}

/*
 * Pull char from RX FIFO.
 */
unsigned char uart_pullchar(void)
{
    unsigned char c;
    while(PIR1bits.RC1IF == 1) {
        idle(100); // Wait for RX completed
    }
    if(uart_rx_bytes <= 0) return 0; // EMPTY
    c = uart_rfifo[uart_rx_rptr++];
    if(uart_rx_rptr > UART_BUF_SIZE) uart_rx_rptr = 0;
    uart_rx_bytes--;
    if(uart_rx_bytes <= 0) uart_rx_bytes = 0;
    if((uart_rx_xon == 0) && (uart_rx_bytes < ((UART_BUF_SIZE * 3) / 10))) {  // Buffer will be empty
       UART_CTSOUT = 0; // ON
       uart_rx_xon = 0xff;
    }
    return c;
}

unsigned char uart_pushchar(unsigned char c, unsigned int timeout)
{
    unsigned int i;
    if(timeout != 0){
        for(i = timeout; i > 0; i--){
            if((UART_RTSIN == 0) && (TXSTAbits.TRMT1 == 1)) break;
            idle(100);
        }
        if(i != 0) goto _l1;  // Send and return;
        return UART_TIMEOUT; // Timeout
    }
    if((TXSTAbits.TRMT1 == 0) || (UART_RTSIN != 0)) return UART_TIMEOUT; // If timeout=0 return immidiately.
_l1:
    TXREG = c;
    return 0;
}

void uart_pushxon(unsigned int timeout)
{
    unsigned int i;
#if 0
   if(uart_rx_wakeup == 0) return;
    if(timeout != 0) {
        for(i = timeout; i > 0; i--){
            if(TXSTAbits.TRMT1 == 1) break;
            idle(100); // 100us
        }
        return;
    } else {
        while(TXSTAbits.TRMT1 == 0) idle(100);
    }
#endif
   //    TXREG = UART_CH_XON;
    UART_CTSOUT = 0;
}

void uart_pushxoff(unsigned int timeout)
{
    unsigned int i;
#if 0
   if(uart_rx_wakeup == 0) return;
    if(timeout != 0) {
        for(i = timeout; i > 0; i--){
            if(TXSTAbits.TRMT1 == 1) break;
            idle(100); // 100us
        }
        return;
    } else {
        while(TXSTAbits.TRMT1 == 0) idle(100);
    }
#endif
   //    TXREG = UART_CH_XOFF;
    UART_CTSOUT = 1;
}

void uart_break(void)
{
    TXSTAbits.SENDB1 = 1;
}

unsigned char uart_getstat(void)
{
    unsigned char s = uart_rx_sts;
    if(uart_rx_wakeup != 0) s |= UART_WAKEUP;
    return s;
}