<?php
/**
 * 文字列関係ユーティリティ
 * @author okada
 * @package CFW_Util
 */
class CFW_Util_String{

	const NORMALIZE_NONE = 0;
	const NORMALIZE_TRIM_LEFT= 1;
	const NORMALIZE_TRIM_RIGHT= 2;
	const NORMALIZE_TRIM_ALL = 3;
	const NORMALIZE_DELETE_SPACE= 4;
	const NORMALIZE_DELETE_WIDE_SPACE = 8;
	const NORMALIZE_DELETE_TAB = 16;
	const NORMALIZE_DELETE_ALL_WHITE_SPACE = 31;
	const NORMALIZE_TAB_TO_SPACE = 32;
	const NORMALIZE_TO_NARROW = 64;
	const NORMALIZE_TO_LOWER = 128;
	const NORMALIZE_TO_UPPER = 256;
	const NORMALIZE_TO_ALPHA= 512;
	const NORMALIZE_TO_DIGIT = 1024;
	const NORMALIZE_TO_ALPHA_DIGIT = 1536;
	const NORMALIZE_TO_NUMERIC = 2048;
	const NORMALIZE_DEFAULT = 35;
	/**
	 * 入力オブジェクトを文字列化
	 * @param object $o 入力オブジェクト
	 * @return string 入力オブジェクトを文字列化したもの
	 */
	public static function toString($o){
		$result = "";
		if ($o == null) {
			return $result;
		}
		else {
			if ($o instanceof Date) {
				$result = $o->format("%Y/%m/%d %H:%M:%S");
			}
			else {
				$result = "".$o;
				$result = self::deleteEscapeSequence($result);
			}
		}
		return $result;
	}
	/**
	 * 文字列の右側指定文字数分取得
	 * @param string $source 元の文字列
	 * @param string $length 長さ
	 * @return string 元の文字列の右側
	 */
	public static function right($source, $length){
		$sourceLength = mb_strlen($source);
		if ($sourceLength < $length) return $source;
		return mb_substr($source, ($sourceLength - $length), $length);
	}
	/**
	 * （余計だと思われる）エスケープシーケンスの削除
	 * 縦タブ、EOF文字、null文字を削除する。
	 * 特にnullインジェクションを回避するためにシステム外部の入力に対して必ず通す必要がある。
	 * @param string $s 対象文字列
	 * @return string 削除後の文字列
	 */
	public static function deleteEscapeSequence($s){
		$result = mb_ereg_replace("\v","",mb_ereg_replace("\x1a","",mb_ereg_replace("\x00","",$s)));
		return $result;
	}
	/**
	 * 改行削除
	 * @param string $s 元の文字列
	 * @return string $result 削除後の文字列
	 */
	public static function deleteNewLine($s){
		$result = mb_ereg_replace("\r\n", "", $s);
		return $result;
	}
	/**
	 * tab -> 空白の削除
	 * @param string $s 元の文字列
	 * @return string 変換後の文字列
	 */
	public static function tabToWhiteSpace($s){
		$result = mb_ereg_replace("\t"," ",$s);
		return $result;
	}
	/**
	 * 空白削除
	 * 全角でも半角でもスペースっぽいものは取り除く、タブ文字も追加
	 * 文字列中の空白も対象
	 * @param string $s 元の文字列
	 * @return string $result 削除後の文字列
	 */
	public static function deleteAllWhiteSpace($s){
		$result = mb_ereg_replace("(　| |\t)+", "", $s);
		return $result;
	}
	/**
	 * trim()拡張
	 * 全角でも半角でもスペースっぽいものは取り除く、タブ文字も追加
	 * @param string $s 元の文字列
	 * @return string 削除後の文字列
	 */
	public static function trim($s){
		$result = mb_ereg_replace("^(　| |\t)+", "", $s);
		$result = mb_ereg_replace("(　| |\t)+$", "", $result);
		return $result;
	}
	/**
	 * trim()拡張
	 * 全角でも半角でもスペースっぽいものは取り除く、タブ文字も追加
	 * @param string $s 元の文字列
	 * @return string $result 削除後の文字列
	 */
	public static function trimLeft($s){
		$result = mb_ereg_replace("^(　| |\t)+","",$s);
			return $result;
	}
	/**
	 * trim()拡張
	 * 全角でも半角でもスペースっぽいものは取り除く、タブ文字も追加
	 * @param string $s 元の文字列
	 * @return string $result 削除後の文字列
	 */
	public static function trimRight($s){
		$result = mb_ereg_replace("(　| |\t)+$","",$s);
		return $result;
	}
	/**
	 * 全角カナ、全角英数を半角に変換
	 * @param string $s 対象文字列
	 * @return string $result 半角変換結果
	 */
	public static function toNarrow($s){
		$result = mb_convert_kana(mb_convert_kana($s, "a"), "k");
		$result = str_replace("￥", "\\", $result);
		return $result;
	}
	/**
	 * 半角カナ、半角英数を全角に変換
	 * @param string $s 対象文字列
	 * @return string $result 全角に変換した文字列
	 */
	public static function toWide($s){
		$result = mb_convert_kana(mb_convert_kana($s, "A"), "K");
		$result = str_replace("\\", "￥", $result);
		return $result;
	}
	/**
	 * 英数字を正規化する
	 * すべての空白、タブ文字、全角空白を取り除く
	 * @param string $s 元の文字列
	 * @return string $result 正規化結果
	 */
	public static function normalizeAlphaNumeric($s){
		$object = self::deleteEscapeSequence($s);
		$result = mb_ereg_replace("[^0-9.\\-a-zA-Z]", "", $object);
		return $result;
	}
	/**
	 * 数字項目を正規化する
	 * すべての空白、タブ文字、全角空白を取り除く
	 * @param string $s 元の文字列
	 * @return string $result 正規化結果
	 */
	public static function normalizeDigit($s){
		$object = self::deleteEscapeSequence($s);
		$result = mb_ereg_replace("[^0-9]", "", $object);
		return $result;
	}
	/**
	 * 英数字を正規化する
	 * すべての空白、タブ文字、全角空白を取り除く
	 * @param string $s 元の文字列
	 * @return strign $result 正規化結果
	 */
	public static function normalizeAlpha($s){
		$object = self::deleteEscapeSequence($s);
		$result = mb_ereg_replace("[^a-zA-Z]","", $object);
		return $result;
	}
	/**
	 * 数字項目を正規化する
	 * すべての空白、タブ文字、全角空白を取り除く
	 * カンマを削除
	 * @param string $s 元の文字列
	 * @return string $result 正規化結果
	 */
	public static function normalizeNumeric($s){
		$escape = self::deleteEscapeSequence($s);
		$narrow = self::toNarrow($escape);
		$line = self::deleteNewLine($narrow);
		$space = self::deleteAllWhiteSpace($line);
		$result = mb_ereg_replace("[^0-9.\\-]", "", $space);
		return $result;
	}
    /**
     * normalizeString
     * @param string $s 元の文字列
     * @param integer $mode 正規化モード
     * @return string $result 正規化された文字列
     */
    public static function normalize($s,$mode = self::NORMALIZE_DEFAULT){
        $norm = self::deleteEscapeSequence($s);
        if (($mode & CFW_Util_String::NORMALIZE_TRIM_LEFT) == CFW_Util_String::NORMALIZE_TRIM_LEFT) {
            $norm = self::trimLeft($norm);
        }
        if (($mode & CFW_Util_String::NORMALIZE_TRIM_RIGHT) == CFW_Util_String::NORMALIZE_TRIM_RIGHT) {
            $norm = self::trimRight($norm);
        }
        if (($mode & CFW_Util_String::NORMALIZE_DELETE_SPACE) == CFW_Util_String::NORMALIZE_DELETE_SPACE) {
            $norm = mb_ereg_replace(" ", "", $norm);
        }
        if (($mode & CFW_Util_String::NORMALIZE_DELETE_WIDE_SPACE) == CFW_Util_String::NORMALIZE_DELETE_WIDE_SPACE) {
            $norm = mb_ereg_replace("　", "", $norm);
        }
        if (($mode & CFW_Util_String::NORMALIZE_DELETE_TAB) == CFW_Util_String::NORMALIZE_DELETE_TAB) {
            $norm = mb_ereg_replace("\t", "", $norm);
        }
        if (($mode & CFW_Util_String::NORMALIZE_TO_NARROW) == CFW_Util_String::NORMALIZE_TO_NARROW) {
            $norm = self::toNarrow($norm);
        }
        if (($mode & CFW_Util_String::NORMALIZE_TO_LOWER) == CFW_Util_String::NORMALIZE_TO_LOWER){
            $norm = mb_strtolower($norm);
        }
        if (($mode & CFW_Util_String::NORMALIZE_TO_UPPER) == CFW_Util_String::NORMALIZE_TO_UPPER) {
            $norm = mb_strtoupper($norm);
        }
        if (($mode & CFW_Util_String::NORMALIZE_TO_ALPHA_DIGIT) == CFW_Util_String::NORMALIZE_TO_ALPHA_DIGIT) {
            $norm = self::normalizeAlphaNumeric($norm);
        }else{
            if (($mode & CFW_Util_String::NORMALIZE_TO_DIGIT) == CFW_Util_String::NORMALIZE_TO_DIGIT) {
                $norm = self::normalizeDigit($norm);
            }
            if (($mode & CFW_Util_String::NORMALIZE_TO_ALPHA) == CFW_Util_String::NORMALIZE_TO_ALPHA) {
                $norm = self::normalizeAlpha($norm);
            }
        }
        if (($mode & CFW_Util_String::NORMALIZE_TO_NUMERIC) == CFW_Util_String::NORMALIZE_TO_NUMERIC) {
            $norm = self::normalizeNumeric($norm);
        }
        $result = $norm;
        return $result;
    }
	/**
	 * normalizeString
	 * @param string $s 元の文字列
	 * @param integer $mode 正規化モード
	 * @return string $result 正規化された文字列
	 * @remarks normalize()に移動
	 */
	public static function normalizeString($s,$mode){
	    //旧バージョンのサポート
	    return self::normalize($s,$mode);
	}

	/**
	 * 特定文字列を繰り返した文字列を返す
	 * @param string $s 繰り返したい文字列
	 * @param int $times 回数
	 * @return string 文字列を指定回数繰り返した文字列
	 */
	public static function repeat($s, $times){
		$result = "";
		for($i = 0; $i < $times; $i++){
			$result .= $s;
		}
		return $result;
	}
	public static function upperCamel($original){
        $s = preg_replace("#[_\-]#","_",$original);
        if(strpos($s,"_") === false){
            //区切り文字が無くて全部大文字の場合は小文字にして返す。
            //全部大文字にして元と同じだったら元から全てが大文字
            $upper = strtoupper($s);
            if ($upper == $s)
            {
                return strtoupper(substr($s,0,1)) . strtolower( substr($s,1) );
            }
            return strtoupper(substr($s,0,1)) . substr($s,1);
        }
        else{
            return self::camelize($s,true);
        }
	}
    /**
     * アンダースコア、ハイフンなどで区切られた文字列を先頭小文字のらくだ形式 camelFormatに変換する.
     * <p>全てが大文字または全てが小文字の場合は全て小文字にする。</p>
     * @param string $original 元になる文字列
     * @return string|unknown|string
     */
    public static function lowerCamel($original){
        $s = preg_replace("#[_\-]#","_",$original);
        if(strpos($s,"_") === false){
            //区切り文字が無くて全部大文字の場合は小文字にして返す。
            //全部小文字にして元と同じだったら元から全てが小文字
            $lower = strtolower($s);
            if($lower == $s){
                return $lower;
            }
            //全部大文字にして元と同じだったら元から全てが大文字、そうでなければcamlize済み
            $upper = strtoupper($s);
            if ($upper == $s)
            {
                return $lower;
            }
            //camel済み
            if(strlen($s) <= 1){
                //1文字だったら考えるまでもない
                return strtolower($s);
            }
            //JDKのBeanプロパティと同等の仕様にする。
            $c0 =  substr( $s,0,1);
            $c1 = substr( $s,1,1);
            //2文字以上大文字が連続しているなら「そういうワードである」とみなしてそのまま
            if (('A' <= $c0 && $c0 <= 'Z') && ('A' <= $c1 && $c1 <= 'Z'))
            {
                return $s;
            }
            return strtolower($c0) . substr($s,1);
        }
        else{
            return self::camelize($s,false);
        }
    }

    /**
     * @param string $s 元になる文字列
     * @param boolean $toUpper 最初の文字を大文字にする場合にtrueに設定する
     * @return string
     */
    private static function camelize($s,$toUpper = false)
    {
        $result = "";
        $stringArray = split("_",$s);
        for ($i = 0; $i < count($stringArray); $i++)
        {
            $element = $stringArray[$i];
            if($result == ""){
                if($toUpper){
                    $result .= ucfirst(strtolower($element));
                }
                else{
                    $result .= strtolower($element);
                }
            }
            else{
               $result .= ucfirst(strtolower($element));
            }
        }
        return $result;
    }

    /**
     * ランダム文字列生成(簡易)
     *
     * 適当にランダムな文字列を生成する。
     * ネタ元はhttp://tipstip.com/post/32/92/
     * @param integer $length
     * @return string
     */
    public static function randomString($length=8){
        $f='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $s='';
        $count = strlen($f)-1;
        for($i=0;$i<$length;$i++) $s.=substr($f,mt_rand(0,$count),1);
        return $s;
    }

    /**
     * キーと値の組み合わせを記述した文字列をキーと値に分割する
     * @param unknown_type $str
     * @param unknown_type $separator
     * @return multitype:unknown "key"と"value"を要素キーに持つ配列
     */
    public static function splitKeyValue($str,$separator = "="){
        $position = strpos($str,$separator);
        if($position === false){
            return array( "key" => self::trim($str),"value"=> "");
        }
        $key = self::trim( substr($str,0,$position - 1) );
        $value = self::trim( substr($str,$position + 1) );
        return array( "key" => $key,"value"=> $value);
    }
}