<?php
Zend_Loader::loadClass("CFW_Data_Parameter");
Zend_Loader::loadClass("CFW_Data_Criteria");
Zend_Loader::loadClass("CFW_Models_Entity");
Zend_Loader::loadClass("CFW_Data_Connection");
Zend_Loader::loadClass("CFW_Data_Criteria_ColumnValueCondition");
Zend_Loader::loadClass("CFW_Util_Dump");
Zend_Loader::loadClass("CFW_Log_Logger");
Zend_Loader::loadClass("CFW_Log_LoggerFactory");


/**
 * データソース基本機能
 * @author okada
 * @package CFW_Data
 */
class CFW_Data_DataSource{
	/**
	 * @var CFW_Data_Connection
	 */
	var $connection;
	/**
	 * @var CFW_Models_EntityPropery
	 */
	var $entityProperty = null;
    /**
     * @var string
     */
    var $tableName = "";

    /**
     * @var string
     */
    var $selectQuery;
    /**
     * @var string
     */
    var $countQuery;
    /**
     * @var string
     */
    var $insertQuery;
    /**
     * @var string
     */
    var $updateQuery;
    /**
     * @var string
     */
    var $deleteQuery;

    /**
     * @var array
     */
    var $selectFields = array();
    /**
     * @var array
     */
    var $updateFields = array();

    /**
     * ログ出力
     * @var CFW_Log_Logger
     */
    private  $logger;
    var $currentAuthUser;
    
	/**
	 * 構築 
	 * 
	 * @param CFW_Data_Connection $connection
	 * @param CFW_Models_EntityProperty $property
	 * @return unknown_type
	 */
	public function __construct($connection,$property = null){
		$this->connection= $connection;
		$this->entityProperty = $property;

		$this->logger = CFW_Log_LoggerFactory::getLogger("cfw_debug",__CLASS__);
		$this->currentAuthUser = null;
	}
	/**
	 * 接続オープン
	 * @return unknown_type
	 */
	public function connect(){
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
        $this->connection->connect();
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"end",$this->currentAuthUser);
	}
	/**
	 * 接続解除
	 * @return unknown_type
	 */
	public function disconnect(){
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
		$this->connection->disconnect();
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"end",$this->currentAuthUser);
	}
	/**
	 * 既存のプロパティを設定する。
	 * @param unknown_type $property
	 * @return unknown_type
	 */
	public function setProperty($property){
		$this->entityProperty = $property;
	}
	/**
	 * 既存の接続を設定する.
	 *
	 * これを使う場合は呼び出し元が接続|接続解除を制御する
	 * @param unknown_type $connection
	 * @return unknown_type
	 */
	public function setConnection($connection){
        $this->currentAuthUser = $connection->currentAuthUser;
		
		$this->connection = $connection;
	}
    /**
     * エンティティを保存する.
     * @param unknown_type $data
     * @param unknown_type $fields
     * @return number
     */
    public function save($data,$fields = array()){
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,$data,$this->currentAuthUser);
						
		$this->setupProperty($fields);
        //データ接続は外からもらうし、要らない
        //$this->connection->beginTransaction();
        $entities = array();
        if(is_array($data)){
            $entities = $data;
        }
        else{
            $entities[] = $data;
        }
        $primaryKeys = $this->entityProperty->getPrimaryKeys();
        $criteria = $this->createPrimaryKeyCriteria($primaryKeys,null);
        $criteria->assemble();
        //各アイテムで違うのはパラメータの値だけなのでこれでいいはず
        $this->buildInsertQuery($this->updateFields);
        $this->buildUpdateQuery($criteria,$this->updateFields);
        $this->buildDeleteQuery($criteria);

        //system date
        $sysdate = $this->connection->systemDate();
        $createdAtField = CFW_Util_String::lowerCamel($this->entityProperty->getCreatedAtField());
        $modifiedAtField = CFW_Util_String::lowerCamel($this->entityProperty->getModifiedAtField());
        
        //削除、更新、挿入をアイテム毎に実行
        $result = 0;
        foreach($entities as  $entity){
            $parameters = array();
            if($entity->isDeleted){
                $criteria = $this->createPrimaryKeyCriteria($primaryKeys,$entity);
                $criteria->assemble();
                $result += $this->executeDelete($criteria);
            }
            else{
                if($entity->isModified){
                    if($entity->isNew){
                    	$entity->$createdAtField = $sysdate;
                    	$entity->$modifiedAtField = $sysdate;
                    	$result += $this->executeInsert($entity);
                    }
                    else{
                    	$entity->$modifiedAtField = $sysdate;
                    	$criteria = $this->createPrimaryKeyCriteria($primaryKeys,$entity);
                        $criteria->assemble();
                        $result += $this->executeUpdate($entity,$criteria);
                    }
                }
            }

        }
        //データ接続は外からもらうし、要らない
		//$this->connection->commitTransaction();
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"end",$this->currentAuthUser);
        return $result;


    }
    /**
     * 主キーによる選択条件を作る
     * @param unknown_type $primaryKeys
     * @param unknown_type $entity
     * @return CFW_Data_Criteria
     */
    function createPrimaryKeyCriteria($primaryKeys,$entity){
        $criteria = new CFW_Data_Criteria();
        foreach($primaryKeys as $primaryKey ){
            $value = null;
            if($entity != null){
                $n = CFW_Util_String::lowerCamel($primaryKey->fieldName);
                if(isset($entity->$n)) $value = $entity->$n;

            }
            $criteria->addWhere(new CFW_Data_Criteria_ColumnValueCondition($primaryKey->fieldName,$value,CFW_Data_Criteria::OPERATOR_EQ));

        }
        return $criteria;
    }
    /**
     * 挿入実行
     * @param  $entity
     * @return unknown_type
     */
    private function executeInsert($entity){
        $parameters= array();

        foreach($this->updateFields as $field){
            //TODO:作成日、更新日、作成者、更新者
            $parameter = null;
            $value = null;
            $n = CFW_Util_String::lowerCamel($field->fieldName);
            if(isset($entity->$n)) $value = $entity->$n;

            $parameter = new CFW_Data_Parameter(
                ":" . $field->alias(),
                $value
            );
            $parameters[] = $parameter;

        }

        $command = $this->connection->createCommand($this->insertQuery,$parameters);
        $result = $command->executeUpdate();
        //IDがあったら最新の値をセット
        if($this->entityProperty->identityField != ""){
            $id = $this->connection->lastIdentity($this->entityProperty->getName());
            $n = $this->entityProperty->identityField;
            $entity->$n = intval( $id );
        }
        $entity->isNew =false;
        $entity->isModified = false;
        return $result;
    }
    /**
     * 更新実行
     * @param unknown_type $entity
     * @param CFW_Data_Criteria $criteria
     * @return unknown
     */
    private function executeUpdate($entity,CFW_Data_Criteria $criteria){
        $parameters= array();
		
        foreach($this->updateFields as $fields){
            //TODO:作成日、更新日、作成者、更新者
            $parameter = null;
            //作成日は更新させない
            if($fields->fieldName == $this->entityProperty->getCreatedAtField()) continue;

            $value = null;
            $n = CFW_Util_String::lowerCamel($fields->fieldName);
            if(isset($entity->$n)) $value = $entity->$n;
            $parameter = new CFW_Data_Parameter(
            	":" . $fields->alias(),
                $value
            );
            $parameters[] = $parameter;

        }
        $parameters = array_merge($parameters,$criteria->getParameters());
        $command = $this->connection->createCommand($this->updateQuery,$parameters);
        $result = $command->executeUpdate();
        $entity->isNew =false;
        $entity->isModified = false;
        return $result;

    }
    /**
     * 削除実行
     * @param unknown_type $criteria
     * @return unknown_type
     */
    private function executeDelete($criteria){
        $fields = $this->entityProperty->getFields();
        $parameters= $criteria->getParameters();
        $command = $this->connection->createCommand($this->deleteQuery,$parameters);
        return $command->executeUpdate();
    }
    /**
     * データの追加
     * @param unknown_type $data
     * @param unknown_type $fields
     * @return unknown_type
     */
    public function insert($data,$fields = array()){
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,$data,$this->currentAuthUser);
		
		$this->setupProperty($fields);
        $entities = array();
        if(is_array($data)){
            $entities = $data;
        }
        else{
            $entities[] = $data;
        }
        $this->buildInsertQuery($this->updateFields);
        $result = 0;
        foreach($entities as $entity){
            $result += $this->executeInsert($entity);
        }
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"end",$this->currentAuthUser);
        return $result;
    }
    /**
     * データの更新
     * @param $data
     * @param $criteria
     * @param $fields
     * @return unknown_type
     */
    public function update($data,$criteria = null,$fields = array()){
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,$data,$this->currentAuthUser);
		
		if($criteria == null) $criteria = new CFW_Data_Criteria();
        $this->setupProperty($fields);
        $fields = $this->entityProperty->getFields();
        $entities = array();
        if(is_array($data)){
            $entities = $data;
        }
        else{
            $entities[] = $data;
        }
        $criteria->assemble();
        $this->buildUpdateQuery($criteria,$this->updateFields);

        $result = 0;
        foreach($entities as $entity){
            $result += $this->executeUpdate($entity,$criteria);
        }
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"end",$this->currentAuthUser);
        return $result;
    }
    /**
     * データの削除
     * @param $criteria
     * @return unknown_type
     */
    public function delete($criteria){
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,$criteria,$this->currentAuthUser);
		
		$this->setupProperty();
        if($criteria == null) $criteria = new CFW_Data_Criteria();
        $criteria->assemble();
        $this->buildDeleteQuery($criteria);
        $result = $this->executeDelete($criteria);
		
        $this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"end",$this->currentAuthUser);
        return $result;
    }
    /**
     * データの検索
     * @param $criteria
     * @param $fields
     * @return unknown_type
     */
    public function find($criteria = null,$fields = array()){
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,$criteria,$this->currentAuthUser);
		
		$this->setupProperty($fields);
        if($criteria == null) $criteria = new CFW_Data_Criteria();
        $criteria->assemble();
        $selectFields = $this->getSelectFields();
        $this->buildSelectQuery($criteria,$selectFields);
        $command = $this->connection->createCommand($this->selectQuery,$criteria->getParameters());
        $result = $command->executeQueryAsObject($this->entityProperty->entityClass,$this->entityProperty->getName());

        $this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"end",$this->currentAuthUser);
        return $result;
    }
    /**
     * 件数取得
     * 
     * 本当は count()にしたいのだけど多分無理
     * @param unknown_type $criteria
     * @return unknown_type
     */
    public function getCount($criteria = null){
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,$criteria,$this->currentAuthUser);
		
		if($criteria == null) $criteria = new CFW_Data_Criteria();
        $criteria->assemble();
        $this->buildCountQuery($criteria,array());
        $command = $this->connection->createCommand($this->countQuery,$criteria->getParameters());
        $result = $command->executeScalar();

		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"end",$this->currentAuthUser);
		return $result;

    }
    /**
     * 主キーによる検索
     * @param $entity
     * @return unknown_type
     */
    public function findByPk($entity){
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,$entity,$this->currentAuthUser);
		
		$this->setupProperty(array());
        $criteria = $this->createPrimaryKeyCriteria($this->entityProperty->getPrimaryKeys(),$entity);
        $criteria->assemble();
        $fields = $this->getSelectFields();
        $this->buildSelectQuery($criteria,$fields);
        $command = $this->connection->createCommand($this->selectQuery,$criteria->getParameters());
        $result = $command->executeQueryAsObject($this->entityProperty->entityClass,$this->entityProperty->getName());

        $this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
        return $result;
        
    }
    /**
     * 対象エンティティが存在するか
     * @param $entity
     * @return boolean
     */
    public function exists($entity){
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"start",$this->currentAuthUser);
		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,$entity,$this->currentAuthUser);
		
		$this->setupProperty(array());
        $criteria = $this->createPrimaryKeyCriteria($this->entityProperty->getPrimaryKeys(),$entity);
        $criteria->assemble();
        $fields = $this->getSelectFields();
        $this->buildSelectQuery($criteria,$fields);
        $command = $this->connection->createCommand($this->selectQuery,$criteria->getParameters());
        $result = $command->executeQuery($this->entityProperty->entityClass,$this->entityProperty->getName());

		$this->logger->log(CFW_Log_Level::DEBUG,__FUNCTION__,"end",$this->currentAuthUser);
        if(count($result) > 0) return true;
        return false;

    }
    /**
     * プロパティ構築
     * @param $fields
     * @return unknown_type
     */
    public function setupProperty($fields = array()){
        if($this->entityProperty == null){
            $this->entityProperty = $this->connection->describe($this->tableName);
        }
        $this->updateFields = array();
        if(count( $fields ) > 0){
            //fieldを指定されたときは指定されたfieldだけdoUpdateする
            $this->entityProperty->setDoUpdate(false);
            foreach($fields as $field){
                $p = $this->entityProperty->$field;
                if($p != null){
                	if($p->isIdentity)continue;
                    $p->doUpdate = true;
                    $this->updateFields[$p->fieldName] = $p;
                }
            }
        }
        else{
          	$this->updateFields = $this->entityProperty->getUpdateFields();
        }
    }
    /**
     * 選択列取得
     * @return unknown_type
     */
    function getSelectFields(){
        $fields= array();
        foreach($this->entityProperty->getFields() as $f){
            if($f->doUpdate){
                $fields[$f->alias()] =  $f->fullName();
            }
        }
        return $fields;
    }
    /**
     * 選択クエリー組み立て
     * @param CFW_Data_Criteria $c
     * @return unknown_type
     */
    public function buildSelectQuery(CFW_Data_Criteria $c,$fields){
            $query = "SELECT";
            $columns = "";
            $where =  "";
            $orderBy =  "";
            $groupBy =  "";
            $having =  "";
            $limits =  "";

            //columns 構築
            foreach($fields as $key => $field){
                if($columns != "") $columns .= ",";
                $columns .= $field . " AS " . $key;

            }
            if($c != null){
                //where句構築
                $where = $c->getWhereExpression();
                //orderBy構築
                $orderBy = $c->getOrderByExpression();
                //group by 構築
                $groupBy = $c->getGroupByExpression();
                //having構築
                $having = $c->getHavingExpression();
                //limit,offset
                if($c->getLimit() > 0){
                    $limits .= "LIMIT " . $c->getLimit();
                }
                if($c->getOffset() > 0){
                    $limits .= " OFFSET " . $c->getOffset();
                }
            }
            if($c->queryOption != ""){
                $query .= " ".$c->queryOption;
            }
            if($columns == ""){
                $query .= " *";
            }
            else{
                $query .= " " . $columns;
            }
            //TODO:複数テーブル対応
            $query .= " FROM " . $this->entityProperty->getName();
            if($where != ""){
                $query .= " WHERE " . $where;
            }
            if($groupBy != ""){
                $query .= " GROUP BY " . $groupBy;
            }
            if($having != ""){
                $query .= " HAVING " . $having;
            }
            if($orderBy != ""){
                $query .= " ORDER BY " . $orderBy;
            }

            $this->selectQuery = $query;

    }
    /**
     * 件数取得用選択クエリー組み立て
     * @param CFW_Data_Criteria $c
     * @return unknown_type
     */
    public function buildCountQuery(CFW_Data_Criteria $c){
            $query = "SELECT";
            $columns = "";
            $where =  "";
            $orderBy =  "";
            $groupBy =  "";
            $having =  "";
            $limits =  "";

           $columns .= "COUNT(*)";
            if($c != null){
                //where句構築
                $where = $c->getWhereExpression();
            }
            if($c->queryOption != ""){
                $query .= " ".$c->queryOption;
            }
            if($columns == ""){
                $query .= " *";
            }
            else{
                $query .= " " . $columns;
            }
            //TODO:複数テーブル対応
            $query .= " FROM " . $this->entityProperty->getName();
            if($where != ""){
                $query .= " WHERE " . $where;
            }

            $this->countQuery = $query;

    }
    /**
     * 更新クエリー組み立て
     * @param $criteria
     * @param $fields
     * @return unknown_type
     */
    public function buildUpdateQuery($criteria,$fields){
        $query = "UPDATE";
        $columns = "";
        $where =  "";

        //columns 構築
        foreach($fields as $field){
            //作成日は何があっても更新させない
            if($field->fieldName== $this->entityProperty->createdAtField) continue;
            if($columns != "") $columns .= ",";

            $columns .= $field->fieldName . " = ?";
        }

        if($criteria != null){
            //where句構築
            $where = $criteria->getWhereExpression();
        }
        $query .= " " . $this->entityProperty->getName();
        $query .= " SET " . $columns;

        if($where != ""){
            $query .= " WHERE " . $where;
        }
        $this->updateQuery = $query;
    }
    /**
     * 挿入クエリー組み立て
     * @param unknown_type $fields
     * @return unknown_type
     */
    public function buildInsertQuery($fields){
        $query = "INSERT INTO";
        $columns = "";
        $values = "";

        //columns 構築
        foreach($fields as $field){
            if($columns != ""){
                $columns .= ",";
                $values .= ",";
            }
            $columns .= $field->fieldName ;
            $values .= " ? ";
        }
        $query .= " " . $this->entityProperty->getName();
        $query .= " (" . $columns . ")" ;
        $query .= "  VALUES(" . $values . ")";

        $this->insertQuery = $query;

    }
    /**
     * 削除クエリー組み立て
     * @param unknown_type $criteria
     * @return unknown_type
     */
    public function buildDeleteQuery($criteria){
        $query = "DELETE";
        $columns = "";
        $where =  "";

        if($criteria != null){
            //where句構築
            $where = $criteria->getWhereExpression();
        }
        $query .= " " . $this->entityProperty->getName();

        if($where != ""){
            $query .= " WHERE " . $where;
        }
        $this->deleteQuery = $query;

    }
}