<?php
/**
 * データベースに接続するデータソースの抽象
 * @author okada
 *
 */
class CFW_Database_AbstractDataSource extends CFW_Model_DataSource{
    /**
     * 設定
     * @var unknown_type
     */
    var $config;
    /**
     * データベース接続
     * @var unknown_type
     */
	var $connection;
	
	/**
	 * 構築
	 * @param $config
	 * @return unknown_type
	 */
	public function __construct(&$config = null){
        $this->config = $config;	    
	}
	/**
	 * (non-PHPdoc)
	 * @see Model/CFW_Model_DataSource#connect()
	 */
	public function connect(){
	    $this->connection = CFW_Database_AbstractConnection::getConnection($this->config);
	    $this->connection->connect();
	}
	/**
	 * (non-PHPdoc)
	 * @see Model/CFW_Model_DataSource#disconnect()
	 */
    public function disconnect(){
        $this->connection->disconnect();
        $this->connection = null;
    }
    /**
     * (non-PHPdoc)
     * @see Model/CFW_Model_DataSource#insert()
     */
	public function insert(&$model,$values){
	    $command = $this->_createInsertCommand($model,$values);
        return $command->executeUpdate();
	    
	}
	/**
	 * (non-PHPdoc)
	 * @see Model/CFW_Model_DataSource#update()
	 */
	public function update(&$model,$values,$criteria = null){
	    if($criteria != null){
            $criteria->assemble();
	    }
        $command = $this->_createUpdateCommand($model,$values,$criteria);
        
        return $command->executeUpdate();
	}
	/**
	 * (non-PHPdoc)
	 * @see Model/CFW_Model_DataSource#delete()
	 */
	public function delete(&$model,$criteria = null){
        $criteria->assemble();
        $command = $this->_createDeleteCommand($model,$criteria);
        return $command->executeUpdate();
	}
	/**
	 * クエリー直接実行
	 * @param $query
	 * @param $params
	 * @return unknown_type
	 */
    public function query($query,$params){
        $command = new CFW_Database_Command($query);
        $command->addParameters($params);
        
        return $command->executeQuery();
    }
	/**
	 * 
	 * @param CFW_Model $model
	 * @param CFW_Database_Criteria $criteria
	 * @param $fields
	 * @param $order
	 * @param $limit
	 * @param $offset
	 * @return unknown_type
	 */
	public function read(&$model,$criteria = null,$fields = null){
	    if(! $model instanceof CFW_Database_TableModel){
	        throw new Exception("Invalid Model Type");
	    }
	    $table = $model;
	    
       if($criteria == null){   
           $criteria = new CFW_Database_Criteria();
       }
	   if(count($criteria->orderBy) == 0){
	       foreach($table->_primaryKeys as $k){
                $criteria->addOrderBy($k);
	       }
	   }
	   $criteria->assemble();
	   
	   $countCriteria = new CFW_Database_Criteria();
	   $countCriteria->where = $criteria->where;
       $countCriteria->having = $criteria->having;
       $countCriteria->groupBy = $criteria->groupBy;
       
	   $cnt = $this->count($table,$countCriteria);
	   if($criteria->limit > 0){
    	   if($cnt <= $criteria->offset){
    	       //これ以上ない
    	       return new $table->_dataTableType();   
    	   }
    	   if($criteria->limit > ($cnt - $criteria->offset)){
    	       $criteria->limit = $cnt - $criteria->offset;
    	   }
	   }
	   $query = $this->_buildSelectQuery($table,$criteria,$fields);
	   
	   $command = new CFW_Database_Command($query);
	   $command->setConnection($this->connection);
	   $command->addParameters($criteria->getParameters());
	   
	   $dataSet = $table->newDataTable();
	   $command->fill($dataSet);
	   
	   return $dataSet;
	   
	}
	/**
	 * (non-PHPdoc)
	 * @see Model/CFW_Model_DataSource#count()
	 */
	public function count(&$model,$criteria = null){
       if($criteria == null){   
           $criteria = new CFW_Database_Criteria();
       }
       
       $criteria->assemble();
       
       
       $query = $this->_buildSelectQuery($model,$criteria,array("COUNT(*) AS CNT"));
       
       $command = new CFW_Database_Command($query);
       $command->setConnection($this->connection);
       $command->addParameters($criteria->getParameters());
       
       $result = $command->executeQuery();
       
       
       return $result->rows[0]->CNT;
       
	    
	}
	/**
	 * 修飾済みテーブル名生成
	 * @param $model
	 * @return unknown_type
	 */
    function _buildFullName(&$model){
        $fn = "";
        if( !empty($model->_catalog)) $fn .= $model->_catalog . "."; 
        if( !empty($model->_schema)) $fn .= $model->_schema. "."; 
        if( !empty($model->_name)) $fn .= $model->_name; 
        
        return $fn;
    }
	/**
	 * (non-PHPdoc)
	 * @see Model/CFW_Model_DataSource#describe()
	 */
	public function describe(&$model){
		$config = CFW_Common_Config::factory("core");
		$cache = CFW_Common_Cache::factory($config->datasource->toArray());
		$cacheKey = str_replace(".","__",$this->_buildFullName($model));
        $cols = $cache->load($cacheKey);
        if($cols){
        	return $cols;
        }

        
		$cols = array();
		$q = "select *,
		 columnproperty(object_id(table_name), column_name,'IsIdentity') is_identity 
		from information_schema.columns
where table_schema = :schema and table_name = :tableName order by ordinal_position";
		$command = new CFW_Database_Command($q);
		$command->setConnection($this->connection);
		$command->addParameter(":schema",$model->_schema);
        $command->addParameter(":tableName",$model->_name);
		$reader = $command->executeReader();
		while($reader->read()){
			$col = new CFW_Model_DataColumn();
            $col->schema = $reader->get("TABLE_SCHEMA");
			$col->tableName = $reader->get("TABLE_NAME");
            $col->name = $reader->get("COLUMN_NAME");
            $col->default = $reader->get("COLUMN_DEFAULT");
            $col->nullable =  ($reader->get("IS_NULLABLE") == "YES" ? true : false);
            $col->type = $reader->get("DATA_TYPE");
            $col->length = $reader->get("CHARACTER_OCTET_LENGTH");
            $col->precision = $reader->get("NUMERIC_PRECISION");
            $col->scale = $reader->get("NUMERIC_SCALE");
            $col->identity = ($reader->get("is_identity") == 1 ? true:false);
            
			$cols[$col->name] = $col;
		}
		
		$q = "SELECT ccu.column_name as COLUMN_NAME 
FROM  information_schema.table_constraints tc
inner join information_schema.constraint_column_usage ccu
on  tc.table_catalog=ccu.table_catalog
AND tc.table_schema=ccu.table_schema
AND tc.table_name=ccu.table_name
WHERE tc.table_catalog= :catalog
AND tc.table_schema = :schema
AND tc.table_name= :tableName
AND tc.constraint_type='PRIMARY KEY'
AND tc.constraint_name=ccu.constraint_name";
        $command = new CFW_Database_Command($q);
        $command->setConnection($this->connection);
        $command->addParameter(":catalog",$model->_catalog);
        $command->addParameter(":schema",$model->_schema);
        $command->addParameter(":tableName",$model->_name);
        $reader = $command->executeReader();
        while($reader->read()){
        	$pkColumn = $reader->get("COLUMN_NAME");
        	if(array_key_exists($pkColumn,$cols)){
        		$cols[$pkColumn]->primaryKey = true;
        	}
        }
		
        $cache->save($cols,$cacheKey);
        return $cols;
		
	}
	/**
	 * 選択クエリー文字列生成
	 * @param $model
	 * @param $criteria
	 * @param $columns
	 * @return unknown_type
	 */
	function _buildSelectQuery(&$model,$criteria = null,$columns = null){
	    $fromClause = $this->_buildFullName($model);
	    
		$selectClause= "";
		if($columns !== null){
			foreach($columns as $col){
				if($selectClause !== "") $selectClause .=", ";
				$selectClause .= $col;
			}
			
		}
		else{
			$selectClause = "*";
		}
		$whereClause = "";
        if($criteria !== null){
            $whereClause = $criteria->getWhereExpression();
        }	
        $orderClause = "";
        if($criteria != null){
            $orderClause = $criteria->getOrderByExpression();
        }
        $groupByClause = "";
        if($criteria != null){
            $groupByClause = $criteria->getGroupByExpression();
        }
        $havingClause = "";
        if($criteria != null){
            $havingByClause = $criteria->getHavingExpression();
        }
        
        $query = "";
        //$limit = 0 なら全部
        if($criteria->limit == 0){
            $query = "SELECT {$selectClause} FROM {$fromClause}";
            if($whereClause !== ""){
                $query .= " WHERE {$whereClause}";
            }
            if($groupByClause !== ""){
                $query .= " GROUP BY {$groupByClause}";
            }
            if($havingClause !== ""){
                $query .= " HAVING {$havingClause}";
            }
            if($orderClause !== ""){
                $query .= " ORDER BY {$orderClause}";
            }
        }
        else{
            $first = $criteria->offset + 1;
            $last = $criteria->offset + $criteria->limit;
            $innerquery = "SELECT TOP {$last} {$selectClause} FROM {$fromClause}";
            if($whereClause !== ""){
                $innerquery .= " WHERE {$whereClause}";
            }
            if($groupByClause !== ""){
                $innerquery .= " GROUP BY {$groupByClause}";
            }
            if($havingClause !== ""){
                $innerquery .= " HAVING {$havingClause}";
            }
            if($orderClause !== ""){
                $innerquery .= " ORDER BY {$orderClause}";
            }
            
            $outerquery = "SELECT TOP {$criteria->limit} * FROM ( {$innerquery} ) as inner_table";
            
            $reverseOrder = "";
            if($criteria != null){
                $reverseOrder .= $criteria->getOrderByReverseExpression(); 
            }
            
            if($reverseOrder !== ""){
                $outerquery .= " ORDER BY {$reverseOrder}";
            }
            
            $query = "SELECT * FROM ( {$outerquery} ) as outer_table";
            if($orderClause !== ""){
                $query .= " ORDER BY {$orderClause}";
            }
            
            
        }
        
        return $query;
        
	}
	/**
	 * 挿入クエリー生成
	 * @param $model
	 * @param $fields
	 * @return unknown_type
	 */
	function _createInsertCommand(&$model,$fields){
	    if($model->_columns == null){
	        $model->_columns = $this->describe($model);
	    }
	    $columns = "";
	    $values = "";
	    $params = array();
	    foreach($fields as $field => $value){
	        if(!array_key_exists($field, $model->_columns)){
	            continue;
	        }
	        
	        $col = $model->_columns[$field];
	        if($col->identity)
	        {
	            continue;
	        }
	        if($columns != ""){
	            $columns .= ", ";
	            $values .= ", "; 
	        }
            $columns .= $field;
            $values .= ":". $field;
            
	        $params[":". $field] = array("value" => $value);
	        
	    }
	    $query = "INSERT INTO {$model->_name} ({$columns}) VALUES({$values})";
	    $command = new CFW_Database_Command($query);
        $command->setConnection($this->connection);
	    $command->addParameters($params);
	    return $command;
	    
        	       
	}
	/**
	 * 更新クエリー生成
	 * @param $model
	 * @param $fields
	 * @param $criteria
	 * @return unknown_type
	 */
    function _createUpdateCommand(&$model,$fields,$criteria = nulll){
        if($model->_columns == null){
            $model->_columns = $this->describe($model);
        }
        $columns = "";
        $params = array();
        foreach($fields as $field => $value){
            if(!array_key_exists( $field, $model->_columns)){
                continue;
            }
            
            $col = $model->_columns[$field];
            if($col->identity)
            {
                continue;
            }
            if($columns != ""){
                $columns .= ", ";
            }
            $columns .= $field . " = " . ":". $field;
            $params[":". $field] = array("value" => $value);
                        
            
            
        }
        $whereClause = "";
        if($criteria != null){
            $whereClause = $criteria->getWhereExpression();
            $params = array_merge($params,$criteria->getParameters());
        }
        $query = "UPDATE {$model->_name} SET {$columns}";
        if($whereClause != ""){
            $query .= " WHERE " . $whereClause;
        }
        $command = new CFW_Database_Command($query);
        $command->setConnection($this->connection);
        $command->addParameters($params);
        return $command;
        
    }
    /**
     * 削除クエリー生成
     * @param $model
     * @param $criteria
     * @return unknown_type
     */
    function _createDeleteCommand(&$model,$criteria = null){
        $whereClause = "";
        $params = array();
        if($criteria != null){
            $whereClause = $criteria->getWhereExpression();
            $params = array_merge($params,$criteria->getParameters());
        }
                
        $query = "DELETE FROM {$model->_name}";
        if($whereClause != ""){
            $query .= " WHERE " . $whereClause;
        }
        $command = new CFW_Database_Command($query);
        $command->setConnection($this->connection);
        $command->addParameters($criteria->getParameters());
        return $command;
        
    }
    /**
     * 最後に追加した行のidentity列、またはauto_increment列の値を取得
     * @return unknown_type
     */
    public function getLastId(){
        $sql = 'SELECT SCOPE_IDENTITY() as id';
        $command = new CFW_Database_Command($sql);
        $command->setConnection($this->connection);

        $result = $command->executeQuery();
       
       
        $id = (int) $result->rows[0]->id;
        
        return $id;
    }
	
}
?>