using System;
using System.Collections.Generic;
using System.Text;
using System.Data;
using NUnit.Framework;
using CFW.FieldProperties;
using CFW.Database;
using CFW.Database.TableCommand;

namespace CFWDatabaseTest.SQLite
{
    [TestFixture]
    public class TableSelectCommand
    {
        FieldProperty field1;
        FieldProperty field2;
        FieldProperty field3;
        FieldProperty field4;
        FieldProperty field5;
        FieldProperty field6Sub1;
        FieldProperty field6Sub2;

        CFW.Database.Config.ConnectionConfigElement config;
        CFW.Database.Connection con;

        [NUnit.Framework.SetUp]
        public void init()
        {
            field1 = FieldProperty.NewInstance("int", "test_table3", "field1", "[field1]", 4, 10, 0, false, 0, true);
            field2 = FieldProperty.NewInstance("varchar", "test_table3", "field2", "[field2]", 50, 0, 0, false, "", false);
            field3 = FieldProperty.NewInstance("datetime", "test_table3", "field3", "[field3]", 16, 23, 3, true, null, false);
            field4 = FieldProperty.NewInstance("decimal", "test_table3", "field4", "[field4]", 12, 10, 2, true, null, false);
            field5 = FieldProperty.NewInstance("smallint", "test_table3", "field5", "[field5]", 2, 5, 0, true, null, false);
            field6Sub1 = FieldProperty.NewInstance("int", "test_table3", "field6_sub1", "[field6_sub1]", 4, 10, 0, true, null, false);
            field6Sub2 = FieldProperty.NewInstance("int", "test_table3", "field6_sub2", "[field6_sub2]", 4, 10, 0, true, null, false);

            config = CFW.Database.Config.ConnectionConfigFactory.GetConfig("sqlite");
            con = CFW.Database.ConnectionFactory.GetConnection(config);
        }

        [NUnit.Framework.TearDown]
        public void term()
        {
        }
        [Test]
        public void Get()
        {

            con.Open();

            List<TestTableRow> testList = get(1);

            Assert.AreEqual(1, testList.Count);
            TestTableRow row = testList[0];

            Assert.AreEqual(1, row.Field1);
            Assert.AreEqual("aaaa", row.Field2);
            Assert.AreEqual(new DateTime(2001, 2, 3), row.Field3);
            Assert.AreEqual(102.34M, row.Field4);
            Assert.AreEqual(10, row.Field5);
            Assert.AreEqual(100, row.Field6Sub1);
            Assert.AreEqual(1000, row.Field6Sub2);

            
            con.Close();
        }
        [Test]
        public void Update()
        {
            TestTableRow original = new TestTableRow();
            original.Field1 = 101;
            original.Field2 = "1234567890";
            original.Field3 = new DateTime(1970,2,3);
            original.Field4 = 123.45M;
            original.Field5 = 101;
            original.Field6Sub1 = 10100;
            original.Field6Sub2 = 101000;


            con.Open();

            CFW.Database.TableCommand.TableInsertCommand command = new CFW.Database.TableCommand.TableInsertCommand("test_table3");
            command.AddInsert(field1,original.Field1);
            command.AddInsert(field2, original.Field2);
            command.AddInsert(field3, original.Field3);
            command.AddInsert(field4,original.Field4);
            command.AddInsert(field5,original.Field5);
            command.AddInsert(field6Sub1, original.Field6Sub1);
            command.AddInsert(field6Sub2, original.Field6Sub2);

            int result = con.ExecuteUpdate(command.GetCommand());
            Assert.Greater(result, 0);

            con.Close();

            con.Open();
            List<TestTableRow> testList = get(101);

            Assert.AreEqual(1, testList.Count);
            TestTableRow row = testList[0];

            Assert.AreEqual(101, row.Field1);
            Assert.AreEqual("1234567890", row.Field2);
            Assert.AreEqual(new DateTime(1970, 2, 3), row.Field3);
            Assert.AreEqual(123.45M, row.Field4);
            Assert.AreEqual(101, row.Field5);
            Assert.AreEqual(10100, row.Field6Sub1);
            Assert.AreEqual(101000, row.Field6Sub2);
            con.Close();


            con.Open();

            CFW.Database.TableCommand.TableUpdateCommand update = new CFW.Database.TableCommand.TableUpdateCommand("test_table3");
            update.AddUpdate( field2, original.Field2);
            update.AddUpdate(field3, original.Field3);
            update.AddUpdate(field4, original.Field4);
            update.AddUpdate(field5, original.Field5);
            update.AddUpdate(field6Sub1, 10111);
            update.AddUpdate(field6Sub2, 101111);
            update.AddWhere(new ColumnValueCondition( field1, original.Field1, "="));

            result = con.ExecuteUpdate(update.GetCommand());
            Assert.Greater(result, 0);

            con.Close();


            con.Open();

            testList = get(original.Field1);

            Assert.AreEqual(1, testList.Count);
            row = testList[0];

            Assert.AreEqual(101, row.Field1);
            Assert.AreEqual("1234567890", row.Field2);
            Assert.AreEqual(new DateTime(1970, 2, 3), row.Field3);
            Assert.AreEqual(123.45M, row.Field4);
            Assert.AreEqual(101, row.Field5);
            Assert.AreEqual(10111, row.Field6Sub1);
            Assert.AreEqual(101111, row.Field6Sub2);
            con.Close();

            con.Open();

            CFW.Database.TableCommand.TableDeleteCommand delete = new CFW.Database.TableCommand.TableDeleteCommand("test_table3");
            delete.AddWhere(new ColumnValueCondition(field1, original.Field1, "="));

            result = con.ExecuteUpdate(delete.GetCommand());
            Assert.Greater(result, 0);

            con.Close();


            con.Open();
            testList = get(original.Field1);

            Assert.AreEqual(0, testList.Count);

            con.Close();
        }
        public void UpdateWithNull()
        {
            TestTableRow original = new TestTableRow();
            original.Field1 = 101;
            original.Field2 = "1234567890";
            original.Field3 = new DateTime(1970, 2, 3);
            original.Field4 = 123.45M;
            original.Field5 = 101;
            original.Field6Sub1 = 10100;
            original.Field6Sub2 = 101000;


            con.Open();

            CFW.Database.TableCommand.TableInsertCommand command = new CFW.Database.TableCommand.TableInsertCommand("test_table3");
            command.AddInsert(field1, original.Field1);
            command.AddInsert(field2, original.Field2);
            command.AddInsert(field3, original.Field3);
            command.AddInsert(field4, original.Field4);
            command.AddInsert(field5, null);
            command.AddInsert(field6Sub1, null);
            command.AddInsert(field6Sub2, null);

            int result = con.ExecuteUpdate(command.GetCommand());
            Assert.Greater(result, 0);

            con.Close();

            con.Open();
            List<TestTableRow> testList = get(101);

            Assert.AreEqual(1, testList.Count);
            TestTableRow row = testList[0];

            Assert.AreEqual(101, row.Field1);
            Assert.AreEqual("1234567890", row.Field2);
            Assert.AreEqual(new DateTime(1970, 2, 3), row.Field3);
            Assert.AreEqual(123.45M, row.Field4);
            Assert.IsNull(row.Field5);
            Assert.IsNull(row.Field6Sub1);
            Assert.IsNull(row.Field6Sub2);
            con.Close();


            con.Open();

            CFW.Database.TableCommand.TableUpdateCommand update = new CFW.Database.TableCommand.TableUpdateCommand("test_table3");
            update.AddUpdate(field2, original.Field2);
            update.AddUpdate(field3, original.Field3);
            update.AddUpdate(field4, original.Field4);
            update.AddUpdate(field5, null);
            update.AddUpdate(field6Sub1, null);
            update.AddUpdate(field6Sub2, null);
            update.AddWhere(new ColumnValueCondition(field1, original.Field1, "="));

            result = con.ExecuteUpdate(update.GetCommand());
            Assert.Greater(result, 0);

            con.Close();


            con.Open();

            testList = get(original.Field1);

            Assert.AreEqual(1, testList.Count);
            row = testList[0];

            Assert.AreEqual(101, row.Field1);
            Assert.AreEqual("1234567890", row.Field2);
            Assert.AreEqual(new DateTime(1970, 2, 3), row.Field3);
            Assert.AreEqual(123.45M, row.Field4);
            Assert.IsNull(row.Field5);
            Assert.IsNull(row.Field6Sub1);
            Assert.IsNull(row.Field6Sub2);
            con.Close();

            con.Open();

            CFW.Database.TableCommand.TableDeleteCommand delete = new CFW.Database.TableCommand.TableDeleteCommand("test_table3");
            delete.AddWhere(new ColumnValueCondition(field1, original.Field1, "="));

            result = con.ExecuteUpdate(delete.GetCommand());
            Assert.Greater(result, 0);

            con.Close();


            con.Open();
            testList = get(original.Field1);

            Assert.AreEqual(0, testList.Count);

            con.Close();
        }
        [Test]
        public void GetList()
        {
            con.Open();

            CFW.Database.TableCommand.TableSelectCommand command = new CFW.Database.TableCommand.TableSelectCommand("test_table3 as testtest");
            field1.TableName = "testtest";
            field2.TableName = "testtest";
            field3.TableName = "testtest";
            command.AddSelect(field1);
            command.AddSelect(field2);
            command.AddSelect(field3);

            command.AddWhere(new ColumnValueCondition( field3, new DateTime(2002,1,1), ">"));
            command.AddOrderBy(field3,"DESC");


            List<TestTableRow> testList = con.ExecuteQueryAsTableRows<TestTableRow>(command.GetCommand());
            Assert.AreEqual(2, testList.Count);
            TestTableRow row = testList[0];

            Assert.AreEqual(3, row.Field1);
            Assert.AreEqual("cccc", row.Field2);
            Assert.AreEqual(new DateTime(2003, 4, 5), row.Field3);
            Assert.IsNull(row.Field4);
            Assert.IsNull(row.Field5);
            Assert.IsNull(row.Field6Sub1);
            Assert.IsNull(row.Field6Sub2);

            row = testList[1];

            Assert.AreEqual(2, row.Field1);
            Assert.AreEqual("bbbb", row.Field2);
            Assert.AreEqual(new DateTime(2002, 3, 4), row.Field3);
            Assert.IsNull(row.Field4);
            Assert.IsNull(row.Field5);
            Assert.IsNull(row.Field6Sub1);
            Assert.IsNull(row.Field6Sub2);


            con.Close();

        }

        List<TestTableRow> get(int f1)
        {
            CFW.Database.TableCommand.TableSelectCommand command = new CFW.Database.TableCommand.TableSelectCommand("test_table3");
            command.AddSelect(field1);
            command.AddSelect(field2);
            command.AddSelect(field3);
            command.AddSelect(field4);
            command.AddSelect(field5);
            command.AddSelect(field6Sub1);
            command.AddSelect(field6Sub2);

            command.AddWhere(new ColumnValueCondition( field1, f1, "="));

            List<TestTableRow> testList = con.ExecuteQueryAsTableRows<TestTableRow>(command.GetCommand());

            return testList;

        }
    }
}
