package jp.sourceforge.observoice.resources;

import static org.easymock.EasyMock.anyInt;
import static org.easymock.EasyMock.eq;
import static org.easymock.EasyMock.expect;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import java.util.Arrays;
import java.util.Calendar;
import java.util.List;

import jp.sourceforge.observoice.dao.CommentDao;
import jp.sourceforge.observoice.dao.CourseDao;
import jp.sourceforge.observoice.dao.CourseScheduleDao;
import jp.sourceforge.observoice.entity.Course;
import jp.sourceforge.observoice.entity.CourseSchedule;
import jp.sourceforge.observoice.entity.Location;

import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.junit.Test;
import org.seasar.framework.unit.annotation.EasyMock;

import com.sun.jersey.api.client.ClientResponse.Status;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.api.client.WebResource;

public class CourseResourceTest extends ResourceTest {

    @EasyMock(register = true)
    private CourseDao courseDao;
    @EasyMock(register = true)
    private CommentDao commentDao;
    @EasyMock(register = true)
    private CourseScheduleDao courseScheduleDao;

    private static final int COURSE_ID = 100;
    private static final int DAY = Calendar.MONDAY;
    private static final int PERIOD = 7;
    private static final String Q = "検索ワード";
    private static final Course COURSE0 = newCourse();
    private static final Course COURSE1 = newCourse();
    private static final List<Course> COURSES = Arrays.asList(COURSE0, COURSE1);
    private static final List<CourseSchedule> COURSE_SCHEDULES = Arrays.asList(
            newSchedule(), newSchedule());
    private static final long COMMENT_COUNT = 2;

    private static Course newCourse() {
        Course c = new Course();
        c.courseId = COURSE_ID;
        c.courseTitle = "講義名";
        c.lecturer = "担当教員";
        c.year = 2012;
        c.semester = 1;
        c.location = new Location();
        c.location.locationId = 10;
        c.location.locationName = "講義室";
        return c;
    }

    private static CourseSchedule newSchedule() {
        CourseSchedule s = new CourseSchedule();
        s.courseId = COURSE_ID;
        s.day = DAY;
        s.period = PERIOD;
        return s;
    }

    public void recordパラメータなしで講義情報を検索する() {
        expect(courseDao.search(eq(0), eq(0), eq((String) null), anyInt()))
                .andReturn(COURSES);
        expect(courseScheduleDao.selectByCourseId(COURSE_ID)).andReturn(
                COURSE_SCHEDULES).times(COURSES.size());
        expect(commentDao.countByCourseId(COURSE_ID)).andReturn(COMMENT_COUNT)
                .times(COURSES.size());
    }

    @Test
    public void パラメータなしで講義情報を検索する() throws JSONException {
        JSONObject courses = getCourses(r);
        assertThat(courses.length(), is(2));
        assertThat(courses.getJSONArray("result").length(),
                is(courses.getInt("count")));
    }

    public void record曜日を指定して講義情報を検索する() {
        expect(courseDao.search(eq(DAY), eq(0), eq((String) null), anyInt()))
                .andReturn(COURSES);
        expect(courseScheduleDao.selectByCourseId(COURSE_ID)).andReturn(
                COURSE_SCHEDULES).times(COURSES.size());
        expect(commentDao.countByCourseId(COURSE_ID)).andReturn(COMMENT_COUNT)
                .times(COURSES.size());
    }

    @Test
    public void 曜日を指定して講義情報を検索する() throws JSONException {
        getCourses(r.queryParam("day", String.valueOf(DAY)));
    }

    public void record時限を指定して講義情報を検索する() {
        expect(courseDao.search(eq(0), eq(PERIOD), eq((String) null), anyInt()))
                .andReturn(COURSES);
        expect(courseScheduleDao.selectByCourseId(COURSE_ID)).andReturn(
                COURSE_SCHEDULES).times(COURSES.size());
        expect(commentDao.countByCourseId(COURSE_ID)).andReturn(COMMENT_COUNT)
                .times(COURSES.size());
    }

    @Test
    public void 時限を指定して講義情報を検索する() throws JSONException {
        getCourses(r.queryParam("period", String.valueOf(PERIOD)));
    }

    public void record検索文字列を指定して講義情報を検索する() {
        expect(courseDao.search(eq(0), eq(0), eq(Q), anyInt())).andReturn(
                COURSES);
        expect(courseScheduleDao.selectByCourseId(COURSE_ID)).andReturn(
                COURSE_SCHEDULES).times(COURSES.size());
        expect(commentDao.countByCourseId(COURSE_ID)).andReturn(COMMENT_COUNT)
                .times(COURSES.size());
    }

    @Test
    public void 検索文字列を指定して講義情報を検索する() throws JSONException {
        getCourses(r.queryParam("q", Q));
    }

    private JSONObject getCourses(WebResource r) {
        return r.path("/courses.json").get(JSONObject.class);
    }

    public void record講義IDを指定して講義情報を取得する() {
        expect(courseDao.select(COURSE_ID)).andReturn(COURSE0);
        expect(courseScheduleDao.selectByCourseId(COURSE_ID)).andReturn(
                COURSE_SCHEDULES);
        expect(commentDao.countByCourseId(COURSE_ID)).andReturn(COMMENT_COUNT);
    }

    @Test
    public void 講義IDを指定して講義情報を取得する() {
        JSONObject course = getCourse();
        assertThat(course.length(), is(8));
    }

    public void record存在しない講義IDを指定して講義情報を取得しようとすると404が返る() {
        expect(courseDao.select(COURSE_ID)).andReturn(null);
    }

    @Test(expected = UniformInterfaceException.class)
    public void 存在しない講義IDを指定して講義情報を取得しようとすると404が返る() {
        getCourseAndExpectNotFoundResponse();
        fail();
    }

    private void getCourseAndExpectNotFoundResponse() {
        try {
            getCourse();
        } catch (UniformInterfaceException e) {
            assertThat(e.getResponse().getClientResponseStatus(),
                    is(Status.NOT_FOUND));
            throw e;
        }
        fail();
    }

    private JSONObject getCourse() {
        return r.path("/courses/" + COURSE_ID + ".json").get(JSONObject.class);
    }

}
