package jp.sourceforge.observoice.resources;

import static org.easymock.EasyMock.anyInt;
import static org.easymock.EasyMock.eq;
import static org.easymock.EasyMock.expect;
import static org.easymock.EasyMock.isA;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import jp.sourceforge.observoice.dao.CommentDao;
import jp.sourceforge.observoice.entity.Comment;
import jp.sourceforge.observoice.util.DateFormatUtil;

import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.junit.Test;
import org.seasar.framework.unit.annotation.EasyMock;

import com.sun.jersey.api.client.ClientResponse.Status;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.representation.Form;

public class CommentResourceTest extends ResourceTest {

    @EasyMock(register = true)
    private CommentDao commentDao;

    private static final int COURSE_ID = 100;
    private static final Date SINCE = newDate();
    private static final Date UNTIL = newDate();
    private static final String ILLEGAL_FORMAT_DATE = "2010/01/01";
    private static final Comment COMMENT0 = newComment(0);
    private static final Comment COMMENT1 = newComment(1);
    private static final List<Comment> COMMENTS = Arrays.asList(COMMENT0,
            COMMENT1);

    private static Date newDate() {
        try {
            return DateFormatUtil.parseRfc2822(DateFormatUtil
                    .formatRfc2822(new Date()));
        } catch (ParseException e) {
            throw new RuntimeException(e);
        }
    }

    private static Comment newComment(long commentId) {
        Comment c = new Comment();
        c.commentId = commentId;
        c.courseId = COURSE_ID;
        c.userId = "12345";
        c.type = 0;
        c.content = "コメント" + commentId;
        c.created = new Date();
        return c;
    }

    private JSONObject getComments(long courseId, WebResource r) {
        return r.path("courses/" + COURSE_ID + "/comments.json").get(
                JSONObject.class);
    }

    public void record範囲を指定せずに講義のコメントを取得する() {
        expect(commentDao.selectByCourseId(eq(COURSE_ID), anyInt())).andReturn(
                COMMENTS);
    }

    @Test
    public void 範囲を指定せずに講義のコメントを取得する() throws JSONException {
        JSONObject response = getComments(COURSE_ID, r);
        assertThat(response.length(), is(2));
        assertThat(response.getJSONArray("result").length(),
                is(response.getInt("count")));
        assertThat(response.getInt("count"), is(2));
    }

    public void recordはじまりを指定して講義のコメントを取得する() {
        expect(
                commentDao.selectByCourseIdAfter(eq(COURSE_ID), eq(SINCE),
                        anyInt())).andReturn(COMMENTS);
    }

    public void はじまりを指定して講義のコメントを取得する() {
        JSONObject response = getComments(COURSE_ID,
                r.queryParam("since", DateFormatUtil.formatRfc2822(SINCE)));
        assertThat(response.length(), is(2));
    }

    public void recordおわりを指定して講義のコメントを取得する() {
        expect(
                commentDao.selectByCourseIdBefore(eq(COURSE_ID), eq(SINCE),
                        anyInt())).andReturn(COMMENTS);
    }

    public void おわりを指定して講義のコメントを取得する() {
        JSONObject response = getComments(COURSE_ID,
                r.queryParam("until", DateFormatUtil.formatRfc2822(UNTIL)));
        assertThat(response.length(), is(2));
    }

    public void recordはじまりとおわりを指定して講義のコメントを取得する() {
        expect(
                commentDao.selectByCourseIdBetween(eq(COURSE_ID), eq(SINCE),
                        eq(UNTIL), anyInt())).andReturn(COMMENTS);
    }

    public void はじまりとおわりを指定して講義のコメントを取得する() {
        JSONObject response = getComments(
                COURSE_ID,
                r.queryParam("since", DateFormatUtil.formatRfc2822(SINCE))
                        .queryParam("until",
                                DateFormatUtil.formatRfc2822(UNTIL)));
        assertThat(response.length(), is(2));
    }

    @Test(expected = UniformInterfaceException.class)
    public void 不正な日付形式を使って講義のコメントを取得しようとすると400が返る() {
        getCommentsAndExpectBadRequestResponse(COURSE_ID,
                r.queryParam("since", ILLEGAL_FORMAT_DATE));
        fail();
    }

    private void getCommentsAndExpectBadRequestResponse(long courseId,
            WebResource r) {
        try {
            getComments(COURSE_ID, r);
        } catch (UniformInterfaceException e) {
            assertThat(e.getResponse().getClientResponseStatus(),
                    is(Status.BAD_REQUEST));
            throw e;
        }
    }

    private JSONObject postComment(long courseId, Object request) {
        return r.path("courses/" + courseId + "/comments/post.json").post(
                JSONObject.class, request);
    }

    public void record内容付きのコメントを投稿する() {
        expect(commentDao.insert(isA(Comment.class))).andReturn(1);
    }

    @Test
    public void 内容付きのコメントを投稿する() throws JSONException {
        Form request = new Form();
        request.add("userId", COMMENT0.userId);
        request.add("type", COMMENT0.type);
        request.add("content", COMMENT0.content);
        JSONObject response = postComment(COURSE_ID, request);

        assertThat(response.length(), is(4));
        assertThat(response.getString("userId"), is(COMMENT0.userId));
        assertThat(response.getInt("type"), is(COMMENT0.type));
        assertThat(response.getString("content"), is(COMMENT0.content));
        assertThat(response.has("created"), is(true));
    }

    public void record内容がないコメントを投稿する() {
        expect(commentDao.insert(isA(Comment.class))).andReturn(1);
    }

    @Test
    public void 内容がないコメントを投稿する() throws JSONException {
        Form request = new Form();
        request.add("userId", COMMENT0.userId);
        request.add("type", COMMENT0.type);
        JSONObject response = postComment(COURSE_ID, request);

        assertThat(response.length(), is(4));
        assertThat(response.getString("userId"), is(COMMENT0.userId));
        assertThat(response.getInt("type"), is(COMMENT0.type));
        assertThat(response.get("content"), is(JSONObject.NULL));
        assertThat(response.has("created"), is(true));
    }

    @Test(expected = UniformInterfaceException.class)
    public void ユーザIDが付与されていないコメントを投稿すると400が返る() {
        Form request = new Form();
        request.add("type", COMMENT0.type);

        postCommentAndExpectBadRequestResponse(request);
        fail();
    }

    @Test(expected = UniformInterfaceException.class)
    public void 理解サインが付与されていないコメントを投稿すると400が返る() {
        Form request = new Form();
        request.add("userId", COMMENT0.userId);

        postCommentAndExpectBadRequestResponse(request);
        fail();
    }

    private void postCommentAndExpectBadRequestResponse(Form request) {
        try {
            postComment(COURSE_ID, request);
        } catch (UniformInterfaceException e) {
            assertThat(e.getResponse().getClientResponseStatus(),
                    is(Status.BAD_REQUEST));
            throw e;
        }
    }

}
