/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.xwork2.validator.validators;

import com.opensymphony.xwork2.validator.ValidationException;

/**
 * <!-- START SNIPPET: javadoc -->
 * StringLengthFieldValidator checks that a String field is of a certain length.  If the "minLength"
 * parameter is specified, it will make sure that the String has at least that many characters.  If
 * the "maxLength" parameter is specified, it will make sure that the String has at most that many
 * characters.  The "trim" parameter determines whether it will {space: @link String#trim(), blank: @link StringUtils#trimBlank()} the
 * String before performing the length check.  If unspecified, the String will be trimmed.
 * <!-- END SNIPPET: javadoc -->
 * <p/>
 *
 *
 * <!-- START SNIPPET: parameters -->
 * <ul>
 * 	  <li>fieldName - The field name this validator is validating. Required if using Plain-Validator Syntax otherwise not required</li>
 *    <li>maxLength - The max length of the field value. Default ignore.</li>
 *    <li>minLength - The min length of the field value. Default ignore.</li>
 *    <li>trim - Trim the field value before evaluating its min/max length. ("space" or "blank" or "") Default is "blank"</li>
 * </ul>
 * <!-- END SNIPPET: parameters -->
 *
 *
 * <pre>
 * <!--START SNIPPET: example -->
 *		&lt;validators&gt;
 *          &lt;!-- Plain Validator Syntax --&gt;
 * 			&lt;validator type="stringlength"&gt;
 * 				&lt;param name="fieldName"&gt;myPurchaseCode&lt;/param&gt;
 * 				&lt;param name="minLength"&gt;10&lt;/param&gt;
 *              &lt;param name="maxLength"&gt;10&lt;/param&gt;
 *              &lt;param name="trim"&gt;blank&lt;/param&gt;
 *              &lt;message&gt;Your purchase code needs to be 10 characters long&lt;/message&gt;
 *          &lt;/validator&gt;
 *
 * 			&lt;!-- Field Validator Syntax --&gt;
 * 			&lt;field name="myPurchaseCode"&gt;
 * 				&lt;param name="minLength"&gt;10&lt;/param&gt;
 *              &lt;param name="maxLength&gt;10&lt;/param&gt;
 *              &lt;param name="trim"&gt;space&lt;/param&gt;
 *              &lt;message&gt;Your purchase code needs to be 10 characters long&lt;/message&gt;
 * 			&lt;/field-name&gt;
 *      &lt;/validators&gt;
 * <!-- END SNIPPET: example -->
 * </pre>
 *
 */
public class StringLengthFieldValidator extends AbstractStringFieldValidator {

	private Integer maxLength = null;
	private Integer minLength = null;
	private Integer length = null;

	/**
	 * @return the maxLength
	 */
	public Integer getMaxLength() {
		return maxLength;
	}

	/**
	 * @param maxLength the maxLength to set
	 */
	public void setMaxLength(Integer maxLength) {
		this.maxLength = maxLength;
	}

	/**
	 * @return the minLength
	 */
	public Integer getMinLength() {
		return minLength;
	}

	/**
	 * @param minLength the minLength to set
	 */
	public void setMinLength(Integer minLength) {
		this.minLength = minLength;
	}

	/**
	 * @return the length
	 */
	public Integer getLength() {
		return length;
	}

	/**
	 * @param length the length to set
	 */
	public void setLength(Integer length) {
		this.length = length;
	}

	@Override
	protected void validateString(Object object, String value) throws ValidationException {
		length = value.length();

		// only check for a minimum value if the min parameter is set
		if (minLength != null && length < minLength) {
			addFieldError(getFieldName(), object);
			return;
		}

		// only check for a maximum value if the max parameter is set
		if (maxLength != null && length > maxLength) {
			addFieldError(getFieldName(), object);
			return;
		}
	}
}
