/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.xwork2.validator.validators;

import nuts.core.lang.StringUtils;

import com.opensymphony.xwork2.validator.ValidationException;


/**
 * <!-- START SNIPPET: javadoc -->
 * RequiredStringFieldValidator checks that a String field is non-null and has a length > 0.
 * (i.e. it isn't "").  The "trim" parameter determines whether it will {space: @link String#trim(), blank: @link StringUtils#trimBlank()}
 * the String before performing the length check.  If unspecified, the String will be trimmed.
 * <!-- END SNIPPET: javadoc -->
 * <p/>
 *
 * <!-- START SNIPPET: parameters -->
 * <ul>
 * 		<li>fieldName - The field name this validator is validating. Required if using Plain-Validator Syntax otherwise not required</li>
 *      <li>trim - trim the field name value before validating (default is blank)</li>
 * </ul>
 * <!-- END SNIPPET: parameters -->
 *
 *
 * <pre>
 * <!-- START SNIPPET: examples -->
 *     &lt;validators&gt;
 *         &lt;!-- Plain-Validator Syntax --&gt;
 *         &lt;validator type="requiredstring"&gt;
 *             &lt;param name="fieldName"&gt;username&lt;/param&gt;
 *             &lt;param name="trim"&gt;spance&lt;/param&gt;
 *             &lt;message&gt;username is required&lt;/message&gt;
 *         &lt;/validator&gt;
 *
 *         &lt;!-- Field-Validator Syntax --&gt;
 *         &lt;field name="username"&gt;
 *         	  &lt;field-validator type="requiredstring"&gt;
 *                 &lt;param name="trim"&gt;blank&lt;/param&gt;
 *                 &lt;message&gt;username is required&lt;/message&gt;
 *            &lt;/field-validator&gt;
 *         &lt;/field&gt;
 *     &lt;/validators&gt;
 * <!-- END SNIPPET: examples -->
 * </pre>
 *
 */
public class RequiredStringFieldValidator extends AbstractStringFieldValidator {

	/**
	 * @see nuts.ext.xwork2.validator.validators.AbstractStringFieldValidator#validate(java.lang.Object)
	 */
	@Override
	public void validate(Object object) throws ValidationException {
		String value = trimFieldValue(object);
		validateString(object, value);
	}

	@Override
	protected void validateString(Object object, String value) throws ValidationException {
		if (StringUtils.isEmpty(value)) {
			addFieldError(getFieldName(), object);
		}
	}
}
