/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.xwork2;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.opensymphony.xwork2.Action;
import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.LocaleProvider;
import com.opensymphony.xwork2.Validateable;
import com.opensymphony.xwork2.interceptor.ValidationWorkflowAware;
import com.opensymphony.xwork2.util.ValueStack;


/**
 * Provides a default implementation for the most common actions.
 * See the documentation for all the interfaces this class implements for more detailed information.
 */
@SuppressWarnings("serial")
public class ActionSupport implements Action, Validateable, ValidationWorkflowAware, TextProvider,
		ActionValidationAware, ApplicationValidationAware, SessionValidationAware, LocaleProvider,
		StateProvider, Serializable {

	protected static Log log = LogFactory.getLog(ActionSupport.class);

    private transient TextProvider textProvider;
    
    private transient StateProvider stateProvider;

    private final ActionValidationAwareSupport actionValidationAware;

    private final ApplicationValidationAwareSupport applicationValidationAware;

    private final SessionValidationAwareSupport sessionValidationAware;

    /**
	 * Constructor
	 */
	public ActionSupport() {
		textProvider = new TextProviderFactory().createInstance(getClass(), this);
		stateProvider = new SessionStateProvider(getClass().getName() + '.');

		actionValidationAware = new ActionValidationAwareSupport();
		applicationValidationAware = new ApplicationValidationAwareSupport();
		sessionValidationAware = new SessionValidationAwareSupport();
	}

	/**
	 * @return the textProvider
	 */
	public TextProvider getTextProvider() {
		return textProvider;
	}

	/**
	 * @param textProvider the textProvider to set
	 */
	public void setTextProvider(TextProvider textProvider) {
		this.textProvider = textProvider;
	}

	/**
	 * @return the stateProvider
	 */
	public StateProvider getStateProvider() {
		return stateProvider;
	}

	/**
	 * @param stateProvider the stateProvider to set
	 */
	public void setStateProvider(StateProvider stateProvider) {
		this.stateProvider = stateProvider;
	}

	/**
     * @return "input"
     */
    public String getInputResultName() {
    	return INPUT;
    }
    
    /**
     * @return "input"
     * @throws Exception if an error occurs
     */
    public String input() throws Exception {
        return INPUT;
    }
    
    /**
     * @return "success"
     * @throws Exception if an error occurs
     */
    public String doDefault() throws Exception {
        return SUCCESS;
    }

    /**
     * A default implementation that does nothing an returns "success".
     * <p/>
     * Subclasses should override this method to provide their business logic.
     * <p/>
     * See also {@link com.opensymphony.xwork2.Action#execute()}.
     *
     * @return returns {@link #SUCCESS}
     * @throws Exception  can be thrown by subclasses.
     */
    public String execute() throws Exception {
        return SUCCESS;
    }

    /**
     * A default implementation that validates nothing.
     * Subclasses should override this method to provide validations.
     */
    public void validate() {
    }

    /**
     * <!-- START SNIPPET: pause-method -->
     * Stops the action invocation immediately (by throwing a PauseException) and causes the action invocation to return
     * the specified result, such as {@link #SUCCESS}, {@link #INPUT}, etc.
     * <p/>
     *
     * The next time this action is invoked (and using the same continuation ID), the method will resume immediately
     * after where this method was called, with the entire call stack in the execute method restored.
     * <p/>
     *
     * Note: this method can <b>only</b> be called within the {@link #execute()} method.
     * <!-- END SNIPPET: pause-method -->
     *
     * @param result the result to return - the same type of return value in the {@link #execute()} method.
     */
    public void pause(String result) {
    }

    /**
     * @see com.opensymphony.xwork2.LocaleProvider#getLocale()
     */
    public Locale getLocale() {
        ActionContext ctx = ActionContext.getContext();
        if (ctx != null) {
            return ctx.getLocale();
        } else {
            log.debug("Action context not initialized");
            return null;
        }
    }

    /**
	 * @see nuts.ext.xwork2.StateProvider#loadState(java.lang.String)
	 */
	public Object loadState(String name) {
		return stateProvider.loadState(name);
	}

	/**
	 * @see nuts.ext.xwork2.StateProvider#saveState(java.lang.String, java.lang.Object)
	 */
	public void saveState(String name, Object value) {
		stateProvider.saveState(name, value);
	}

	/**
     * @see com.opensymphony.xwork2.TextProvider#hasKey(java.lang.String)
     */
    public boolean hasKey(String key) {
        return textProvider.hasKey(key);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String)
     */
    public String getText(String aTextName) {
        return textProvider.getText(aTextName);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String)
     */
    public String getText(String aTextName, String defaultValue) {
        return textProvider.getText(aTextName, defaultValue);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String, java.lang.String)
     */
    public String getText(String aTextName, String defaultValue, String obj) {
        return textProvider.getText(aTextName, defaultValue, obj);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.util.List)
     */
    public String getText(String aTextName, List<Object> args) {
        return textProvider.getText(aTextName, args);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String[])
     */
    public String getText(String key, String[] args) {
        return textProvider.getText(key, args);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String, java.util.List)
     */
    public String getText(String aTextName, String defaultValue, List<Object> args) {
        return textProvider.getText(aTextName, defaultValue, args);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String, java.lang.String[])
     */
    public String getText(String key, String defaultValue, String[] args) {
        return textProvider.getText(key, defaultValue, args);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String, java.util.List, com.opensymphony.xwork2.util.ValueStack)
     */
    public String getText(String key, String defaultValue, List<Object> args, ValueStack stack) {
        return textProvider.getText(key, defaultValue, args, stack);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String, java.lang.String[], com.opensymphony.xwork2.util.ValueStack)
     */
    public String getText(String key, String defaultValue, String[] args, ValueStack stack) {
        return textProvider.getText(key, defaultValue, args, stack);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getTexts()
     */
    public ResourceBundle getTexts() {
        return textProvider.getTexts();
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getTexts(java.lang.String)
     */
    public ResourceBundle getTexts(String aBundleName) {
        return textProvider.getTexts(aBundleName);
    }

	/**
	 * getTextAsBoolean
	 * @param name resource name
	 * @return boolean value
	 */
	public Boolean getTextAsBoolean(String name) {
		return textProvider.getTextAsBoolean(name);
	}

	/**
	 * getTextAsBoolean
	 * @param name resource name
	 * @param defaultValue default value
	 * @return boolean value
	 */
	public Boolean getTextAsBoolean(String name, Boolean defaultValue) {
		return textProvider.getTextAsBoolean(name, defaultValue);
	}

	/**
	 * getTextAsInt
	 * @param name resource name
	 * @return integer value
	 */
	public Integer getTextAsInt(String name) {
		return textProvider.getTextAsInt(name);
	}

	/**
	 * getTextAsInt
	 * @param name resource name
	 * @param defaultValue default value
	 * @return integer value
	 */
	public Integer getTextAsInt(String name, Integer defaultValue) {
		return textProvider.getTextAsInt(name, defaultValue);
	}

	/**
	 * getTextAsLong
	 * @param name resource name
	 * @return long value
	 */
	public Long getTextAsLong(String name) {
		return textProvider.getTextAsLong(name);
	}

	/**
	 * getTextAsLong
	 * @param name resource name
	 * @param defaultValue default value
	 * @return long value
	 */
	public Long getTextAsLong(String name, Long defaultValue) {
		return textProvider.getTextAsLong(name, defaultValue);
	}

	/**
	 * getTextAsMap
	 * @param name resource name
	 * @return map value
	 */
	public Map getTextAsMap(String name) {
		return textProvider.getTextAsMap(name);
	}
	
	/**
	 * getTextAsMap
	 * @param name resource name
	 * @param defaultValue default value
	 * @return map value
	 */
	public Map getTextAsMap(String name, Map defaultValue) {
		return textProvider.getTextAsMap(name, defaultValue);
	}
	
	/**
	 * @see com.opensymphony.xwork2.ValidationAware#setFieldErrors(java.util.Map)
	 */
	public void setFieldErrors(Map<String, List<String>> errorMap) {
        actionValidationAware.setFieldErrors(errorMap);
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#getFieldErrors()
     */
    public Map<String, List<String>> getFieldErrors() {
        return actionValidationAware.getFieldErrors();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#setActionErrors(java.util.Collection)
     */
    public void setActionErrors(Collection<String> errorMessages) {
        actionValidationAware.setActionErrors(errorMessages);
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#getActionErrors()
     */
    public Collection<String> getActionErrors() {
        return actionValidationAware.getActionErrors();
    }

    /**
     * @see nuts.ext.xwork2.ActionValidationAware#setActionWarnings(java.util.Collection)
     */
    public void setActionWarnings(Collection<String> warnMessages) {
        actionValidationAware.setActionWarnings(warnMessages);
    }

    /**
     * @see nuts.ext.xwork2.ActionValidationAware#getActionWarnings()
     */
    public Collection<String> getActionWarnings() {
        return actionValidationAware.getActionWarnings();
    }

    /**
     * @see nuts.ext.xwork2.ActionValidationAware#setActionConfirms(java.util.Collection)
     */
    public void setActionConfirms(Collection<String> confirmMessages) {
        actionValidationAware.setActionConfirms(confirmMessages);
    }

    /**
     * @see nuts.ext.xwork2.ActionValidationAware#getActionConfirms()
     */
    public Collection<String> getActionConfirms() {
        return actionValidationAware.getActionConfirms();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#setActionMessages(java.util.Collection)
     */
    public void setActionMessages(Collection<String> messages) {
        actionValidationAware.setActionMessages(messages);
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#getActionMessages()
     */
    public Collection<String> getActionMessages() {
        return actionValidationAware.getActionMessages();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#addFieldError(java.lang.String, java.lang.String)
     */
    public void addFieldError(String fieldName, String errorMessage) {
        actionValidationAware.addFieldError(fieldName, errorMessage);
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#addActionError(java.lang.String)
     */
    public void addActionError(String anErrorMessage) {
        actionValidationAware.addActionError(anErrorMessage);
    }

    /**
     * @see nuts.ext.xwork2.ActionValidationAware#addActionWarning(java.lang.String)
     */
    public void addActionWarning(String anWarnMessage) {
        actionValidationAware.addActionWarning(anWarnMessage);
    }

    /**
     * @see nuts.ext.xwork2.ActionValidationAware#addActionConfirm(java.lang.String)
     */
    public void addActionConfirm(String anConfirmMessage) {
        actionValidationAware.addActionConfirm(anConfirmMessage);
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#addActionMessage(java.lang.String)
     */
    public void addActionMessage(String aMessage) {
        actionValidationAware.addActionMessage(aMessage);
    }
    /**
     * @see com.opensymphony.xwork2.ValidationAware#hasFieldErrors()
     */
    public boolean hasFieldErrors() {
        return actionValidationAware.hasFieldErrors();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#hasActionErrors()
     */
    public boolean hasActionErrors() {
        return actionValidationAware.hasActionErrors();
    }

    /**
     * @see nuts.ext.xwork2.ActionValidationAware#hasActionWarnings()
     */
    public boolean hasActionWarnings() {
        return actionValidationAware.hasActionWarnings();
    }

    /**
     * @see nuts.ext.xwork2.ActionValidationAware#hasActionConfirms()
     */
    public boolean hasActionConfirms() {
        return actionValidationAware.hasActionConfirms();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#hasActionMessages()
     */
    public boolean hasActionMessages() {
        return actionValidationAware.hasActionMessages();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#hasErrors()
     */
    public boolean hasErrors() {
        return actionValidationAware.hasErrors();
    }

    /**
     * @return true if action has messages
     */
    public boolean hasMessages() {
        return actionValidationAware.hasMessages();
    }

    /**
     * Clears field errors. Useful for Continuations and other situations
     * where you might want to clear parts of the state on the same action.
     */
    public void clearFieldErrors() {
        actionValidationAware.clearFieldErrors();
    }

    /**
     * Clears action errors. Useful for Continuations and other situations
     * where you might want to clear parts of the state on the same action.
     */
    public void clearActionErrors() {
        actionValidationAware.clearActionErrors();
    }

    /**
     * Clears all errors. Useful for Continuations and other situations
     * where you might want to clear parts of the state on the same action.
     */
    public void clearErrors() {
        actionValidationAware.clearErrors();
    }

	/**
	 * Clears action warning list.
	 * <p/>
	 * Will clear the list that contains action warning.
	 */
	public synchronized void clearActionWarnings() {
		actionValidationAware.clearActionWarnings();
	}

	/**
	 * Clears action confirm list.
	 * <p/>
	 * Will clear the list that contains action confirm.
	 */
	public synchronized void clearActionConfirms() {
		actionValidationAware.clearActionConfirms();
	}

	/**
	 * Clears action messages list.
	 * <p/>
	 * Will clear the list that contains action messages.
	 */
	public synchronized void clearActionMessages() {
		actionValidationAware.clearActionMessages();
	}

	/**
     * Clears messages. Useful for Continuations and other situations
     * where you might want to clear parts of the state on the same action.
     */
    public void clearMessages() {
        actionValidationAware.clearMessages();
    }

    /**
     * Clears all errors and messages. Useful for Continuations and other situations
     * where you might want to clear parts of the state on the same action.
     */
    public void clearErrorsAndMessages() {
        actionValidationAware.clearErrorsAndMessages();
    }

	/**
	 * @see ApplicationValidationAware#setApplicationErrors(java.util.Map)
	 */
	public synchronized void setApplicationErrors(Map<String, String> errorMessages) {
		applicationValidationAware.setApplicationErrors(errorMessages);
	}

	/**
	 * @see ApplicationValidationAware#getApplicationErrors()
	 */
	public synchronized Map<String, String> getApplicationErrors() {
		return applicationValidationAware.getApplicationErrors();
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#setApplicationWarnings(java.util.Map)
	 */
	public synchronized void setApplicationWarnings(Map<String, String> applicationWarnings) {
		applicationValidationAware.setApplicationWarnings(applicationWarnings);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#getApplicationWarnings()
	 */
	public synchronized Map<String, String> getApplicationWarnings() {
		return applicationValidationAware.getApplicationWarnings();
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#setApplicationConfirms(java.util.Map)
	 */
	public synchronized void setApplicationConfirms(Map<String, String> applicationConfirms) {
		applicationValidationAware.setApplicationConfirms(applicationConfirms);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#getApplicationConfirms()
	 */
	public synchronized Map<String, String> getApplicationConfirms() {
		return applicationValidationAware.getApplicationConfirms();
	}

	/**
	 * @see ApplicationValidationAware#setApplicationMessages(java.util.Map)
	 */
	public synchronized void setApplicationMessages(Map<String, String> messages) {
		applicationValidationAware.setApplicationMessages(messages);
	}

	/**
	 * @see ApplicationValidationAware#getApplicationMessages()
	 */
	public synchronized Map<String, String> getApplicationMessages() {
		return applicationValidationAware.getApplicationMessages();
	}

	/**
	 * @see ApplicationValidationAware#addApplicationError(java.lang.String, java.lang.String)
	 */
	public synchronized void addApplicationError(String id, String anErrorMessage) {
		applicationValidationAware.addApplicationError(id, anErrorMessage);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#addApplicationWarning(java.lang.String,
	 *      java.lang.String)
	 */
	public synchronized void addApplicationWarning(String id, String aWarningMessage) {
		applicationValidationAware.addApplicationWarning(id, aWarningMessage);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#addApplicationConfirm(java.lang.String,
	 *      java.lang.String)
	 */
	public synchronized void addApplicationConfirm(String id, String aConfirmMessage) {
		applicationValidationAware.addApplicationConfirm(id, aConfirmMessage);
	}

	/**
	 * @see ApplicationValidationAware#addApplicationMessage(java.lang.String, java.lang.String)
	 */
	public synchronized void addApplicationMessage(String id, String aMessage) {
		applicationValidationAware.addApplicationMessage(id, aMessage);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#removeApplicationError(java.lang.String)
	 */
	public String removeApplicationError(String id) {
		return applicationValidationAware.removeApplicationError(id);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#removeApplicationWarning(java.lang.String)
	 */
	public String removeApplicationWarning(String id) {
		return applicationValidationAware.removeApplicationWarning(id);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#removeApplicationConfirm(java.lang.String)
	 */
	public String removeApplicationConfirm(String id) {
		return applicationValidationAware.removeApplicationConfirm(id);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#removeApplicationMessage(java.lang.String)
	 */
	public String removeApplicationMessage(String id) {
		return applicationValidationAware.removeApplicationMessage(id);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#hasApplicationError(java.lang.String)
	 */
	public synchronized boolean hasApplicationError(String id) {
		return applicationValidationAware.hasApplicationError(id);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#hasApplicationWarning(java.lang.String)
	 */
	public synchronized boolean hasApplicationWarning(String id) {
		return applicationValidationAware.hasApplicationWarning(id);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#hasApplicationConfirm(java.lang.String)
	 */
	public synchronized boolean hasApplicationConfirm(String id) {
		return applicationValidationAware.hasApplicationConfirm(id);
	}

	/**
	 * @see ApplicationValidationAware#hasApplicationMessage(java.lang.String)
	 */
	public synchronized boolean hasApplicationMessage(String id) {
		return applicationValidationAware.hasApplicationMessage(id);
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#hasApplicationErrors()
	 */
	public synchronized boolean hasApplicationErrors() {
		return applicationValidationAware.hasApplicationErrors();
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#hasApplicationWarnings()
	 */
	public synchronized boolean hasApplicationWarnings() {
		return applicationValidationAware.hasApplicationWarnings();
	}

	/**
	 * @see nuts.ext.xwork2.ApplicationValidationAware#hasApplicationConfirms()
	 */
	public synchronized boolean hasApplicationConfirms() {
		return applicationValidationAware.hasApplicationConfirms();
	}

	/**
	 * @see ApplicationValidationAware#hasApplicationMessages()
	 */
	public synchronized boolean hasApplicationMessages() {
		return applicationValidationAware.hasApplicationMessages();
	}

	/**
	 * Clears application errors map.
	 * <p/>
	 * Will clear the map that contains application errors.
	 */
	public synchronized void clearApplicationErrors() {
		applicationValidationAware.clearApplicationErrors();
	}

	/**
	 * Clears application warning map.
	 * <p/>
	 * Will clear the map that contains application warning.
	 */
	public synchronized void clearApplicationWarnings() {
		applicationValidationAware.clearApplicationWarnings();
	}

	/**
	 * Clears application confirm map.
	 * <p/>
	 * Will clear the map that contains application confirm.
	 */
	public synchronized void clearApplicationConfirms() {
		applicationValidationAware.clearApplicationConfirms();
	}

	/**
	 * Clears application messages map.
	 * <p/>
	 * Will clear the map that contains application messages.
	 */
	public synchronized void clearApplicationMessages() {
		applicationValidationAware.clearApplicationMessages();
	}

	/**
	 * Clears all application error and messages.
	 * <p/>
	 * Will clear all the maps.
	 */
	public synchronized void clearApplicationErrorsAndMessages() {
		applicationValidationAware.clearErrorsAndMessages();
	}

	/**
	 * @see SessionValidationAware#setSessionErrors(java.util.Map)
	 */
	public synchronized void setSessionErrors(Map<String, String> errorMessages) {
		sessionValidationAware.setSessionErrors(errorMessages);
	}

	/**
	 * @see SessionValidationAware#getSessionErrors()
	 */
	public synchronized Map<String, String> getSessionErrors() {
		return sessionValidationAware.getSessionErrors();
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#setSessionWarnings(java.util.Map)
	 */
	public synchronized void setSessionWarnings(Map<String, String> sessionWarnings) {
		sessionValidationAware.setSessionWarnings(sessionWarnings);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#getSessionWarnings()
	 */
	public synchronized Map<String, String> getSessionWarnings() {
		return sessionValidationAware.getSessionWarnings();
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#setSessionConfirms(java.util.Map)
	 */
	public synchronized void setSessionConfirms(Map<String, String> sessionConfirms) {
		sessionValidationAware.setSessionConfirms(sessionConfirms);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#getSessionConfirms()
	 */
	public synchronized Map<String, String> getSessionConfirms() {
		return sessionValidationAware.getSessionConfirms();
	}

	/**
	 * @see SessionValidationAware#setSessionMessages(java.util.Map)
	 */
	public synchronized void setSessionMessages(Map<String, String> messages) {
		sessionValidationAware.setSessionMessages(messages);
	}

	/**
	 * @see SessionValidationAware#getSessionMessages()
	 */
	public synchronized Map<String, String> getSessionMessages() {
		return sessionValidationAware.getSessionMessages();
	}

	/**
	 * @see SessionValidationAware#addSessionError(java.lang.String, java.lang.String)
	 */
	public synchronized void addSessionError(String id, String anErrorMessage) {
		sessionValidationAware.addSessionError(id, anErrorMessage);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#addSessionWarning(java.lang.String,
	 *      java.lang.String)
	 */
	public synchronized void addSessionWarning(String id, String aWarningMessage) {
		sessionValidationAware.addSessionWarning(id, aWarningMessage);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#addSessionConfirm(java.lang.String,
	 *      java.lang.String)
	 */
	public synchronized void addSessionConfirm(String id, String aConfirmMessage) {
		sessionValidationAware.addSessionConfirm(id, aConfirmMessage);
	}

	/**
	 * @see SessionValidationAware#addSessionMessage(java.lang.String, java.lang.String)
	 */
	public synchronized void addSessionMessage(String id, String aMessage) {
		sessionValidationAware.addSessionMessage(id, aMessage);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#removeSessionError(java.lang.String)
	 */
	public String removeSessionError(String id) {
		return sessionValidationAware.removeSessionError(id);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#removeSessionWarning(java.lang.String)
	 */
	public String removeSessionWarning(String id) {
		return sessionValidationAware.removeSessionWarning(id);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#removeSessionConfirm(java.lang.String)
	 */
	public String removeSessionConfirm(String id) {
		return sessionValidationAware.removeSessionConfirm(id);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#removeSessionMessage(java.lang.String)
	 */
	public String removeSessionMessage(String id) {
		return sessionValidationAware.removeSessionMessage(id);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#hasSessionError(java.lang.String)
	 */
	public synchronized boolean hasSessionError(String id) {
		return sessionValidationAware.hasSessionError(id);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#hasSessionWarning(java.lang.String)
	 */
	public synchronized boolean hasSessionWarning(String id) {
		return sessionValidationAware.hasSessionWarning(id);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#hasSessionConfirm(java.lang.String)
	 */
	public synchronized boolean hasSessionConfirm(String id) {
		return sessionValidationAware.hasSessionConfirm(id);
	}

	/**
	 * @see SessionValidationAware#hasSessionMessage(java.lang.String)
	 */
	public synchronized boolean hasSessionMessage(String id) {
		return sessionValidationAware.hasSessionMessage(id);
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#hasSessionErrors()
	 */
	public synchronized boolean hasSessionErrors() {
		return sessionValidationAware.hasSessionErrors();
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#hasSessionWarnings()
	 */
	public synchronized boolean hasSessionWarnings() {
		return sessionValidationAware.hasSessionWarnings();
	}

	/**
	 * @see nuts.ext.xwork2.SessionValidationAware#hasSessionConfirms()
	 */
	public synchronized boolean hasSessionConfirms() {
		return sessionValidationAware.hasSessionConfirms();
	}

	/**
	 * @see SessionValidationAware#hasSessionMessages()
	 */
	public synchronized boolean hasSessionMessages() {
		return sessionValidationAware.hasSessionMessages();
	}

	/**
	 * Clears session errors map.
	 * <p/>
	 * Will clear the map that contains session errors.
	 */
	public synchronized void clearSessionErrors() {
		sessionValidationAware.clearSessionErrors();
	}

	/**
	 * Clears session warning map.
	 * <p/>
	 * Will clear the map that contains session warning.
	 */
	public synchronized void clearSessionWarnings() {
		sessionValidationAware.clearSessionWarnings();
	}

	/**
	 * Clears session confirm map.
	 * <p/>
	 * Will clear the map that contains session confirm.
	 */
	public synchronized void clearSessionConfirms() {
		sessionValidationAware.clearSessionConfirms();
	}

	/**
	 * Clears session messages map.
	 * <p/>
	 * Will clear the map that contains session messages.
	 */
	public synchronized void clearSessionMessages() {
		sessionValidationAware.clearSessionMessages();
	}

	/**
	 * Clears all session error and messages.
	 * <p/>
	 * Will clear all the maps.
	 */
	public synchronized void clearSessionErrorsAndMessages() {
		sessionValidationAware.clearErrorsAndMessages();
	}

	/**
     * Clone action
     * @return cloned object
     * @throws CloneNotSupportedException if clone failed
     */
    public Object clone() throws CloneNotSupportedException {
        return super.clone();
    }

}
