/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.struts2.util;

import java.text.SimpleDateFormat;
import java.util.Calendar;

import javax.servlet.http.HttpServletRequest;

import nuts.core.lang.ExceptionUtils;
import nuts.core.lang.StringUtils;
import nuts.core.servlet.HttpServletUtils;
import nuts.core.servlet.RequestLoggingFilter;
import nuts.core.util.Pager;
import nuts.core.util.Sorter;
import nuts.ext.struts2.actions.NutsTextConstants;
import nuts.ext.xwork2.StateProvider;


/**
 */
public class ActionUtilities extends ActionHelper {

	/**
	 * Constructor
	 */
	public ActionUtilities() {
		super();
	}

	/**
	 * @return servlet exception stack trace
	 */
	public String getServletExceptionStackTrace() {
		HttpServletRequest req = StrutsContextUtils.getServletRequest();
		Throwable ex = (Throwable)req.getAttribute("javax.servlet.error.exception");
		if (ex != null) {
			return ExceptionUtils.getStackTrace(ex);
		}
		else {
			return "";
		}
	}

	/**
	 * @return relativeURI
	 */
	public String getRelativeURI() {
		return HttpServletUtils.getRelativeURI(StrutsContextUtils.getServletRequest());
	}

	/**
	 * @return requestURI
	 */
	public String getRequestURI() {
		return HttpServletUtils.getRequestURI(StrutsContextUtils.getServletRequest());
	}
	
	/**
	 * @return requestURL
	 */
	public String getRequestURL() {
		return getRequestURL(0);
	}
	
	/**
	 * @param maxLength max length of url
	 * @return requestURL
	 */
	public String getRequestURL(int maxLength) {
		HttpServletRequest req = StrutsContextUtils.getServletRequest();
		String url = (String)req.getAttribute(RequestLoggingFilter.REQUEST_URL);
		if (StringUtils.isEmpty(url)) {
			url = (String)req.getAttribute("javax.servlet.forward.request_uri");
			if (StringUtils.isEmpty(url)) {
				url = req.getRequestURI();
			}
		}
		if (maxLength > 0 && url.length() > maxLength) {
			url = url.substring(0, maxLength) + "...";
		}
		return url;
	}
	
	/**
	 * @return requestElapsedTime
	 */
	public String getRequestElapsedTime() {
		HttpServletRequest req = StrutsContextUtils.getServletRequest();
		Long start = (Long)req.getAttribute(RequestLoggingFilter.REQUEST_TIME);
		if (start != null) {
			long end = System.currentTimeMillis();
			long elapse = end - start;
			if (elapse < 1000) {
				return elapse + "ms";
			}
			else {
				return (elapse / 1000) + "s";
			}
		}
		return "";
	}
	
	/**
	 * @return csv file time string
	 */
	public String getCsvFileTime() {
		String format = getAction().getText("date-format-file", "_(yyyyMMdd-HHmmss)");
		SimpleDateFormat sdf = new SimpleDateFormat(format);
		return sdf.format(Calendar.getInstance().getTime());
	}
	
	/**
	 * load sorter parameters from stateProvider
	 * @param sorter sorter
	 * @throws Exception if an error occurs
	 */
	public void loadSorterParams(Sorter sorter) throws Exception {
		StateProvider sp = getAction().getStateProvider();
		
		if (StringUtils.isBlank(sorter.getColumn())) {
			String sc = (String)sp.loadState("list.sc");
			if (StringUtils.isBlank(sc)) {
				sc = getAction().getText(NutsTextConstants.LIST_SORTER_COLUMN, (String)null);
			}
			if (StringUtils.isNotBlank(sc)) {
				sorter.setColumn(sc);
			}
		}
		if (StringUtils.isNotBlank(sorter.getColumn())) {
			if (StringUtils.isBlank(sorter.getDirection())) {
				String sd = (String)sp.loadState("list.sd");
				if (StringUtils.isBlank(sd)) {
					sd = getAction().getText(NutsTextConstants.LIST_SORTER_DIRECTION, Sorter.ASC);
				}
				sorter.setDirection(sd);
			}
		}
	}
	
	/**
	 * load pager limit parameters from stateProvider
	 * @param pager pager
	 * @throws Exception if an error occurs
	 */
	public void loadLimitParams(Pager pager) throws Exception {
		StateProvider sp = getAction().getStateProvider();
		if (pager.getLimit() == null) {
			try {
				pager.setLimit(Integer.parseInt((String)sp.loadState("list.pl")));
			}
			catch (Exception e) {
				pager.setLimit(getAction().getTextAsInt(NutsTextConstants.LIST_PAGER_LIMIT, 20));
			}
		}
		limitPage(pager);
	}
	
	/**
	 * save sorter parameters to stateProvider
	 * @param sorter sorter
	 * @throws Exception if an error occurs
	 */
	public void saveSorterParams(Sorter sorter) throws Exception {
		StateProvider sp = getAction().getStateProvider();
		sp.saveState("list.sc", sorter.getColumn());
		sp.saveState("list.sd", sorter.getDirection());
	}
	
	/**
	 * save pager limit parameters to stateProvider
	 * @param pager pager
	 * @throws Exception if an error occurs
	 */
	public void saveLimitParams(Pager pager) throws Exception {
		StateProvider sp = getAction().getStateProvider();
		sp.saveState("list.pl", pager.getLimit());
	}
	
	/**
	 * if pager.limit > maxLimit then set pager.limit = maxLimit
	 * @param pager pager
	 */
	public void limitPage(Pager pager) {
		int maxLimit = getAction().getTextAsInt(NutsTextConstants.PAGER_MAX_LIMIT, 100);
		if (pager.getLimit() > maxLimit) {
			pager.setLimit(maxLimit);
		}
	}
	
	/**
	 * hasPermit
	 * @param action action
	 * @return true if action has access permit
	 */
	public boolean hasPermit(String action) {
		return true;
	}
}
