/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.struts2.components;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.struts2.views.annotations.StrutsTag;
import org.apache.struts2.views.annotations.StrutsTagAttribute;

import com.opensymphony.xwork2.util.ValueStack;

/**
 * <!-- START SNIPPET: javadoc --> Render an HTML input tag of type select. <!-- END SNIPPET:
 * javadoc -->
 * <p/>
 * <b>Examples</b>
 * 
 * <pre>
 * &lt;!-- START SNIPPET: example --&gt;
 * 
 * &lt;s:select label=&quot;Pets&quot;
 *        name=&quot;petIds&quot;
 *        list=&quot;petDao.pets&quot;
 *        listKey=&quot;id&quot;
 *        listValue=&quot;name&quot;
 *        multiple=&quot;true&quot;
 *        size=&quot;3&quot;
 *        required=&quot;true&quot;
 *        value=&quot;%{petDao.pets.{id}}&quot;
 * /&gt;
 * 
 * &lt;s:select label=&quot;Months&quot;
 *        name=&quot;months&quot;
 *        headerKey=&quot;-1&quot; headerValue=&quot;Select Month&quot;
 *        list=&quot;#{'01':'Jan', '02':'Feb', [...]}&quot;
 *        value=&quot;selectedMonth&quot;
 *        required=&quot;true&quot;
 * /&gt;
 * 
 * // The month id (01, 02, ...) returned by the getSelectedMonth() call
 * // against the stack will be auto-selected
 * 
 * &lt;!-- END SNIPPET: example --&gt;
 * </pre>
 * <p/>
 * <!-- START SNIPPET: exnote --> Note: For any of the tags that use lists (select probably being
 * the most ubiquitous), which uses the OGNL list notation (see the "months" example above), it
 * should be noted that the map key created (in the months example, the '01', '02', etc.) is typed.
 * '1' is a char, '01' is a String, "1" is a String. This is important since if the value returned
 * by your "value" attribute is NOT the same type as the key in the "list" attribute, they WILL NOT
 * MATCH, even though their String values may be equivalent. If they don't match, nothing in your
 * list will be auto-selected.
 * <p/>
 * <!-- END SNIPPET: exnote -->
 */
@StrutsTag(
		name = "select", 
		tldTagClass = "nuts.ext.struts2.views.jsp.ui.SelectTag", 
		description = "Render a select element", 
		allowDynamicAttributes = true)
public class Select extends ListUIBean {
	private final static String TEMPLATE = "n-select";

	protected String emptyOption;
	protected String headerKey;
	protected String headerValue;
	protected String multiple;
	protected String size;

	/**
	 * Constructor
	 * 
	 * @param stack value stack
	 * @param request request
	 * @param response response
	 */
	public Select(ValueStack stack, HttpServletRequest request, HttpServletResponse response) {
		super(stack, request, response);
	}

	protected String getDefaultTemplate() {
		return TEMPLATE;
	}

	/**
	 * evaluate extra parameters
	 */
	public void evaluateExtraParams() {
		super.evaluateExtraParams();

		if (emptyOption != null) {
			addParameter("emptyOption", findValue(emptyOption, Boolean.class));
		}

		if (multiple != null) {
			addParameter("multiple", findValue(multiple, Boolean.class));
		}

		if (size != null) {
			addParameter("size", findString(size));
		}

		if ((headerKey != null) && (headerValue != null)) {
			addParameter("headerKey", findString(headerKey));
			addParameter("headerValue", findString(headerValue));
		}
	}

	/**
	 * @param emptyOption the emptyOption to set
	 */
	@StrutsTagAttribute(description = "Whether or not to add an empty (--) option after the header option", type = "Boolean", defaultValue = "false")
	public void setEmptyOption(String emptyOption) {
		this.emptyOption = emptyOption;
	}

	/**
	 * @param headerKey the headerKey to set
	 */
	@StrutsTagAttribute(description = " Key for first item in list. Must not be empty! '-1' and '' is correct, '' is bad.")
	public void setHeaderKey(String headerKey) {
		this.headerKey = headerKey;
	}

	/**
	 * @param headerValue the headerValue to set
	 */
	@StrutsTagAttribute(description = "Value expression for first item in list")
	public void setHeaderValue(String headerValue) {
		this.headerValue = headerValue;
	}

	/**
	 * @param multiple the multiple to set
	 */
	@StrutsTagAttribute(description = " Creates a multiple select. The tag will pre-select multiple values"
			+ " if the values are passed as an Array or a Collection(of appropriate types) via the value attribute. If one of the keys equals"
			+ " one of the values in the Collection or Array it wil be selected", type = "Boolean", defaultValue = "false")
	public void setMultiple(String multiple) {
		this.multiple = multiple;
	}

	/**
	 * @param size the size to set
	 */
	@StrutsTagAttribute(description = "Size of the element box (# of elements to show)", type = "Integer")
	public void setSize(String size) {
		this.size = size;
	}
}
