/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.struts2.components;

import java.io.IOException;
import java.io.Writer;
import java.util.Date;

import nuts.core.lang.StringEscapeUtils;
import nuts.ext.xwork2.util.ContextUtils;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.components.ContextBean;
import org.apache.struts2.views.annotations.StrutsTag;
import org.apache.struts2.views.annotations.StrutsTagAttribute;

import com.opensymphony.xwork2.TextProvider;
import com.opensymphony.xwork2.util.ValueStack;

/**
 * <!-- START SNIPPET: javadoc -->
 *
 * Used to get the property of a <i>value</i>, which will default to the top of
 * the stack if none is specified.
 *
 * <!-- END SNIPPET: javadoc -->
 *
 * <p/>
 *
 *
 * <!-- START SNIPPET: params -->
 *
 * <ul>
 * <li>default (String) - The default value to be used if <u>value</u> attribute
 * is null</li>
 * <li>escape (Boolean) - Escape HTML. Default to true</li>
 * <li>value (Object) - value to be displayed</li>
 * </ul>
 *
 * <!-- END SNIPPET: params -->
 *
 *
 * <pre>
 * &lt;!-- START SNIPPET: example --&gt;
 *
 * &lt;s:push value=&quot;myBean&quot;&gt;
 *     &lt;!-- Example 1: --&gt;
 *     &lt;s:property value=&quot;myBeanProperty&quot; /&gt;
 *
 *     &lt;!-- Example 2: --&gt;TextUtils
 *     &lt;s:property value=&quot;myBeanProperty&quot; default=&quot;a default value&quot; /&gt;
 * &lt;/s:push&gt;
 *
 * &lt;!-- END SNIPPET: example --&gt;
 * </pre>
 *
 * <pre>
 * &lt;!-- START SNIPPET: exampledescription --&gt;
 *
 * Example 1 prints the result of myBean's getMyBeanProperty() method.
 * Example 2 prints the result of myBean's getMyBeanProperty() method and if it is null, print 'a default value' instead.
 *
 * &lt;!-- END SNIPPET: exampledescription --&gt;
 * </pre>
 *
 *
 * <pre>
 * &lt;!-- START SNIPPET: i18nExample --&gt;
 *
 * &lt;s:property value=&quot;getText('some.key')&quot; /&gt;
 *
 * &lt;!-- END SNIPPET: i18nExample --&gt;
 * </pre>
 *
 */
@StrutsTag(
		name = "property",
		tldBodyContent = "empty",
		tldTagClass = "nuts.ext.struts2.views.jsp.PropertyTag",
		description = "Print out expression which evaluates against the stack")
public class Property extends ContextBean {
	private static final Log log = LogFactory.getLog(Property.class);

	/**
	 * Constructor
	 * 
	 * @param stack value stack
	 */
	public Property(ValueStack stack) {
		super(stack);
	}

	/**
	 * PASSWORD_FORMAT = "password-format";
	 */
	public static final String PASSWORD_FORMAT = "password-format";

	/**
	 * ESCAPE_HTML = "html";
	 */
	public static final String ESCAPE_HTML = "html";
	
	/**
	 * ESCAPE_JAVASCRIPT = "js";
	 */
	public static final String ESCAPE_JAVASCRIPT = "js";
	
	/**
	 * DEFAULT_PASSWORD_FORMAT = "******";
	 */
	public static final String DEFAULT_PASSWORD_FORMAT = "******";

	private String defaultValue;
	private String value;
	private String format;
	private Object rawvalue;
	private String escape = ESCAPE_HTML;

	/**
	 * @see org.apache.struts2.components.Component#start(java.io.Writer)
	 */
	public boolean start(Writer writer) {
		boolean result = super.start(writer);

		boolean pushed = false;

		String actualValue = null;

		if (rawvalue != null) {
			getStack().push(rawvalue);
			pushed = true;
			value = "top";
		}

		if (value == null) {
			value = "top";
		}
		else {
			value = stripExpressionIfAltSyntax(value);
		}

		if (format == null) {
			// exception: don't call findString(), since we don't want the
			// expression parsed in this one case. it really
			// doesn't make sense, in fact.
			actualValue = (String) getStack().findValue(value, String.class);
		}
		else {
			Object av = getStack().findValue(value);

			if (av == null) {
			}
			else if ("password".equalsIgnoreCase(format)) {
				TextProvider tp = ContextUtils.findTextProviderInStack(getStack());
				actualValue = tp.getText(PASSWORD_FORMAT, DEFAULT_PASSWORD_FORMAT);
			}
			else {
				if (av instanceof Boolean) {
					CBoolean b = new CBoolean(getStack());
					b.setVar(getVar());
					b.setName(value);
					b.setFormat(format);
					b.start(writer);
					b.end(writer, "");
				}
				else if (av instanceof Date) {
					CDate d = new CDate(getStack());
					d.setVar(getVar());
					d.setValue(av);
					d.setFormat(format);
					d.start(writer);
					d.end(writer, "");
				}
				else if (av instanceof Number) {
					CNumber n = new CNumber(getStack());
					n.setVar(getVar());
					n.setValue(av);
					n.setFormat(format);
					n.start(writer);
					n.end(writer, "");
				}
				else if (av instanceof String) {
					actualValue = (String)av;
				}
				else {
					actualValue = (String)getStack().findValue(value, String.class);
				}
			}
		}

		try {
			if (actualValue == null) {
				actualValue = defaultValue;
			}

			if (actualValue != null) {
				write(writer, actualValue);
			}
		}
		catch (IOException e) {
			log.warn("Could not print out value '" + actualValue + "'", e);
		}

		if (pushed) {
			getStack().pop();
		}
		return result;
	}

	private void write(Writer writer, String value) throws IOException {
		if (getVar() == null) {
			writer.write(prepare(value));
		}
		else {
			putInContext(prepare(value));
		}
	}

	private String prepare(String value) {
		String result = value;
		if (ESCAPE_HTML.equals(escape)) {
			result = StringEscapeUtils.escapeHtml(result);
		}
		else if (ESCAPE_JAVASCRIPT.equals(escape)) {
			result = StringEscapeUtils.escapeJavaScript(result);
		}
		return result;
	}

	/**
	 * @param defaultValue the defaultValue to set
	 */
	@StrutsTagAttribute(description = "The default value to be used if <u>value</u> attribute is null")
	public void setDefault(String defaultValue) {
		this.defaultValue = defaultValue;
	}

	/**
	 * @param escape the escape to set
	 */
	@StrutsTagAttribute(description = " Whether to escape HTML", defaultValue = "html")
	public void setEscape(String escape) {
		this.escape = escape;
	}

	/**
	 * @param rawvalue the rawvalue to set
	 */
	@StrutsTagAttribute(description = "Value to be displayed", type = "Object")
	public void setRawvalue(Object rawvalue) {
		this.rawvalue = rawvalue;
	}

	/**
	 * @param value the value to set
	 */
	@StrutsTagAttribute(description = "Value to be displayed", type = "String", defaultValue = "&lt;top of stack&gt;")
	public void setValue(String value) {
		this.value = value;
	}

	/**
	 * @param format the format to set
	 */
	@StrutsTagAttribute(description = "format to be used", type = "String")
	public void setFormat(String format) {
		this.format = format;
	}

}
