/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.struts2.components;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import nuts.core.lang.StringUtils;
import nuts.ext.struts2.NutsStrutsConstants;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.components.UIBean;
import org.apache.struts2.views.annotations.StrutsTag;
import org.apache.struts2.views.annotations.StrutsTagAttribute;

import com.opensymphony.xwork2.inject.Inject;
import com.opensymphony.xwork2.util.ValueStack;

/**
 * <!-- START SNIPPET: javadoc -->
 * <p>
 * Renders a jquery datetime picker.
 * </p>
 * 
 * <!-- END SNIPPET: javadoc -->
 * 
 * <b>Examples</b>
 * 
 * <pre>
 * &lt;!-- START SNIPPET: expl1 --&gt;
 * 
 * Example 1:
 *     &lt;r:datetimepicker name=&quot;order.date&quot; label=&quot;Order Date&quot; /&gt;
 * Example 2:
 *     &lt;r:datetimepicker name=&quot;delivery.date&quot; label=&quot;Delivery Date&quot; format=&quot;date&quot;  /&gt;
 * 
 * &lt;!-- END SNIPPET: expl1 --&gt;
 * </pre>
 * <p/>
 * 
 */
@StrutsTag(
		name = "datetimepicker", 
		tldTagClass = "nuts.ext.struts2.views.jsp.ui.DateTimePickerTag", 
		description = "Render datetimepicker", 
		allowDynamicAttributes = true)
public class DateTimePicker extends UIBean {
	/**
	 * TEMPLATE = "n-datetimepicker";
	 */
	public static final String TEMPLATE = "n-datetimepicker";

	protected static final Log log = LogFactory.getLog(DateTimePicker.class);

	protected static final String DEFAULT_DATETIME_FORMAT = "datetime";

	protected static final String DEFAULT_DATETIME_PATTERN = "yyyy-MM-dd HH:mm:ss";

	protected String maxlength;
	protected String readonly;
	protected String size;

	protected String inline;
	protected String format;
	protected String pattern;
	protected String options;
	protected String defaults;
	
	/**
	 * Constructor
	 * 
	 * @param stack value stack
	 * @param request request
	 * @param response response
	 */
	public DateTimePicker(ValueStack stack, HttpServletRequest request,
			HttpServletResponse response) {
		super(stack, request, response);
	}

	protected String getDefaultTemplate() {
		return TEMPLATE;
	}

	@SuppressWarnings("unchecked")
	protected Class getValueClassType() {
		return null;
	}

	/**
	 * Evaluate extra parameters
	 */
	protected void evaluateExtraParams() {
		super.evaluateExtraParams();

		if (size != null) {
			addParameter("size", findString(size));
		}

		if (maxlength != null) {
			addParameter("maxlength", findString(maxlength));
		}

		if (readonly != null) {
			addParameter("readonly", findValue(readonly, Boolean.class));
		}

		if (inline != null) {
			addParameter("inline", findValue(inline, Boolean.class));
		}

		addParameter("format", getDateTimeFormat());
		addParameter("pattern", getDateTimePattern());
		
		if (options != null) {
			addParameter("options", findString(options));
		}

		if (defaults != null) {
			addParameter("defaults", defaults);
		}
	}

	/**
	 * get date time format from text
	 * 
	 * @return date time format
	 */
	protected String getDateTimeFormat() {
		if (format != null) {
			addParameter("format", findString(format));
		}

		String value = null;
		if (format != null) {
			value = findString(format);
		}
		if (value == null) {
			value = (String)parameters.get("format");
		}
		if (StringUtils.isEmpty(value)) {
			value = DEFAULT_DATETIME_FORMAT;
		}		
		return value;
	}

	/**
	 * get date time pattern from text
	 * 
	 * @return date time pattern
	 */
	protected String getDateTimePattern() {
		String value = null;
		if (pattern != null) {
			value = findString(pattern);
		}
		if (value == null) {
			value = (String)parameters.get("pattern");
		}
		if (StringUtils.isEmpty(value)) {
			String format = (String)parameters.get("format");
			value = CDate.getDatePattern(getStack(), format);
		}
		if (StringUtils.isEmpty(value)) {
			value = DEFAULT_DATETIME_PATTERN;
		}		
		return value;
	}

	/**
     * @param defaults defaults
     */
    @Inject(value=NutsStrutsConstants.NUTS_DATETIMEPICKER_DEFAULTS, required=false)
    public void setDefaults(String defaults) {
    	this.defaults = defaults;
    }

	/**
	 * @param maxlength the maxlength to set
	 */
	@StrutsTagAttribute(description = "HTML maxlength attribute", type = "Integer")
	public void setMaxlength(String maxlength) {
		this.maxlength = maxlength;
	}

	/**
	 * @param maxlength the maxlength to set
	 */
	@StrutsTagAttribute(description = "Deprecated. Use maxlength instead.", type = "Integer")
	public void setMaxLength(String maxlength) {
		this.maxlength = maxlength;
	}

	/**
	 * @param readonly the readonly to set
	 */
	@StrutsTagAttribute(description = "Whether the input is readonly", type = "Boolean", defaultValue = "false")
	public void setReadonly(String readonly) {
		this.readonly = readonly;
	}

	/**
	 * @param size the size to set
	 */
	@StrutsTagAttribute(description = "HTML size attribute", type = "Integer")
	public void setSize(String size) {
		this.size = size;
	}

	/**
	 * @param inline the inline to set
	 */
	@StrutsTagAttribute(description = "The inline attribute", required = false, type = "Boolean", defaultValue = "false")
	public void setInline(String inline) {
		this.inline = inline;
	}

	/**
	 * @param format the format to set
	 */
	@StrutsTagAttribute(description = "The format attribute", required = false)
	public void setFormat(String format) {
		this.format = format;
	}

	/**
	 * @param pattern the pattern to set
	 */
	@StrutsTagAttribute(description = "The pattern attribute", required = false)
	public void setPattern(String pattern) {
		this.pattern = pattern;
	}

	/**
	 * @param options the options to set
	 */
	@StrutsTagAttribute(description = "The datetime picker javascript options", required = false)
	public void setOptions(String options) {
		this.options = options;
	}
}
