/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.struts2.components;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Map;

import nuts.core.io.CsvWriter;
import nuts.core.lang.StringUtils;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.components.Component;
import org.apache.struts2.util.MakeIterator;
import org.apache.struts2.views.annotations.StrutsTag;
import org.apache.struts2.views.annotations.StrutsTagAttribute;

import com.opensymphony.xwork2.util.ValueStack;

/**
 * Render CSV.
 * 
 * Example:
 * 
 * <pre>
 * &lt;s:csv list=&quot;list&quot; columns=&quot;columns&quot; columnKeys=&quot;columnKyes&quot; seperator=&quot;,&quot; quotechar='&quot;' escapechar='&quot;'/&gt;
 * 
 * </pre>
 * 
 */
@SuppressWarnings("unchecked")
@StrutsTag(
		name = "csv", 
		tldTagClass = "nuts.ext.struts2.views.jsp.CsvTag", 
		description = "Render CSV")
public class Csv extends Component {

	private static final org.apache.commons.logging.Log log = LogFactory.getLog(Csv.class);

	protected Object list;
	protected Object columns;

	protected String separator;
	protected String quotechar;
	protected String escapechar;

	/**
	 * Constructor
	 * 
	 * @param stack value stack
	 */
	public Csv(ValueStack stack) {
		super(stack);
	}

	/**
	 * @see org.apache.struts2.components.Component#end(java.io.Writer, java.lang.String)
	 */
	public boolean end(Writer writer, String body) {
		evaluateParams();

		ValueStack stack = getStack();

		CsvWriter cw = new CsvWriter(writer);
		if (StringUtils.isNotEmpty(separator)) {
			cw.setSeparator(separator.charAt(0));
		}
		if (StringUtils.isNotEmpty(quotechar)) {
			cw.setQuotechar(quotechar.charAt(0));
		}
		if (StringUtils.isNotEmpty(escapechar)) {
			cw.setEscapechar(escapechar.charAt(0));
		}
		
		try {
			ArrayList<Object> line = new ArrayList<Object>();
			for (Iterator it = MakeIterator.convert(columns); it.hasNext();) {
				Object c = it.next();
				Object hidden = PropertyUtils.getProperty(c, "hidden");
				if (!Boolean.TRUE.equals(hidden)) {
					String name = (String)PropertyUtils.getProperty(c, "name");
					String header = (String)PropertyUtils.getProperty(c, "header");
					if (header == null) {
						header = name;
					}
					line.add(header);
				}
			}
			cw.writeNext(line);

			if (list != null) {
				for (Iterator it = MakeIterator.convert(list); it.hasNext();) {
					Object d = it.next();
					line.clear();
					for (Iterator it2 = MakeIterator.convert(columns); it2.hasNext();) {
						Object c = it2.next();
						Object hidden = PropertyUtils.getProperty(c, "hidden");
						if (Boolean.TRUE.equals(hidden)) {
							continue;
						}

						String name = (String)PropertyUtils.getProperty(c, "name");
						
						String format = null;

						Object oFormat = PropertyUtils.getProperty(c, "format");
						if (oFormat != null) {
							format = (String)PropertyUtils.getProperty(oFormat, "type");
						}
						
						Object value = null;
						if ("code".equals(format)) {
							Object codemap = PropertyUtils.getProperty(oFormat, "codemap");
							if (codemap instanceof String) {
								codemap = stack.findValue((String)codemap, Map.class);
							}
							if (codemap instanceof Map) {
								value = ((Map)codemap).get(PropertyUtils.getProperty(d, name)); 
							}
						}
						else if ("expression".equals(format)) {
							try {
								stack.push(d);
								Object exp = PropertyUtils.getProperty(oFormat, "expression");
								if (exp instanceof String) {
									value = stack.findValue((String)exp);
								}
							}
							finally {
								stack.pop();
							}
						}
						else {
							StringWriter sw = new StringWriter();
							try {
								stack.push(d);
								Property p = new Property(stack);
								p.setValue(name);
								p.setFormat(format);
								p.setEscape(null);
								p.start(sw);
								p.end(sw, "");
							}
							finally {
								stack.pop();
							}
							value = sw.toString();
						}
						line.add(value);
					}
					cw.writeNext(line);
				}
			}
			cw.flush();
		}
		catch (Exception e) {
			log.error("csv", e);
			try {
				writer.write(ExceptionUtils.getFullStackTrace(e));
			}
			catch (IOException e2) {
			}
		}
		finally {
		}

		return super.end(writer, "");
	}

	/**
	 * Evaluate parameters
	 */
	protected void evaluateParams() {
		Object value;

		// list
		if (list == null) {
			list = parameters.get("list");
		}

		value = null;
		if (list instanceof String) {
			value = findValue((String) list);
			if (!MakeIterator.isIterable(value)) {
				value = null;
			}
		}
		else if (MakeIterator.isIterable(list)) {
			value = list;
		}
		list = value;


		// columns
		if (columns == null) {
			columns = parameters.get("columns");
		}

		value = null;
		if (columns instanceof String) {
			value = findValue((String)columns);
		}
		else if (MakeIterator.isIterable(columns)) {
			value = columns;
		}

        if (value == null) {
			fieldError("columns", "The requested columns '" + columns + "' could not be resolved as a collection/array/map/enumeration/iterator type.", null);
		}
		else {
			addParameter("columns", value);
		}


		if (separator != null) {
			separator = findString(separator);
		}
		if (quotechar != null) {
			quotechar = findString(quotechar);
		}
		if (escapechar != null) {
			escapechar = findString(escapechar);
		}
	}

	/**
	 * @param list the list to set
	 */
	@StrutsTagAttribute(description = "the property for the list")
	public void setList(Object list) {
		this.list = list;
	}

	/**
	 * @param columns the columns to set
	 */
	@StrutsTagAttribute(description = "the property for the columns")
	public void setColumns(Object columns) {
		this.columns = columns;
	}

	/**
	 * @param separator the separator to set
	 */
	@StrutsTagAttribute(description = "the property for separator")
	public void setSeparator(String separator) {
		this.separator = separator;
	}

	/**
	 * @param quotechar the quotechar to set
	 */
	@StrutsTagAttribute(description = "the property for quotechar")
	public void setQuotechar(String quotechar) {
		this.quotechar = quotechar;
	}

	/**
	 * @param escapechar the escapechar to set
	 */
	@StrutsTagAttribute(description = "the property for escapechar")
	public void setEscapechar(String escapechar) {
		this.escapechar = escapechar;
	}

}
