/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.struts2.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import nuts.core.collections.CollectionUtils;
import nuts.core.lang.ObjectUtils;
import nuts.core.lang.StringUtils;
import nuts.core.orm.dao.DataAccessException;
import nuts.core.orm.dao.ModelDAO;
import nuts.core.orm.dao.ModelMetaData;
import nuts.core.orm.dao.QueryParameter;
import nuts.core.orm.restriction.Orders;
import nuts.core.orm.restriction.Restrictions;
import nuts.core.servlet.URLHelper;
import nuts.core.util.Filter;
import nuts.core.util.Pager;
import nuts.core.util.Query;
import nuts.core.util.Sorter;
import nuts.ext.struts2.CookieStateProvider;
import nuts.ext.struts2.util.StrutsContextUtils;
import nuts.ext.xwork2.SessionStateProvider;
import nuts.ext.xwork2.StateProvider;
import nuts.ext.xwork2.util.ContextUtils;


/**
 * ModelDrivenAction
 * @param <T> data type
 * @param <E> example type
 */
@SuppressWarnings("serial")
public abstract class ModelDrivenAction<T, E extends QueryParameter> extends
		CommonDataAccessClientAction implements NutsTextConstants {

	/**
	 * DEFAULT_DATA_NAME = "d";
	 */
	public final static String DEFAULT_DATA_FIELD_NAME = "d";

	//------------------------------------------------------------
	// result name
	//------------------------------------------------------------
	/**
	 * RESULT_DEFAULT = "";
	 */
	public final static String RESULT_DEFAULT = "";
	
	/**
	 * RESULT_INPUT = "Input";
	 */
	public final static String RESULT_INPUT = "Input";
	
	/**
	 * RESULT_CONFIRM = "Confirm";
	 */
	public final static String RESULT_CONFIRM = "Confirm";
	
	/**
	 * RESULT_SUCCESS = "Success";
	 */
	public final static String RESULT_SUCCESS = "Success";
	
	/**
	 * STATE_LIST = "list";
	 */
	public final static String STATE_LIST = "list";
	
	//------------------------------------------------------------
	// method suffix
	//------------------------------------------------------------
	protected final static String SUFFIX_INPUT = "Input";
	protected final static String SUFFIX_CONFIRM = "Confirm";
	protected final static String SUFFIX_SELECT = "Select";
	protected final static String SUFFIX_EXECUTE = "Execute";
	
	//------------------------------------------------------------
	// scenario & result
	//------------------------------------------------------------
	protected String actionScenario;
	protected String actionResult;
	protected String methodResult;

	//------------------------------------------------------------
	// parameters
	//------------------------------------------------------------
	protected Pager pager = new Pager();
	protected Sorter sorter = new Sorter();
	protected Query query = new Query();

	protected Boolean _load;
	protected Boolean _save;
	protected Map<String, String> urls = new HashMap<String, String>();

	//------------------------------------------------------------
	// config properties
	//------------------------------------------------------------
	protected String dataFieldName = DEFAULT_DATA_FIELD_NAME;
	protected boolean checkAbortOnError = false;
	protected boolean updateSelective = false;
	protected ModelMetaData<T> modelMetaData;
	protected ModelDAO<T, E> modelDAO;

	//------------------------------------------------------------
	// data properties
	//------------------------------------------------------------
	protected T sourceData;
	protected T data;
	protected List<T> dataList;

	/**
	 * Constructor 
	 */
	public ModelDrivenAction() {
	}

	/**
	 * @param id modle id
	 */
	@SuppressWarnings("unchecked")
	public void setModelId(String id) {
		modelMetaData = getDataAccessClient().getMetaData(id);
		modelDAO = getDataAccessClient().createModelDAO(id);
	}
	
	//------------------------------------------------------------
	// dao methods
	//------------------------------------------------------------
	/**
	 * start transaction
	 * @throws DataAccessException if a data access error occurs
	 */
	protected void daoStartTransaction() throws DataAccessException {
		getDataAccessClient().startTransaction();
	}
	
	/**
	 * commit transaction
	 * @throws DataAccessException if a data access error occurs
	 */
	protected void daoCommitTransaction() throws DataAccessException {
		getDataAccessClient().commitTransaction();
	}

	/**
	 * end transaction
	 * @throws DataAccessException if a data access error occurs
	 */
	protected void daoEndTransaction() throws DataAccessException {
		getDataAccessClient().endTransaction();
	}
	
	/**
	 * create example
	 * @return example instance
	 */
	protected E daoCreateExample() {
		return modelDAO.createExample();
	}
	
	/**
	 * daoCountByExample
	 * 
	 * @param exp example
	 * @return count
	 * @throws DataAccessException if a data access error occurs
	 */ 
	protected Integer daoCountByExample(E exp) throws DataAccessException {
		return modelDAO.countByExample(exp);
	}

	/**
	 * selectByPrimaryKey
	 * 
	 * @param key primary key
	 * @return data
	 * @throws DataAccessException if a data access error occurs
	 */ 
	protected T daoSelectByPrimaryKey(T key) throws DataAccessException {
		return modelDAO.selectByPrimaryKey(key);
	}

	/**
	 * daoSelectByExample
	 * 
	 * @param exp example
	 * @return data list
	 * @throws Exception if an error occurs
	 */ 
	protected List<T> daoSelectByExample(E exp) throws Exception {
		return modelDAO.selectByExample(exp);
	}

	/**
	 * daoInsert
	 * 
	 * @param data data
	 * @throws DataAccessException if a data access error occurs
	 */ 
	protected void daoInsert(T data) throws DataAccessException {
		modelDAO.insert(data);
	}

	/**
	 * daoDeleteByPrimaryKey
	 * 
	 * @param key T
	 * @return count of deleted records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	protected int daoDeleteByPrimaryKey(T key) throws DataAccessException {
		return modelDAO.deleteByPrimaryKey(key);
	}

	/**
	 * daoDeleteByExample
	 * 
	 * @param exp example
	 * @return count of deleted records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	protected int daoDeleteByExample(E exp) throws DataAccessException {
		return modelDAO.deleteByExample(exp);
	}

	/**
	 * daoUpdateByPrimaryKey
	 * 
	 * @param data T
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	protected int daoUpdateByPrimaryKey(T data) throws DataAccessException {
		return modelDAO.updateByPrimaryKey(data);
	}

	/**
	 * daoUpdateByPrimaryKeySelective (ignore null properties)
	 * 
	 * @param data T
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	protected int daoUpdateByPrimaryKeySelective(T data) throws DataAccessException {
		return modelDAO.updateByPrimaryKeySelective(data);
	}

	/**
	 * daoUpdateByExample
	 * 
	 * @param data T
	 * @param exp example
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	protected int daoUpdateByExample(T data, E exp) throws DataAccessException {
		return modelDAO.updateByExample(data, exp);
	}

	/**
	 * daoUpdateByExampleSelective (ignore null properties)
	 * 
	 * @param data T
	 * @param exp example
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	protected int daoUpdateByExampleSelective(T data, E exp) throws DataAccessException {
		return modelDAO.updateByExampleSelective(data, exp);
	}

	//------------------------------------------------------------
	// internal methods
	//------------------------------------------------------------
	/**
	 * resolveColumnName
	 * @param name property name
	 * @return column name
	 */
	protected String resolveColumnName(String name) throws Exception {
		return modelMetaData.getColumnName(name);
	}

	/**
	 * resolveColumnAlias
	 * @param name field name
	 * @return column alias
	 */
	protected String resolveColumnAlias(String name) throws Exception {
		return modelMetaData.getColumnAlias(name);
	}

	@SuppressWarnings("unchecked")
	private void removeRedundantParams(Map params) {
		List keys = new ArrayList();
		Set<Entry> s = params.entrySet();
		for (Entry e : s) {
			if (e.getKey() != null) {
				if (e.getKey().toString().startsWith("_") 
						|| StringUtils.isEmpty(e.getValue())) {
					keys.add(e.getKey());
				}
			}
		}
		for (Object key : keys) {
			params.remove(key);
		}
	}

	/**
	 * load list parameters
	 * @return false - send redirect url
	 */
	@SuppressWarnings("unchecked")
	protected boolean loadListParameters() throws Exception {
		StateProvider sp = getStateProvider();
		if (sp instanceof SessionStateProvider) {
			Map<String, Object> pm = (Map<String, Object>)sp.loadState(STATE_LIST);
			if (pm != null) {
				pager = (Pager)pm.get("pager");
				sorter = (Sorter)pm.get("sorter");
				query = (Query)pm.get("query");
			}
		}
		else if (sp instanceof CookieStateProvider) {
			String qs = (String)sp.loadState(STATE_LIST);
			if (StringUtils.isNotBlank(qs)) {
				Map params = URLHelper.parseQueryString(qs);
				removeRedundantParams(params);
				HttpServletRequest request = StrutsContextUtils.getServletRequest();
				HttpServletResponse response = StrutsContextUtils.getServletResponse();
				String url = URLHelper.buildURL(request, params, false);
				response.sendRedirect(url);
				return false;
			}
		}
		return true;
	}
	
	/**
	 * save list url
	 */
	protected void saveListUrl() {
		HttpServletRequest request = StrutsContextUtils.getServletRequest();
		Map<?,?> params = getListParameters();
		urls.put("l", URLHelper.buildURL(request, params, false));
	}
	
	/**
	 * save list parameters
	 * @throws Exception exception
	 */
	protected void saveListParameters() throws Exception {
		StateProvider sp = getStateProvider();
		if (sp instanceof SessionStateProvider) {
			Map<String, Object> pm = new HashMap<String, Object>();
			pm.put("pager", pager);
			pm.put("sorter", sorter);
			pm.put("query", query);
			sp.saveState(STATE_LIST, pm);
		}
		else if (sp instanceof CookieStateProvider) {
			sp.saveState(STATE_LIST, getListParametersString());
		}
	}

	/**
	 * @return list parameters string
	 */
	public String getListParametersString() {
		Map<?,?> params = getListParameters();
		StringBuilder qs = new StringBuilder();
		URLHelper.buildParametersString(params, qs);
		if (qs.length() > 0 && qs.charAt(0) == '?') {
			qs.delete(0, 1);
		}
		return qs.toString();
	}
	
	/**
	 * @return list parameters string
	 */
	@SuppressWarnings("unchecked")
	public Map getListParameters() {
		Map params = new HashMap();
		if (pager.getStart() != null) {
			params.put("p.s", pager.getStart());
		}
		if (pager.getLimit() != null) {
			params.put("p.l", pager.getLimit());
		}
		
		if (StringUtils.isNotBlank(sorter.getColumn())) {
			params.put("s.c", sorter.getColumn());
		}
		if (StringUtils.isNotBlank(sorter.getDirection())) {
			params.put("s.d", sorter.getDirection());
		}
		
		if (StringUtils.isNotBlank(query.getName())) {
			params.put("q.n", query.getName());
		}
		if (StringUtils.isNotBlank(query.getKind())) {
			params.put("q.k", query.getKind());
		}
		
		if (query.getFilters() != null) {
			Iterator<Entry<String, Filter>> i = query.getFilters().entrySet().iterator();
			for ( ; i.hasNext(); ) {
				Entry<String, Filter> e = i.next();
				Filter f = e.getValue();
				if (f != null && f.getValues() != null && !f.getValues().isEmpty()) {
					String prex = "q.fs." + e.getKey() + "."; 
					params.put(prex + "c", f.getComparator());
					params.put(prex + f.getValueType() + "vs", f.getValues());
				}
			}
		}

		return params;
	}

	/**
     * @return "success"
     */
	@Override
    public String getInputResultName() {
    	return SUCCESS;
    }

	/**
	 * getMessage
	 * @param msg msg id
	 * @return message string
	 */
	protected String getMessage(String msg) {
		return getText(msg);
	}
	
	/**
	 * getMessage
	 * @param msg msg id
	 * @param params parameters
	 * @return message string
	 */
	protected String getMessage(String msg, String[] params) {
		return getText(msg, params);
	}

	//------------------------------------------------------------
	// do method
	//------------------------------------------------------------
	/**
	 * doSelectInput 
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doSelectInput() throws Exception {
		setMethodResult(RESULT_INPUT);
		return SUCCESS;
	}

	/**
	 * doSelectExecute 
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doSelectExecute() throws Exception {
		T d = selectData(data);
		if (d != null) {
			data = d;
			setMethodResult(RESULT_SUCCESS);
		}
		else {
			setMethodResult(RESULT_INPUT);
		}
		return SUCCESS;
	}

	/**
	 * doSelectPrint
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doSelectPrint() throws Exception {
		T d = selectData(data);
		if (d != null) {
			data = d;
		}
		setMethodResult(RESULT_DEFAULT);
		return SUCCESS;
	}

	/**
	 * doInsertSelect
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doInsertSelect() throws Exception {
		T d = selectData(data);
		if (d != null) {
			data = d;
			clearIdentityValue(d);
		}
		setMethodResult(RESULT_INPUT);
		return SUCCESS;
	}

	/**
	 * doInsertInput
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doInsertInput() throws Exception {
		clearIdentityValue(data);
		data = prepareDefaultData(data);
		setMethodResult(RESULT_INPUT);
		return SUCCESS;
	}

	/**
	 * doInsertConfirm
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doInsertConfirm() throws Exception {
		if (checkBeforeInsert(data)) {
			addActionConfirm(getMessage(ACTION_CONFIRM_PREFIX + getActionScenario()));
			setMethodResult(RESULT_CONFIRM);
		}
		else {
			setMethodResult(hasErrors() ? RESULT_INPUT : RESULT_CONFIRM);
		}
		return SUCCESS;
	}

	/**
	 * doInsertExecute
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doInsertExecute() throws Exception {
		if (checkBeforeInsert(data)) {
			try {
				daoStartTransaction();
				
				beforeInsert(data);
				insertData(data);
				afterInsert(data);
				
				daoCommitTransaction();
				
				finishInsert(data);
				addActionMessage(getMessage(ACCESS_SUCCESS_PREFIX + getActionScenario()));
				setMethodResult(RESULT_SUCCESS);
			}
			catch (Exception e) {
				log.error(e.getMessage(), e);
				addActionError(getMessage(ACTION_FAILED_PREFIX + getActionScenario(), 
					new String[] { e.getMessage() }));
				setMethodResult(RESULT_INPUT);
			}
			finally {
				daoEndTransaction();
			}
		}
		else {
			setMethodResult(hasErrors() ? RESULT_INPUT : RESULT_CONFIRM);
		}
		return SUCCESS;
	}

	/**
	 * doUpdateInput
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doUpdateInput() throws Exception {
		setMethodResult(RESULT_INPUT);
		return SUCCESS;
	}

	/**
	 * doUpdateSelect
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doUpdateSelect() throws Exception {
		T d = selectData(data);
		if (d != null) {
			data = d;
		}
		setMethodResult(RESULT_INPUT);
		return SUCCESS;
	}

	/**
	 * doUpdateConfirm
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doUpdateConfirm() throws Exception {
		setMethodResult(RESULT_INPUT);

		sourceData = selectData(data);
		if (sourceData != null) {
			if (checkBeforeUpdate(data, sourceData)) {
				addActionConfirm(getMessage(ACTION_CONFIRM_PREFIX + getActionScenario()));
				setMethodResult(RESULT_CONFIRM);
			}
			else {
				setMethodResult(hasErrors() ? RESULT_INPUT : RESULT_CONFIRM);
			}
		}
		return SUCCESS;
	}

	/**
	 * doUpdateExecute
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doUpdateExecute() throws Exception {
		sourceData = selectData(data);
		if (sourceData != null) {
			if (checkBeforeUpdate(data, sourceData)) {
				try {
					daoStartTransaction();
					
					beforeUpdate(data, sourceData);
					updateData(data);
					afterUpdate(data, sourceData);
					
					daoCommitTransaction();
					
					finishUpdate(data, sourceData);
					addActionMessage(getMessage(ACCESS_SUCCESS_PREFIX + getActionScenario()));
					setMethodResult(RESULT_SUCCESS);
				}
				catch (Exception e) {
					log.error(e.getMessage(), e);
					addActionError(getMessage(ACTION_FAILED_PREFIX + getActionScenario(), 
						new String[] { e.getMessage() }));
					setMethodResult(RESULT_INPUT);
				}
				finally {
					daoEndTransaction();
				}
			}
			else {
				setMethodResult(hasErrors() ? RESULT_INPUT : RESULT_CONFIRM);
			}
		}
		return SUCCESS;
	}

	/**
	 * doDeleteConfirm
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doDeleteConfirm() throws Exception {
		setMethodResult(RESULT_INPUT);

		sourceData = selectData(data);
		if (sourceData != null) {
			data = sourceData;
			if (checkBeforeDelete(data, sourceData)) {
				addActionConfirm(getMessage(ACTION_CONFIRM_PREFIX + getActionScenario()));
			}
			setMethodResult(RESULT_CONFIRM);
		}
		return SUCCESS;
	}

	/**
	 * doDeleteExecute
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doDeleteExecute() throws Exception {
		setMethodResult(RESULT_CONFIRM);

		sourceData = selectData(data);
		if (sourceData != null) {
			data = sourceData;
			if (checkBeforeDelete(data, sourceData)) {
				try {
					daoStartTransaction();
					
					beforeDelete(data);
					deleteData(data);
					afterDelete(data);
					
					daoCommitTransaction();
					
					finishDelete(data);
					addActionMessage(getMessage(ACCESS_SUCCESS_PREFIX + getActionScenario()));
					setMethodResult(RESULT_SUCCESS);
				}
				catch (Exception e) {
					log.error(e.getMessage(), e);
					addActionError(getMessage(ACTION_FAILED_PREFIX + getActionScenario(), 
						new String[] { e.getMessage() }));
				}
				finally {
					daoEndTransaction();
				}
			}
			else {
				addActionConfirm(getMessage(ACTION_CONFIRM_PREFIX + getActionScenario()));
			}
		}

		return SUCCESS;
	}

	//------------------------------------------------------------
	// list methods
	//------------------------------------------------------------
	/**
	 * doList
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doList() throws Exception {
		if (Boolean.TRUE.equals(_load)) {
			if (!loadListParameters()) {
				return NONE;
			}
		}

		queryList();
		
		saveListUrl();

		if (Boolean.TRUE.equals(_save)) {
			saveListParameters();
		}
		
		setMethodResult(RESULT_DEFAULT);
		return SUCCESS;
	}

	/**
	 * queryList
	 */
	protected void queryList() throws Exception {
		E exp = daoCreateExample();

		addQueryToExample(exp);
		addOrderToExample(exp);

		queryListByExample(exp);
	}

	/**
	 * queryListByExample
	 * @param exp example
	 */
	protected void queryListByExample(E exp) throws Exception {
		if (pager.getLimit() == null) {
			pager.setLimit(getTextAsInt(LIST_PAGER_LIMIT, 20));
		}

		int maxLimit = getTextAsInt(PAGER_MAX_LIMIT, 100);
		if (pager.getLimit() > maxLimit) {
			pager.setLimit(maxLimit);
		}

		pager.setTotal(daoCountByExample(exp));
		if (pager.getTotal() > 0) {
			pager.normalize();
			exp.setStart(pager.getStart());
			exp.setLimit(pager.getLimit());
			dataList = daoSelectByExample(exp);
		}
		else {
			dataList = new ArrayList<T>();
		}
	}

	/**
	 * addQueryToExample
	 * @param exp E
	 */
	protected void addQueryToExample(E exp) throws Exception {
		Restrictions wc = exp.getRestrictions();

		query.normalize();
		if (query.getFilters() != null) {
			String conjunction = wc.getConjunction();
			if (StringUtils.isNotEmpty(query.getKind())) {
				wc.setConjunction(query.getKind());
			}

			for (Entry<String, Filter> e : query.getFilters().entrySet()) {
				Filter f = e.getValue();
				if (f == null) {
					continue;
				}

				List<?> values = f.getValues();
				if (values == null || values.isEmpty()) {
					continue;
				}

				Object value = values.get(0);
				if (StringUtils.isBlank(value)
						&& !Filter.IN.equals(f.getComparator())
						&& !Filter.BETWEEN.equals(f.getComparator())) {
					continue;
				}

				String name = StringUtils.isEmpty(f.getName()) ? e.getKey() : f.getName();
				String column = resolveColumnName(name);
				if (column == null) {
					continue;
				}

				if (Filter.EQUAL.equals(f.getComparator())) {
					wc.equalTo(column, value);
				}
				else if (Filter.GREATER_THAN.equals(f.getComparator())) {
					wc.greaterThan(column, value);
				}
				else if (Filter.GREATER_EQUAL.equals(f.getComparator())) {
					wc.greaterThanOrEqualTo(column, value);
				}
				else if (Filter.LESS_THAN.equals(f.getComparator())) {
					wc.lessThan(column, value);
				}
				else if (Filter.LESS_EQUAL.equals(f.getComparator())) {
					wc.lessThanOrEqualTo(column, value);
				}
				else if (Filter.LIKE.equals(f.getComparator())) {
					wc.like(column, value);
				}
				else if (Filter.MATCH.equals(f.getComparator())) {
					wc.match(column, value);
				}
				else if (Filter.LEFT_MATCH.equals(f.getComparator())) {
					wc.leftMatch(column, value);
				}
				else if (Filter.RIGHT_MATCH.equals(f.getComparator())) {
					wc.rightMatch(column, value);
				}
				else if (Filter.IN.equals(f.getComparator())) {
					wc.in(column, values);
				}
				else if (Filter.BETWEEN.equals(f.getComparator())) {
					Object v1 = values.get(0);
					Object v2 = values.size() > 1 ? values.get(1) : null;

					if (StringUtils.isBlank(v1) && StringUtils.isBlank(v2)) {
					}
					else if (StringUtils.isBlank(v1)) {
						wc.lessThanOrEqualTo(column, v2);
					}
					else if (StringUtils.isBlank(v2)) {
						wc.greaterThanOrEqualTo(column, v1);
					}
					else {
						wc.between(column, v1, v2);
					}
				}
			}
			wc.setConjunction(conjunction);
		}
	}

	/**
	 * addOrderToExample
	 * @param exp E
	 */
	protected void addOrderToExample(E exp) throws Exception {
		if (StringUtils.isBlank(sorter.getColumn())) {
			sorter.setColumn(getText(LIST_SORTER_COLUMN, ""));
		}
		if (StringUtils.isBlank(sorter.getDirection())) {
			sorter.setDirection(getText(LIST_SORTER_DIRECTION, ""));
		}
		if (!StringUtils.isBlank(sorter.getColumn())) {
			if (StringUtils.isBlank(sorter.getDirection())) {
				sorter.setDirection(Orders.ASC);
			}
			exp.getOrders().addOrder(resolveColumnAlias(sorter.getColumn()), sorter.getDirection());
		}
	}

	//------------------------------------------------------------
	// select methods
	//------------------------------------------------------------
	/**
	 * selectData
	 * @param data data
	 * @return data data found
	 * @throws Exception if an error occurs
	 */
	protected T selectData(T data) throws Exception {
		T d = daoSelectByPrimaryKey(data);
		if (d == null) {
			addActionError(getMessage(ERROR_DATA_NOTFOUND));
		}
		return d;
	}

	/**
	 * beforeCheck
	 * @param data data
	 * @param srcData source data (null on insert)
	 * @return true if do something success
	 * @throws Exception if an error occurs
	 */
	protected boolean beforeCheck(T data, T srcData) throws Exception {
		return true;
	}

	//------------------------------------------------------------
	// insert methods
	//------------------------------------------------------------
	/**
	 * prepareDefaultData
	 * @param data data
	 * @return data
	 */
	protected T prepareDefaultData(T data) throws Exception {
		return data;
	}
	
	/**
	 * insert data
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	protected void insertData(T data) throws Exception {
		daoInsert(data);
	}

	/**
	 * checkBeforeInsert
	 * @param data data
	 * @return true if check success
	 * @throws Exception if an error occurs
	 */
	protected boolean checkBeforeInsert(T data) throws Exception {
		boolean c = true;

		if (!beforeCheck(data, null)) {
			return false;
		}
		
		if (!checkPrimaryKeyOnInsert(data)) {
			c = false;
			if (checkAbortOnError) {
				return false;
			}
		}
		if (!checkUniqueKeyOnInsert(data)) {
			c = false;
			if (checkAbortOnError) {
				return false;
			}
		}
		if (!checkForeignKey(data)) {
			c = false;
			if (checkAbortOnError) {
				return false;
			}
		}
		return c;
	}

	/**
	 * beforeInsert
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	protected void beforeInsert(T data) throws Exception {
	}

	/**
	 * afterInsert
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	protected void afterInsert(T data) throws Exception {
	}

	/**
	 * finishInsert
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	protected void finishInsert(T data) throws Exception {
	}

	//------------------------------------------------------------
	// update methods
	//------------------------------------------------------------
	/**
	 * update data
	 * @return update count
	 * @throws Exception if an error occurs
	 */
	protected Integer updateData(T data) throws Exception {
		int cnt;
		if (updateSelective) {
			cnt = daoUpdateByPrimaryKeySelective(data);
		}
		else {
			cnt = daoUpdateByPrimaryKey(data);
		}
		if (cnt != 1) {
			throw new RuntimeException("The update data count (" + cnt + ") does not equals 1.");
		}
		return cnt;
	}

	/**
	 * checkBeforeUpdate
	 * @param data data
	 * @param srcData srcData
	 * @return true if check success
	 * @throws Exception if an error occurs
	 */
	protected boolean checkBeforeUpdate(T data, T srcData) throws Exception {
		boolean c = true;

		if (!beforeCheck(data, srcData)) {
			return false;
		}
		
		if (!checkUpdatedOnUpdate(data, srcData)) {
			c = false;
			if (checkAbortOnError) {
				return false;
			}
		}
		// primary key can not be modified or null
		if (!checkPrimaryKeyOnUpdate(data, srcData)) {
			c = false;
			if (checkAbortOnError) {
				return false;
			}
		}
		if (!checkUniqueKeyOnUpdate(data, srcData)) {
			c = false;
			if (checkAbortOnError) {
				return false;
			}
		}
		if (!checkForeignKey(data)) {
			c = false;
			if (checkAbortOnError) {
				return false;
			}
		}
		return c;
	}

	/**
	 * beforeUpdate
	 * @param data data
	 * @param srcData srcData
	 * @throws Exception if an error occurs
	 */
	protected void beforeUpdate(T data, T srcData) throws Exception {
	}

	/**
	 * afterUpdate
	 * @param data data
	 * @param srcData srcData
	 * @throws Exception if an error occurs
	 */
	protected void afterUpdate(T data, T srcData) throws Exception {
	}

	/**
	 * finishUpdate
	 * @param data data
	 * @param srcData srcData
	 * @throws Exception if an error occurs
	 */
	protected void finishUpdate(T data, T srcData) throws Exception {
	}

	//------------------------------------------------------------
	// delete methods
	//------------------------------------------------------------
	/**
	 * checkBeforeDelete
	 * @param data data
	 * @param srcData srcData
	 * @return true if check success
	 * @throws Exception if an error occurs
	 */
	protected boolean checkBeforeDelete(T data, T srcData) throws Exception {
		if (!beforeCheck(data, srcData)) {
			return false;
		}
		
		if (!checkUpdatedOnDelete(data, srcData)) {
			return false;
		}

		return true;
	}

	/**
	 * beforeDelete
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	protected void beforeDelete(T data) throws Exception {
	}

	/**
	 * afterDelete
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	protected void afterDelete(T data) throws Exception {
	}

	/**
	 * finishDelete
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	protected void finishDelete(T data) throws Exception {
	}

	/**
	 * delete data
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	protected void deleteData(T data) throws Exception {
		int cnt = daoDeleteByPrimaryKey(data);
		if (cnt != 1) {
			throw new RuntimeException("The deleted data count (" + cnt + ") does not equals 1.");
		}
	}

	//------------------------------------------------------------
	// batch methods
	//------------------------------------------------------------
	/**
	 * addKeyListToExample
	 * @param exp example
	 * @param dataList data list
	 * @return count
	 */
	protected int addKeyListToExample(E exp, List<T> dataList, boolean raiseError) throws Exception {
		String[] keys = modelMetaData.getPrimaryKeys();
		String[] cns = modelMetaData.getPrimaryKeyColumnNames();

		if (keys.length == 1) {
			List<Object> vs = new ArrayList<Object>();
			for (int n = 0; n < dataList.size(); n++) {
				T d = dataList.get(n);
				Object v = modelMetaData.getPropertyValue(d, keys[0]);
				if (v != null) {
					vs.add(v);
				}
				else {
					if (raiseError) {
						throw new RuntimeException("The item[" + n + "] has empty primary key value. (" + d + ")");
					}
				}
			}

			Restrictions wc = exp.getRestrictions();
			wc.in(cns[0], vs);

			return vs.size();
		}
		else if (keys.length > 1) {
			Restrictions wc = exp.getRestrictions();
			wc.setConjunctionToOr();
			
			int count = 0;
			for (int n = 0; n < dataList.size(); n++) {
				T d = dataList.get(n);
				Object[] vs = new Object[keys.length]; 
				for (int i = 0; i < keys.length; i++) {
					Object v = modelMetaData.getPropertyValue(d, keys[i]);
					if (v != null) {
						vs[i] = v;
					}
					else {
						if (raiseError) {
							throw new RuntimeException("The item[" + n + "] has empty primary key value. (" + d + ")");
						}
						vs = null;
						break;
					}
				}
				if (vs != null) {
					wc.open();
					wc.setConjunctionToAnd();
					for (int i = 0; i < cns.length; i++) {
						wc.equalTo(cns[i], vs[i]);
					}
					wc.close();
					wc.setConjunctionToOr();
					count++;
				}
			}
			return count;
		}
		else {
			return 0;
		}
	}

	/**
	 * selectDataList
	 * @param dataList dataList
	 * @return dataList
	 */
	protected List<T> selectDataList(List<T> dataList) throws Exception {
		CollectionUtils.removeNull(dataList);
		if (dataList != null && dataList.size() > 0) {
			E exp = daoCreateExample();

			int count = addKeyListToExample(exp, dataList, false);
			if (count > 0) {
				dataList = daoSelectByExample(exp);
			}
			else {
				dataList = null;
			}
		}
		if (dataList == null || dataList.size() < 1) {
			addActionError(getMessage(ERROR_DATA_LIST_EMPTY));
		}
		return dataList;
	}
	
	//------------------------------------------------------------
	// batch methods
	//------------------------------------------------------------
	/**
	 * doBatchUpdateConfirm
	 * @param bid batch id
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doBatchUpdateConfirm(String bid) throws Exception {
		dataList = selectDataList(dataList);
		if (CollectionUtils.isNotEmpty(dataList)) {
			if (checkBeforeBatchUpdate(dataList, bid)) {
				addActionConfirm(getMessage(ACTION_CONFIRM_PREFIX + getActionScenario(), 
						new String[] { String.valueOf(dataList.size()) }));
			}
		}
		setMethodResult(RESULT_CONFIRM);
		return SUCCESS;
	}

	/**
	 * doBatchUpdateExecute
	 * @param bid batch id
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doBatchUpdateExecute(String bid) throws Exception {
		dataList = selectDataList(dataList);

		if (CollectionUtils.isNotEmpty(dataList) && checkBeforeBatchUpdate(dataList, bid)) {
			try {
				daoStartTransaction();
				
				beforeBatchUpdate(dataList, bid);
				T sample = getBatchUpdateSample(dataList, bid);
				int count = updateDataList(dataList, sample);
				afterBatchUpdate(dataList, bid);
				
				daoCommitTransaction();
				
				finishBatchUpdate(dataList, bid);
				
				if (count == dataList.size()) {
					addActionMessage(getMessage(ACCESS_SUCCESS_PREFIX + getActionScenario(), 
							new String[] { String.valueOf(count) }));
				}
				else {
					addActionWarning(getMessage(ACCESS_SUCCESS_PREFIX + getActionScenario(), 
							new String[] { String.valueOf(count) }));
				}
				setMethodResult(RESULT_SUCCESS);
			}
			catch (Exception e) {
				log.error(e.getMessage(), e);
				addActionError(getMessage(ACTION_FAILED_PREFIX + getActionScenario(), 
					new String[] { e.getMessage() }));
				setMethodResult(RESULT_CONFIRM);
			}
			finally {
				daoEndTransaction();
			}
		}
		else {
			setMethodResult(RESULT_CONFIRM);
		}
		return SUCCESS;
	}

	/**
	 * beforeBatchUpdate
	 * @param dataList data list
	 * @param bid batch id
	 * @throws Exception if an error occurs
	 */
	protected void beforeBatchUpdate(List<T> dataList, String bid) throws Exception {
	}
	
	/**
	 * afterBatchUpdate
	 * @param dataList data list
	 * @param bid batch id
	 * @throws Exception if an error occurs
	 */
	protected void afterBatchUpdate(List<T> dataList, String bid) throws Exception {
	}
	
	/**
	 * finishBatchUpdate
	 * @param dataList data list
	 * @param bid batch id
	 * @throws Exception if an error occurs
	 */
	protected void finishBatchUpdate(List<T> dataList, String bid) throws Exception {
	}

	/**
	 * checkBeforeBatchUpdate
	 * @param dataList data list
	 * @param bid batch id
	 * @return true to continue update
	 * @throws Exception if an error occurs
	 */
	protected boolean checkBeforeBatchUpdate(List<T> dataList, String bid) throws Exception {
		return true;
	}
	
	/**
	 * getBatchUpdateSample
	 * @param dataList data list
	 * @param bid batch id 
	 * @return sample data
	 * @throws Exception if an error occurs
	 */
	protected T getBatchUpdateSample(List<T> dataList, String bid) throws Exception {
		return null;
	}
	
	/**
	 * update data list
	 * @param dataList data list
	 * @param sample sample data
	 * @return updated count
	 * @throws Exception if an error occurs
	 */
	protected Integer updateDataList(List<T> dataList, T sample) throws Exception {
		int cnt = 0;
		if (dataList != null && dataList.size() > 0) {
			E exp = daoCreateExample();

			addKeyListToExample(exp, dataList, true);
			cnt = daoUpdateByExampleSelective(sample, exp);
//			if (cnt != dataList.size()) {
//				throw new RuntimeException("The updated data count (" + cnt + ") does not equals dataList.size(" + dataList.size() + ").");
//			}
		}
		return 0;
	}

	//------------------------------------------------------------
	// batch delete methods
	//------------------------------------------------------------
	/**
	 * doBatchDeleteConfirm
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doBatchDeleteConfirm() throws Exception {
		dataList = selectDataList(dataList);
		if (CollectionUtils.isNotEmpty(dataList)) {
			if (checkBeforeBatchDelete(dataList)) {
				addActionConfirm(getMessage(ACTION_CONFIRM_PREFIX + getActionScenario(), 
					new String[] { String.valueOf(dataList.size()) }));
			}
		}
		setMethodResult(RESULT_CONFIRM);
		return SUCCESS;
	}

	/**
	 * doBatchDeleteExecute
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	protected String doBatchDeleteExecute() throws Exception {
		setMethodResult(RESULT_CONFIRM);

		dataList = selectDataList(dataList);

		if (CollectionUtils.isNotEmpty(dataList) && checkBeforeBatchDelete(dataList)) {
			try {
				daoStartTransaction();
				
				beforeBatchDelete(dataList);
				int count = deleteDataList(dataList);
				afterBatchDelete(dataList);
				
				daoCommitTransaction();
				
				finishBatchDelete(dataList);
				
				if (count == dataList.size()) {
					addActionMessage(getMessage(ACCESS_SUCCESS_PREFIX + getActionScenario(), 
							new String[] { String.valueOf(count) }));
				}
				else {
					addActionWarning(getMessage(ACCESS_SUCCESS_PREFIX + getActionScenario(), 
							new String[] { String.valueOf(count) }));
				}
				setMethodResult(RESULT_SUCCESS);
			}
			catch (Exception e) {
				log.error(e.getMessage(), e);
				addActionError(getMessage(ACTION_FAILED_PREFIX + getActionScenario(), 
					new String[] { e.getMessage() }));
			}
			finally {
				daoEndTransaction();
			}
		}
		return SUCCESS;
	}

	/**
	 * beforeBatchDelete
	 * @param dataList data list
	 * @throws Exception if an error occurs
	 */
	protected void beforeBatchDelete(List<T> dataList) throws Exception {
	}
	
	/**
	 * afterBatchDelete
	 * @param dataList data list
	 * @throws Exception if an error occurs
	 */
	protected void afterBatchDelete(List<T> dataList) throws Exception {
	}
	
	/**
	 * finishBatchDelete
	 * @param dataList data list
	 * @throws Exception if an error occurs
	 */
	protected void finishBatchDelete(List<T> dataList) throws Exception {
	}
	
	/**
	 * checkBeforeBatchDelete
	 * @param dataList data list
	 * @return true to continue delete
	 * @throws Exception if an error occurs
	 */
	protected boolean checkBeforeBatchDelete(List<T> dataList) throws Exception {
		return true;
	}
	
	/**
	 * delete data list
	 * @param dataList data list
	 * @return deleted count
	 * @throws Exception if an error occurs
	 */
	protected Integer deleteDataList(List<T> dataList) throws Exception {
		int cnt = 0;
		if (dataList != null && dataList.size() > 0) {
			E exp = daoCreateExample();

			addKeyListToExample(exp, dataList, true);
			cnt = daoDeleteByExample(exp);
//			if (cnt != dataList.size()) {
//				throw new RuntimeException("The deleted data count (" + cnt + ") does not equals dataList.size(" + dataList.size() + ").");
//			}
		}
		return cnt;
	}

	//------------------------------------------------------------
	// check methods
	//------------------------------------------------------------
	/**
	 * checkPrimaryKeyOnInsert
	 * @param data data
	 * @return true if check successfully
	 * @throws Exception if an error occurs
	 */
	protected boolean checkPrimaryKeyOnInsert(T data) throws Exception {
		String[] pks = modelMetaData.getPrimaryKeys();

		boolean hasNull = false;

		E exp = daoCreateExample();
		Restrictions wc = exp.getRestrictions();
		wc.setConjunctionToAnd();
		for (String pk : pks) {
			Object dv = modelMetaData.getPropertyValue(data, pk);
			if (dv == null) {
				hasNull = true;
				addFieldError(getModelFieldName(pk), getMessage(ERROR_FIELDVALUE_REQUIRED));
			}
			else {
				wc.equalTo(resolveColumnName(pk), dv);
			}
		}
		if (hasNull) {
			return (StringUtils.isNotEmpty(modelMetaData.getIdentityName()));
		}

		exp.setLimit(1);
		if (daoCountByExample(exp) > 0) {
			for (String pk : pks) {
				addFieldError(getModelFieldName(pk), getMessage(ERROR_FIELDVALUE_DUPLICATE));
			}
			addActionError(getMessage(ERROR_DATA_DUPLICATE));
			return false;
		}
		return true;
	}

	/**
	 * @param data data
	 * @param srcData srcData
	 * @return true if check successfully
	 * @throws Exception if an error occurs
	 */
	protected boolean checkPrimaryKeyOnUpdate(T data, T srcData) throws Exception {
		String[] pks = modelMetaData.getPrimaryKeys();

		boolean hasNull = false;
		for (String pk : pks) {
			Object dv = modelMetaData.getPropertyValue(data, pk);
			if (dv == null) {
				hasNull = true;
				addFieldError(getModelFieldName(pk), getMessage(ERROR_FIELDVALUE_REQUIRED));
			}
		}
		if (hasNull) {
			return false;
		}

		return true;
	}

	/**
	 * checkUniqueKeyOnInsert
	 * @param data data
	 * @return true if check successfully
	 * @throws Exception if an error occurs
	 */
	protected boolean checkUniqueKeyOnInsert(T data) throws Exception {
		String[][] ucs = modelMetaData.getUniqueKeyConstraints();
		for (String[] uc : ucs) {
			boolean allNull = true;

			E exp = daoCreateExample();
			Restrictions wc = exp.getRestrictions();
			wc.setConjunctionToAnd();
			for (String up : uc) {
				Object dv = modelMetaData.getPropertyValue(data, up);
				if (dv == null) {
					wc.isNull(resolveColumnName(up));
				}
				else {
					allNull = false;
					wc.equalTo(resolveColumnName(up), dv);
				}
			}

			if (!allNull) {
				exp.setLimit(1);
				if (daoCountByExample(exp) > 0) {
					for (String up : uc) {
						addFieldError(getModelFieldName(up), getMessage(ERROR_FIELDVALUE_DUPLICATE));
					}
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * checkUniqueKeyOnUpdate
	 * @param data data
	 * @param srcData srcData
	 * @return true if check successfully
	 * @throws Exception if an error occurs
	 */
	protected boolean checkUniqueKeyOnUpdate(T data, T srcData) throws Exception {
		String[][] ucs = modelMetaData.getUniqueKeyConstraints();

		for (String[] uc : ucs) {
			boolean dirty = false;
			for (String up : uc) {
				Object ov = modelMetaData.getPropertyValue(srcData, up);
				Object dv = modelMetaData.getPropertyValue(data, up);
				if (!ObjectUtils.equals(ov, dv)) {
					dirty = true;
					break;
				}
			}

			if (dirty) {
				boolean allNull = true;

				E exp = daoCreateExample();
				Restrictions wc = exp.getRestrictions();
				wc.setConjunctionToAnd();

				for (String p : modelMetaData.getPrimaryKeys()) {
					Object ov = modelMetaData.getPropertyValue(srcData, p);
					wc.notEqualTo(resolveColumnName(p), ov);
				}

				for (String up : uc) {
					Object dv = modelMetaData.getPropertyValue(data, up);
					if (dv == null) {
						wc.isNull(resolveColumnName(up));
					}
					else {
						allNull = false;
						wc.equalTo(resolveColumnName(up), dv);
					}
				}
				if (!allNull) {
					exp.setLimit(1);
					if (daoCountByExample(exp) > 0) {
						for (String up : uc) {
							addFieldError(getModelFieldName(up), getMessage(ERROR_FIELDVALUE_DUPLICATE));
						}
						return false;
					}
				}
			}
		}
		return true;
	}

	/**
	 * checkForeignKey
	 * @param data
	 * @return true if check successfully
	 * @throws Exception if an error occurs
	 */
	@SuppressWarnings("unchecked")
	protected boolean checkForeignKey(T data) throws Exception {
		String[][] fa = modelMetaData.getForeignKeyConstraints();
		for (String[] fs : fa) {
			boolean allNull = true;

			String foreign = fs[0];

			ModelDAO fdao = getDataAccessClient().createModelDAO(foreign);
			QueryParameter qp = (QueryParameter)fdao.createExample();
			Restrictions wc = qp.getRestrictions();
			wc.setConjunctionToAnd();
			for (int i = 1; i < fs.length; i += 2) {
				Object dv = modelMetaData.getPropertyValue(data, fs[i]);
				if (dv == null) {
					wc.isNull(fs[i + 1]);
				}
				else {
					allNull = false;
					wc.equalTo(fs[i + 1], dv);
				}
			}

			if (!allNull) {
				qp.setLimit(1);
				if (fdao.countByExample(qp) < 1) {
					for (int i = 1; i < fs.length; i += 2) {
						addFieldError(getModelFieldName(fs[i]), getMessage(ERROR_FIELDVALUE_INCORRECT));
					}
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * checkUpdatedOnUpdate
	 * @param data data
	 * @param srcData srcData
	 * @return true if check successfully
	 * @throws Exception if an error occurs
	 */
	protected boolean checkUpdatedOnUpdate(T data, T srcData) throws Exception {
		if (!checkUpdated(data, srcData)) {
			addActionConfirm(getMessage(CONFIRM_DATA_OVERWRITE));
			return false;
		}
		return true;
	}

	/**
	 * checkUpdatedOnDelete
	 * @param data data
	 * @param srcData srcData
	 * @return true if check successfully
	 * @throws Exception if an error occurs
	 */
	protected boolean checkUpdatedOnDelete(T data, T srcData) throws Exception {
		if (!checkUpdated(data, srcData)) {
			addActionConfirm(getMessage(ACTION_CONFIRM_PREFIX + getActionScenario()));
			return false;
		}
		return true;
	}

	/**
	 * checkUpdated
	 * @param data data
	 * @param srcData srcData
	 * @return true if check successfully
	 * @throws Exception if an error occurs
	 */
	protected boolean checkUpdated(T data, T srcData) throws Exception {
		return true;
	}

	//------------------------------------------------------------
	// other methods
	//------------------------------------------------------------
	/**
	 * clear identity value of data
	 * @param data data
	 * @throws Exception
	 */
	protected void clearIdentityValue(T data) throws Exception {
		if (data != null && modelMetaData.getIdentityName() != null) {
			modelMetaData.setPropertyValue(data, modelMetaData.getIdentityName(), null);
		}
	}

	/**
	 * @param propertyName property name
	 * @return dataName + "." + propertyName
	 */
	protected String getModelFieldName(String propertyName) {
		return dataFieldName + "." + propertyName;
	}
	
	//------------------------------------------------------------
	// setter & getter
	//------------------------------------------------------------
	/**
	 * @return the pager
	 */
	public Pager getPager() {
		return pager;
	}

	/**
	 * @param pager the pager to set
	 */
	public void setPager(Pager pager) {
		this.pager = pager;
	}

	/**
	 * @return the sorter
	 */
	public Sorter getSorter() {
		return sorter;
	}

	/**
	 * @param sorter the sorter to set
	 */
	public void setSorter(Sorter sorter) {
		this.sorter = sorter;
	}

	/**
	 * @return the query
	 */
	public Query getQuery() {
		return query;
	}

	/**
	 * @param query the query to set
	 */
	public void setQuery(Query query) {
		this.query = query;
	}

	/**
	 * @return the dataList
	 */
	public List<T> getDataList() {
		return dataList;
	}

	/**
	 * @param dataList the dataList to set
	 */
	public void setDataList(List<T> dataList) {
		this.dataList = dataList;
	}

	/**
	 * @return dataName
	 */
	protected String getDataFieldName() {
		return dataFieldName;
	}

	/**
	 * @param dataName the dataName to set
	 */
	protected void setDataFieldName(String dataName) {
		this.dataFieldName = dataName;
	}

	/**
	 * @return the updateSelective
	 */
	protected boolean isUpdateSelective() {
		return updateSelective;
	}

	/**
	 * @param updateSelective the updateSelective to set
	 */
	protected void setUpdateSelective(boolean updateSelective) {
		this.updateSelective = updateSelective;
	}

	/**
	 * @return the checkAbortOnError
	 */
	protected boolean isCheckAbortOnError() {
		return checkAbortOnError;
	}

	/**
	 * @param checkAbortOnError the checkAbortOnError to set
	 */
	protected void setCheckAbortOnError(boolean checkAbortOnError) {
		this.checkAbortOnError = checkAbortOnError;
	}

	/**
	 * @return the modelMetaData
	 */
	protected ModelMetaData<T> getModelMetaData() {
		return modelMetaData;
	}

	/**
	 * @param modelMetaData the modelMetaData to set
	 */
	protected void setModelMetaData(ModelMetaData<T> modelMetaData) {
		this.modelMetaData = modelMetaData;
	}

	/**
	 * @param modelDAO the modelDAO to set
	 */
	protected void setModelDAO(ModelDAO<T, E> modelDAO) {
		this.modelDAO = modelDAO;
	}

	/**
	 * @return the modelDAO
	 */
	protected ModelDAO<T, E> getModelDAO() {
		return modelDAO;
	}

	/**
	 * @return the actionScenario
	 */
	public String getActionScenario() {
		if (actionScenario == null) {
			String actionMethod = ContextUtils.getActionMethod();
			if (actionMethod.endsWith(SUFFIX_SELECT)) {
				actionScenario = actionMethod.substring(0, actionMethod.length()
						- SUFFIX_SELECT.length());
			}
			else if (actionMethod.endsWith(SUFFIX_INPUT)) {
				actionScenario = actionMethod.substring(0, actionMethod.length()
						- SUFFIX_INPUT.length());
			}
			else if (actionMethod.endsWith(SUFFIX_CONFIRM)) {
				actionScenario = actionMethod.substring(0, actionMethod.length()
						- SUFFIX_CONFIRM.length());
			}
			else if (actionMethod.endsWith(SUFFIX_EXECUTE)) {
				actionScenario = actionMethod.substring(0, actionMethod.length()
						- SUFFIX_EXECUTE.length());
			}
			else {
				actionScenario = actionMethod;
			}
		}
		return actionScenario;
	}

	/**
	 * @param actionScenario the actionScenario to set
	 */
	public void setActionScenario(String actionScenario) {
		this.actionScenario = actionScenario;
	}

	/**
	 * @return the action result
	 */
	public String getActionResult() {
		if (actionResult == null) {
			String actionMethod = ContextUtils.getActionMethod();
			if (actionMethod.endsWith(SUFFIX_SELECT)
					|| actionMethod.endsWith(SUFFIX_INPUT)
					|| actionMethod.endsWith(SUFFIX_CONFIRM)) {
				if (methodResult == null) {
					methodResult = RESULT_INPUT;
				}
			}
			else if (actionMethod.endsWith(SUFFIX_EXECUTE)) {
				if (methodResult == null) {
					methodResult = hasErrors() ? RESULT_INPUT : RESULT_CONFIRM;
				}
			}
			else {
				if (methodResult == null) {
					methodResult = RESULT_DEFAULT;
				}
			}
			actionResult = getActionScenario() + methodResult;
		}
		return actionResult;
	}

	/**
	 * @param actionResult the actionResult to set
	 */
	protected void setActionResult(String actionResult) {
		this.actionResult = actionResult;
	}

	/**
	 * @param methodResult the methodResult to set
	 */
	protected void setMethodResult(String methodResult) {
		this.methodResult = methodResult;
	}

	/**
	 * @return the urls
	 */
	public Map<String, String> getUrls() {
		return urls;
	}

	/**
	 * @param urls the urls to set
	 */
	public void setUrls(Map<String, String> urls) {
		this.urls = urls;
	}

	/**
	 * @return the load
	 */
	public Boolean get_load() {
		return _load;
	}

	/**
	 * @param load the load to set
	 */
	public void set_load(Boolean load) {
		this._load = load;
	}

	/**
	 * @return the save
	 */
	public Boolean get_save() {
		return _save;
	}

	/**
	 * @param save the save to set
	 */
	public void set_save(Boolean save) {
		this._save = save;
	}

	//--------------------------------------------------------
	// setter & getter shortcuts
	//--------------------------------------------------------
	/**
	 * @return pager
	 */
	public Pager getP() {
		return getPager();
	}

	/**
	 * @param pager the pager to set
	 */
	public void setP(Pager pager) {
		setPager(pager);
	}

	/**
	 * @return sorter
	 */
	public Sorter getS() {
		return getSorter();
	}

	/**
	 * @param sorter the sorter to set
	 */
	public void setS(Sorter sorter) {
		setSorter(sorter);
	}

	/**
	 * @return query
	 */
	public Query getQ() {
		return getQuery();
	}

	/**
	 * @param query the query to set
	 */
	public void setQ(Query query) {
		setQuery(query);
	}

	/**
	 * @return the urls
	 */
	public Map<String, String> getUs() {
		return urls;
	}

	/**
	 * @param urls the urls to set
	 */
	public void setUs(Map<String, String> urls) {
		this.urls = urls;
	}
}