/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.sitemesh.mapper;

import java.util.Properties;

import javax.servlet.http.HttpServletRequest;

import com.opensymphony.module.sitemesh.Config;
import com.opensymphony.module.sitemesh.Decorator;
import com.opensymphony.module.sitemesh.DecoratorMapper;
import com.opensymphony.module.sitemesh.Page;
import com.opensymphony.module.sitemesh.mapper.AbstractDecoratorMapper;

/**
 * The ParameterDecoratorMapper will map a suitable decorator based on request
 * parameters.
 *
 * <p>The ParameterDecoratorMapper is configured via three properties.</p>
 *
 * <p><code>decorator.parameter</code> - the parameter which contains the name of the decorator which will be mapped.
 * The default is "decorator".</p>
 *
 * <p>For example if <code>decorator.parameter</code> is "foobar" then
 * myurl.jsp?foobar=mydecorator will map to the decorator named "mydecorator".</p>
 *
 * <p>You can also supply an optional 'confirmation parameter'.
 * The decorator will only be mapped if the parameter named <code>parameter.name</code> is
 * in the request URI and the value of that parameter is equal to the
 * <code>parameter.value</code> property.</p>
 *
 * <p>For example assuming parameter.name=confirm and parameter.value=true
 * the URI myurl.jsp?decorator=mydecorator&confirm=true will map the decorator mydecorator.
 * where as the URIs myurl.jsp?decorator=mydecorator and myurl.jsp?decorator=mydecorator&confirm=false will
 * not return any decorator.</p>
 *
 * @author <a href="mailto:mcannon@internet.com">Mike Cannon-Brookes</a>
 * @version $Revision: 1.3 $
 *
 * @see com.opensymphony.module.sitemesh.DecoratorMapper
 */
public class ParameterDecoratorMapper extends AbstractDecoratorMapper {
    private String decoratorParameter;
    private String paramName;
    private String paramValue;

	/**
	 * @see com.opensymphony.module.sitemesh.mapper.AbstractDecoratorMapper#init(com.opensymphony.module.sitemesh.Config, java.util.Properties, com.opensymphony.module.sitemesh.DecoratorMapper)
	 */
   public void init(Config config, Properties properties, DecoratorMapper parent) throws InstantiationException {
        super.init(config, properties, parent);

        decoratorParameter = properties.getProperty("decorator.parameter", "decorator");
        paramName = properties.getProperty("parameter.name", null);
        paramValue = properties.getProperty("parameter.value", null);
    }

	/**
	 * @see com.opensymphony.module.sitemesh.mapper.AbstractDecoratorMapper#getDecorator(javax.servlet.http.HttpServletRequest, com.opensymphony.module.sitemesh.Page)
	 */
   public Decorator getDecorator(HttpServletRequest request, Page page) {
        String decorator = request.getParameter(decoratorParameter);

        if ((paramName == null || paramValue.equals(request.getParameter(paramName)))
            && decorator != null) {
                return getNamedDecorator(request, decorator);
        }
        return super.getDecorator(request, page);
    }
}