/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.ibatis;

import java.util.List;

import nuts.core.orm.dao.AbstractModelDAO;
import nuts.core.orm.dao.DataAccessClient;
import nuts.core.orm.dao.DataAccessException;
import nuts.core.orm.dao.DataHandler;
import nuts.core.sql.criterion.SqlQueryParameter;
import nuts.core.sql.criterion.SqlUpdateParameter;

import com.ibatis.sqlmap.client.SqlMapClient;

/**
 * @param <T> model type
 * @param <E> model example type
 */
@SuppressWarnings("unchecked")
public abstract class IBatisModelDAO<T, E extends SqlQueryParameter> extends AbstractModelDAO<T, E> {
	/**
	 * Constructor
	 *
	 * @param dataAccessClient the dataAccessClient to set
	 */
	public IBatisModelDAO(DataAccessClient dataAccessClient) {
		super(dataAccessClient);
	}

	/**
	 * @return the IBatisDataAccessClient
	 */
	public IBatisDataAccessClient getIBatisDataAccessClient() {
		return ((IBatisDataAccessClient)getDataAccessClient());
	}

	/**
	 * @return the sqlMapClient
	 */
	public SqlMapClient getSqlMapClient() {
		return ((IBatisDataAccessClient)getDataAccessClient()).getSqlMapClient();
	}

	/**
	 * @return full sql id
	 */
	protected abstract String getFullSqlId(String id);

	/**
	 * exists
	 * 
	 * @param key T
	 * @return Album
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public boolean exists(T key) throws DataAccessException {
		Integer count = (Integer)getIBatisDataAccessClient().queryForObject(getFullSqlId(SqlMapConstants.COUNT_BY_PRIMARY_KEY), key);
		return count > 0;
	}
	
	/**
	 * countByExample
	 * 
	 * @param example E
	 * @return count
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int countByExample(E example) throws DataAccessException {
		Integer count = (Integer)getIBatisDataAccessClient().queryForObject(getFullSqlId(SqlMapConstants.COUNT_BY_EXAMPLE), example);
		return count;
	}

	/**
	 * selectByPrimaryKey
	 * 
	 * @param key T
	 * @return T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public T selectByPrimaryKey(T key) throws DataAccessException {
		T data = (T)getIBatisDataAccessClient().queryForObject(getFullSqlId(SqlMapConstants.SELECT_BY_PRIMARY_KEY), key);
		return data;
	}

	/**
	 * selectByExampleWithDataHandler
	 * 
	 * @param example E
	 * @param dataHandler data handler
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public void selectByExampleWithDataHandler(E example, DataHandler<T> dataHandler) throws DataAccessException {
		getIBatisDataAccessClient().queryWithDataHandler(getFullSqlId(SqlMapConstants.SELECT_BY_EXAMPLE), example, dataHandler);
	}

	/**
	 * selectByExample
	 * 
	 * @param example E
	 * @return list of T 
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public List<T> selectByExample(E example) throws DataAccessException {
		List<T> list;
		if (example.getStart() != null || example.getLimit() != null) {
			list = (List<T>) getIBatisDataAccessClient().queryForList(
					getFullSqlId(SqlMapConstants.SELECT_BY_EXAMPLE), example, 
					example.getStart() == null ? 0 : example.getStart(),
					example.getLimit() == null ? 0 : example.getLimit());
		}
		else {
			list = (List<T>) getIBatisDataAccessClient().queryForList(
				getFullSqlId(SqlMapConstants.SELECT_BY_EXAMPLE), example);
		}
		return list;
	}

	/**
	 * if (exists(data)) { updateByPrimaryKey(data); } else { insert(data}; }
	 * 
	 * @param data T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public void save(T data) throws DataAccessException {
		if (exists(data)) { 
			updateByPrimaryKey(data); 
		}
		else {
			insert(data);
		}
	}
	
	/**
	 * insert
	 * 
	 * @param data T
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public void insert(T data) throws DataAccessException {
		getIBatisDataAccessClient().insert(getFullSqlId(SqlMapConstants.INSERT), data);
	}


	/**
	 * deleteByPrimaryKey
	 * 
	 * @param key Album
	 * @return count of deleted records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int deleteByPrimaryKey(T key) throws DataAccessException {
		int rows = getIBatisDataAccessClient().delete(getFullSqlId(SqlMapConstants.DELETE_BY_PRIMARY_KEY), key);
		return rows;
	}

	/**
	 * deleteByExample
	 * 
	 * @param example E
	 * @return count of deleted records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int deleteByExample(E example) throws DataAccessException {
		int rows = getIBatisDataAccessClient().delete(getFullSqlId(SqlMapConstants.DELETE_BY_EXAMPLE), example);
		return rows;
	}

	/**
	 * updateByPrimaryKey
	 * 
	 * @param data T
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int updateByPrimaryKey(T data) throws DataAccessException {
		int rows = getIBatisDataAccessClient().update(getFullSqlId(SqlMapConstants.UPDATE_BY_PRIMARY_KEY), data);
		return rows;
	}

	/**
	 * updateByPrimaryKeySelective (ignore null properties)
	 * 
	 * @param data T
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int updateByPrimaryKeySelective(T data) throws DataAccessException {
		int rows = getIBatisDataAccessClient().update(getFullSqlId(SqlMapConstants.UPDATE_BY_PRIMARY_KEY_SELECTIVE), data);
		return rows;
	}

	/**
	 * updateByExample
	 * 
	 * @param data T
	 * @param example E
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int updateByExample(T data, E example) throws DataAccessException {
		SqlUpdateParameter up = new SqlUpdateParameter(data, example);
		int rows = getIBatisDataAccessClient().update(getFullSqlId(SqlMapConstants.UPDATE_BY_EXAMPLE), up);
		return rows;
	}

	/**
	 * updateByExampleSelective (ignore null properties)
	 * 
	 * @param data T
	 * @param example E
	 * @return count of updated records
	 * @throws DataAccessException if a data access error occurs
	 */ 
	public int updateByExampleSelective(T data, E example) throws DataAccessException {
		SqlUpdateParameter up = new SqlUpdateParameter(data, example);
		int rows = getIBatisDataAccessClient().update(getFullSqlId(SqlMapConstants.UPDATE_BY_EXAMPLE_SELECTIVE), up);
		return rows;
	}
}
