/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.ibatis;

import java.sql.SQLException;
import java.util.List;

import nuts.core.orm.dao.DataAccessException;
import nuts.core.orm.dao.DataAccessSession;
import nuts.core.orm.dao.DataHandler;

import com.ibatis.sqlmap.client.SqlMapSession;
import com.ibatis.sqlmap.client.event.RowHandler;

/**
 */
public class IBatisDataAccessSession implements DataAccessSession, SqlMapSessionAware {
	private SqlMapSession sqlMapSession;
	
	/**
	 */
	public IBatisDataAccessSession() {
		super();
	}

	/**
	 * @param sqlMapSession the sqlMapSession to set
	 */
	public IBatisDataAccessSession(SqlMapSession sqlMapSession) {
		super();
		this.sqlMapSession = sqlMapSession;
	}

	/**
	 * @return the sqlMapSession
	 */
	public SqlMapSession getSqlMapSession() {
		return sqlMapSession;
	}

	/**
	 * @param sqlMapSession the sqlMapSession to set
	 */
	public void setSqlMapSession(SqlMapSession sqlMapSession) {
		this.sqlMapSession = sqlMapSession;
	}

	/**
	 * open session
	 */
	public DataAccessSession openSession() {
		return this;
	}

	/**
	 * open session
	 */
	public DataAccessSession getSession() {
		return this;
	}
	
	/**
	 * close session
	 */
	public void close() {
	}
	
	/**
	 * @return true if session is closed
	 */
	public boolean isClosed() {
		return false;
	}

	/**
	 * start transaction
	 * @throws DataAccessException if a data access error occurs
	 */
	public void startTransaction() throws DataAccessException {
		try {
			sqlMapSession.startTransaction();
		} 
		catch (SQLException e) {
			throw new DataAccessException(e);
		}
	}
	
	/**
	 * commit transaction
	 * @throws DataAccessException if a data access error occurs
	 */
	public void commitTransaction() throws DataAccessException {
		try {
			sqlMapSession.commitTransaction();
		} 
		catch (SQLException e) {
			throw new DataAccessException(e);
		}
	}

	/**
	 * end transaction
	 * @throws DataAccessException if a data access error occurs
	 */
	public void endTransaction() throws DataAccessException {
		try {
			sqlMapSession.endTransaction();
		} 
		catch (SQLException e) {
			throw new DataAccessException(e);
		}
	}
	
	/**
	 * Insert is a bit different from other update methods, as it provides
	 * facilities for returning the primary key of the newly inserted row
	 * (rather than the effected rows). This functionality is of course
	 * optional.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * INSERT values.
	 * 
	 * @param id The name of the statement to execute.
	 * @param parameterObject The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @return The primary key of the newly inserted row. This might be
	 *         automatically generated by the RDBMS, or selected from a sequence
	 *         table or other source.
	 * @throws java.sql.DataAccessException If an error occurs.
	 */
	public Object insert(String id, Object parameterObject) throws DataAccessException {
		try {
			return sqlMapSession.insert(id, parameterObject);
		} 
		catch (SQLException e) {
			throw new DataAccessException(e);
		}
	}

	/**
	 * Update can also be used for any other update statement type, such as
	 * inserts and deletes. Update returns the number of rows effected.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * UPDATE values as well as the WHERE clause parameter(s).
	 * 
	 * @param id The name of the statement to execute.
	 * @param parameterObject The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @return The number of rows effected.
	 * @throws java.sql.DataAccessException If an error occurs.
	 */
	public int update(String id, Object parameterObject) throws DataAccessException {
		try {
			return sqlMapSession.update(id, parameterObject);
		} 
		catch (SQLException e) {
			throw new DataAccessException(e);
		}
	}

	/**
	 * Delete returns the number of rows effected.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the DELETE statement.
	 * 
	 * @param id The name of the statement to execute.
	 * @param parameterObject The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @return The number of rows effected.
	 * @throws java.sql.DataAccessException If an error occurs.
	 */
	public int delete(String id, Object parameterObject) throws DataAccessException {
		try {
			return sqlMapSession.delete(id, parameterObject);
		} 
		catch (SQLException e) {
			throw new DataAccessException(e);
		}
	}

	/**
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the SELECT statement.
	 * 
	 * @param id The name of the statement to execute.
	 * @param parameterObject The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @return The single result object populated with the result set data, or
	 *         null if no result was found
	 * @throws java.sql.DataAccessException If more than one result was found, or if
	 *             any other error occurs.
	 */
	public Object queryForObject(String id, Object parameterObject)
			throws DataAccessException {
		try {
			return sqlMapSession.queryForObject(id, parameterObject);
		} 
		catch (SQLException e) {
			throw new DataAccessException(e);
		}
	}

	/**
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the SELECT statement.
	 * 
	 * @param id The name of the statement to execute.
	 * @param parameterObject The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @return A List of result objects.
	 * @throws java.sql.DataAccessException If an error occurs.
	 */
	public List queryForList(String id, Object parameterObject) throws DataAccessException {
		try {
			return sqlMapSession.queryForList(id, parameterObject);
		} 
		catch (SQLException e) {
			throw new DataAccessException(e);
		}
	}

	/**
	 * Executes a mapped SQL SELECT statement that returns data to populate a
	 * number of result objects within a certain range.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the SELECT statement.
	 * 
	 * @param id The name of the statement to execute.
	 * @param parameterObject The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @param skip The number of results to ignore.
	 * @param max The maximum number of results to return.
	 * @return A List of result objects.
	 * @throws java.sql.DataAccessException If an error occurs.
	 */
	public List queryForList(String id, Object parameterObject, int skip, int max)
			throws DataAccessException {
		try {
			return sqlMapSession.queryForList(id, parameterObject, skip, max);
		} 
		catch (SQLException e) {
			throw new DataAccessException(e);
		}
	}

	/**
	 * This is generally a good approach to take when dealing with large sets of
	 * records (i.e. hundreds, thousands...) that need to be processed without
	 * eating up all of the system resources.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the SELECT statement.
	 * 
	 * @param id The name of the statement to execute.
	 * @param parameterObject The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @param dataHandler A DataHandler instance
	 * @throws java.sql.DataAccessException If an error occurs.
	 */
	public void queryWithDataHandler(String id, Object parameterObject,
			final DataHandler dataHandler) throws DataAccessException {
		try {
			sqlMapSession.queryWithRowHandler(id, parameterObject, new RowHandler() {
				public void handleRow(Object data) {
					dataHandler.handleData(data);
				}
			});
		} 
		catch (SQLException e) {
			throw new DataAccessException(e);
		}
	}
}
