/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.fileupload;

import java.io.IOException;

import nuts.core.images.ImageFactory;
import nuts.core.images.ImageWrapper;
import nuts.core.io.IOUtils;

/**
 * Upload Image Bean Object
 */
@SuppressWarnings("serial")
public class UploadImage extends UploadFile {

	private String format = "jpg";
	
	private int quality = 100;

	private ImageWrapper image;
	
	/**
	 * constructor
	 */
	public UploadImage() {
		setContentType("image");
	}

	/**
	 * @return the format
	 */
	public String getFormat() {
		return format;
	}

	/**
	 * @param format the format to set
	 */
	public void setFormat(String format) {
		this.format = format;
	}

	/**
	 * @return the quality
	 */
	public int getQuality() {
		return quality;
	}

	/**
	 * @param quality the quality to set
	 */
	public void setQuality(int quality) {
		this.quality = quality;
	}

	/**
	 * @return the image
	 * @throws IOException if an I/O error occurs
	 */
	public ImageWrapper getImage() throws Exception {
		if (image == null) {
			byte[] data = getData();
			if (data != null) {
				image = ImageFactory.getInstance().makeImage(data);
			}
		}
		return image;
	}

	/**
	 * resize the image
	 * @param scale scale
	 * @throws Exception if an error occurs
	 */
	public void resize(int scale) throws Exception {
		ImageWrapper img = getImage();
		if (img == null) {
			return;
		}

		img = img.resize(scale);

		data = img.getData();

		if (file != null) {
			IOUtils.write(data, file);
		}
	}

	/**
	 * resize the image
	 * @param width width
	 * @param height height
	 * @throws Exception if an error occurs
	 */
	public void resize(int width, int height) throws Exception {
		ImageWrapper img = getImage();
		if (img == null) {
			return;
		}

		img = img.resize(width, height);

		data = img.getData();

		if (file != null) {
			IOUtils.write(data, file);
		}
	}

	/**
     * @return  a string representation of the object.
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		sb.append("{ ");
		sb.append(super.toString());
		sb.append(", ");
		sb.append("format: ").append(format);
		sb.append(" }");

		return sb.toString();
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = super.hashCode();
		result = prime * result + ((format == null) ? 0 : format.hashCode());
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (!super.equals(obj))
			return false;
		if (getClass() != obj.getClass())
			return false;
		UploadImage other = (UploadImage) obj;
		if (format == null) {
			if (other.format != null)
				return false;
		}
		else if (!format.equals(other.format))
			return false;
		return true;
	}

}
