/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 http://nuts.sourceforge.jp
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.ext.fileupload;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Serializable;
import java.util.Arrays;

import nuts.core.io.IOUtils;

import org.apache.commons.lang.xwork.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.vfs.FileObject;
import org.apache.commons.vfs.FileSystemException;


/**
 */
@SuppressWarnings("serial")
public class UploadFile implements Cloneable, Serializable {

	protected static final Log log = LogFactory.getLog(UploadFile.class);

	/**
	 * constructor
	 */
	public UploadFile() {
	}

	protected FileObject file;
	protected String contentType;
	protected String fileName;
	
	protected byte[] data;

	/**
	 * @return the file
	 */
	public FileObject getFile() {
		return file;
	}

	/**
	 * @param file the file to set
	 */
	public void setFile(FileObject file) {
		this.file = file;
	}
	
	/**
	 * @return the contentType
	 */
	public String getContentType() {
		return contentType;
	}
	
	/**
	 * @param contentType the contentType to set
	 */
	public void setContentType(String contentType) {
		this.contentType = contentType;
	}
	
	/**
	 * @return the fileName
	 */
	public String getFileName() {
		return fileName;
	}

	/**
	 * @param fileName the fileName to set
	 */
	public void setFileName(String fileName) {
		this.fileName = fileName;
	}
	
	/**
	 * @return the saveName
	 */
	public String getSaveName() {
		return file == null ? null : file.getName().getURI();
	}

	/**
	 * @param saveName the saveName to set
	 * @throws IOException 
	 */
	public void setSaveName(String saveName) throws IOException {
		if (StringUtils.isNotEmpty(saveName)) {
			this.file = UploadManager.getFileSystemManager().resolveFile(saveName);
		}
	}

	/**
	 * @return the data
	 * @throws FileNotFoundException if file not found
	 * @throws IOException if an I/O error occurs
	 */
	public byte[] getData() throws FileNotFoundException, IOException {
		if (data == null && file != null) {
			data = IOUtils.toByteArray(file);
		}
		return data;
	}

	/**
	 * @param data the data to set
	 * @throws IOException IOException
	 */
	public void setData(byte[] data) throws IOException {
		this.data = data;
		if (this.file != null) {
			IOUtils.write(this.data, this.file);
		}
	}

	/**
	 * @return the fileSize
	 */
	public long getFileSize() {
		if (data != null) {
			return data.length;
		}
		else if (file != null) {
			try {
				if (file.exists()) {
					return file.getContent().getSize();
				}
			}
			catch (FileSystemException e) {
				log.warn("getFileSize", e);
			}
		}
		return 0;
	}

	/**
	 * create a upload file
	 * @throws IOException IOException
	 */
	public void newFile() throws IOException {
		this.file = UploadManager.createUploadFile(this.fileName);
	}
	
	/**
     * @return  a string representation of the object.
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		sb.append("{ ");
		sb.append("file: ").append(file);
		sb.append(", ");
		sb.append("contentType: ").append(contentType);
		sb.append(", ");
		sb.append("fileName: ").append(fileName);
		sb.append(" }");
		
		return sb.toString();
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((contentType == null) ? 0 : contentType.hashCode());
		result = prime * result + ((file == null) ? 0 : file.hashCode());
		result = prime * result + ((fileName == null) ? 0 : fileName.hashCode());
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		UploadFile other = (UploadFile) obj;
		if (contentType == null) {
			if (other.contentType != null)
				return false;
		}
		else if (!contentType.equals(other.contentType))
			return false;
		if (!Arrays.equals(data, other.data))
			return false;
		if (file == null) {
			if (other.file != null)
				return false;
		}
		else if (!file.equals(other.file))
			return false;
		if (fileName == null) {
			if (other.fileName != null)
				return false;
		}
		else if (!fileName.equals(other.fileName))
			return false;
		return true;
	}

	/**
	 * Clone
	 * @throws CloneNotSupportedException if clone not supported
	 * @return Clone Object
	 */
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

}
