/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.xwork2.validator.validators;

import nuts.core.lang.StringUtils;
import nuts.core.validator.CreditCardNumberValidator;

import com.opensymphony.xwork2.validator.ValidationException;


/**
 * <!-- START SNIPPET: javadoc -->
 * Field Validator that checks if the integer specified is within a certain range.
 * <!-- END SNIPPET: javadoc -->
 *
 *
 * <!-- START SNIPPET: parameters -->
 * <ul>
 * 		<li>fieldName - The field name this validator is validating. Required if using Plain-Validator Syntax otherwise not required</li>
 * 		<li>normalize - remove '-'|' ' characters. (default: false)</li>
 *      </li>
 * </ul>
 * <!-- END SNIPPET: parameters -->
 *
 *
 * <pre>
 * <!-- START SNIPPET: examples -->
 * 		&lt;validators>
 *           &lt;!-- Plain Validator Syntax --&gt;
 *           &lt;validator type="creditcardno">
 *               &lt;param name="fieldName"&gt;name&lt;/param&gt;
 *               &lt;message&gt;value should be credit card number&lt;/message&gt;
 *           &lt;/validator&gt;
 *
 *           &lt;!-- Field Validator Syntax --&gt;
 *           &lt;field name="name"&gt;
 *               &lt;field-validator type="creditcardno"&gt;
 *                   &lt;message&gt;value should be credit card number&lt;/message&gt;
 *               &lt;/field-validator&gt;
 *           &lt;/field&gt;
 *      &lt;/validators&gt;
 * <!-- END SNIPPET: examples -->
 * </pre>
 *
 */
public class CreditCardNumberFieldValidator extends AbstractStringFieldValidator {
	private boolean normalize = false;
	private String cardType;

	/**
	 * @return the normalize
	 */
	public boolean isNormalize() {
		return normalize;
	}

	/**
	 * @param normalize the normalize to set
	 */
	public void setNormalize(boolean normalize) {
		this.normalize = normalize;
	}

	/**
	 * @return the cardType
	 */
	public String getCardType() {
		return cardType;
	}

	/**
	 * @param cardType the cardType to set
	 */
	public void setCardType(String cardType) {
		this.cardType = cardType;
	}

	/**
	 * normalize card number
	 * @param value card number
	 * @return normalized card number
	 */
	protected String normalizeCardNumber(String value) {
		return StringUtils.replaceChars(value, "- ", "");
	}
	
	@Override
	protected void validateString(Object object, String value) throws ValidationException {
		if (normalize) {
			value = normalizeCardNumber(value);
		}
		
		char ct = CreditCardNumberValidator.ALL;
		
		if (StringUtils.isNotEmpty(cardType)) {
			Object v = getFieldValue(cardType, object);
			if (v instanceof Character) {
				ct = (Character)v;
			}
			else if (v instanceof String) {
				String sv = (String)v;
				if (sv.length() > 0) {
					ct = sv.charAt(0);
				}
			}
		}

		CreditCardNumberValidator ccv = new CreditCardNumberValidator(ct);
		if (!ccv.isValid(value)) {
			addFieldError(getFieldName(), object);
		}
	}

}
