/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.xwork2.validator.validators;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import net.sf.json.JSONArray;

import nuts.core.lang.ObjectUtils;
import nuts.core.lang.StringUtils;

import org.apache.struts2.util.MakeIterator;

import com.opensymphony.xwork2.validator.ValidationException;


/**
 * <!-- START SNIPPET: javadoc -->
 * Field Validator that checks if the value specified is in the contants value list.
 * <!-- END SNIPPET: javadoc -->
 *
 *
 * <!-- START SNIPPET: parameters -->
 * <ul>
 * 		<li>fieldName - The field name this validator is validating. Required if using Plain-Validator Syntax otherwise not required</li>
 * 		<li>list - the list of the value. Required. </li>
 * 		<li>ignoreCase - compare without case sensitive. </li>
 * </ul>
 * <!-- END SNIPPET: parameters -->
 *
 *
 * <pre>
 * <!-- START SNIPPET: examples -->
 * 		&lt;validators>
 *           &lt;!-- Plain Validator Syntax --&gt;
 *           &lt;validator type="constant">
 *               &lt;param name="fieldName"&gt;name&lt;/param&gt;
 *               &lt;param name="list"&gt;1,2,3&lt;/param&gt;
 *               &lt;message&gt;value should be in ${array}&lt;/message&gt;
 *           &lt;/validator&gt;
 *
 *           &lt;!-- Field Validator Syntax --&gt;
 *           &lt;field name="name"&gt;
 *               &lt;field-validator type="constant"&gt;
 *                   &lt;param name="list"&gt;"a","b","c"&lt;/param&gt;
 *                   &lt;message&gt;value should be in ${array}&lt;/message&gt;
 *               &lt;/field-validator&gt;
 *           &lt;/field&gt;
 *      &lt;/validators&gt;
 * <!-- END SNIPPET: examples -->
 * </pre>
 *
 */
public class ConstantFieldValidator extends AbstractFieldValidator {

	protected Boolean ignoreCase = false;
	protected String array;
	protected String list;
	protected Object constants;

	/**
	 * @return the array
	 */
	public String getArray() {
		return array;
	}

	/**
	 * @param array the array to set
	 */
	public void setArray(String array) {
		this.array = array;
	}

	/**
	 * @return list
	 */
	public String getList() {
		return list;
	}

	/**
	 * @param list the list to set
	 */
	public void setList(String list) {
		this.list = list;
	}

	/**
	 * @return the ignoreCase
	 */
	public Boolean getIgnoreCase() {
		return ignoreCase;
	}

	/**
	 * @param ignoreCase the ignoreCase to set
	 */
	public void setIgnoreCase(Boolean ignoreCase) {
		this.ignoreCase = ignoreCase;
	}

	/**
	 * @return the contants
	 */
	public String getConstants() {
		if (array != null) {
			return array;
		}
		else if (constants instanceof Map) {
			return '[' + StringUtils.join(((Map)constants).values(), ", ", "'", "'") + ']';
		}
		else if (constants != null) {
			return '[' + StringUtils.join(MakeIterator.convert(constants), ", ", "'", "'") + ']';
		}
		return "[]";
	}

	/**
	 * @see com.opensymphony.xwork2.validator.Validator#validate(java.lang.Object)
	 */
	public void validate(Object object) throws ValidationException {
		Object value = getFieldValue(getFieldName(), object);
		if (ObjectUtils.isEmpty(value)) {
			return;
		}
		
		if (StringUtils.isEmpty(list) && StringUtils.isEmpty(array)) {
			throw new ValidationException("the list of [" + getFieldName() + "] can not be empty string");
		}

		try {
			if (StringUtils.isNotEmpty(list)) {
				constants = getFieldValue(list, null);
			}
			if (constants == null && StringUtils.isNotEmpty(array)) {
				JSONArray ja = JSONArray.fromObject(array);
				if (ja != null) {
					constants = JSONArray.toCollection(ja);
				}
			}
			if (constants == null) {
				throw new NullPointerException();
			}
		}
		catch (ValidationException e) {
			throw e;
		}
		catch (Exception e) {
			StringBuilder sb = new StringBuilder();
			sb.append("the list of [");
			sb.append(getFieldName());
			sb.append("] is invalid: ");
			if (list != null) {
				sb.append("list - ");
				sb.append(list);
				sb.append(" ");
			}
			if (array != null) {
				sb.append("array - ");
				sb.append(array);
				sb.append(" ");
			}
			throw new ValidationException(sb.toString());
		}

		if (value instanceof Collection
				|| value instanceof Map
				|| value.getClass().isArray()) {
			Iterator it = MakeIterator.convert(value);
			while (it.hasNext()) {
				if (!validate(object, it.next(), constants)) {
					break;
				}
			}
		}
		else {
			validate(object, value, constants);
		}
	}

	protected boolean validate(Object object, Object value, Object constants) {
		Iterator iterator = null;
		if (MakeIterator.isIterable(constants)) {
			iterator = MakeIterator.convert(constants);
			while (iterator.hasNext()) {
				Object v = iterator.next();
				if (v instanceof Map.Entry) {
					v = ((Map.Entry)v).getKey();
				}

				if (v != null) {
					if (ignoreCase) {
						if (v.toString().equalsIgnoreCase(value.toString())) {
							return true;
						}
					}
					else {
						if (v.toString().equals(value.toString())) {
							return true;
						}
					}
				}
			}

			addFieldError(getFieldName(), object);
			return false;
		}
		return true;
	}
}
