/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.struts2.components;

import java.io.Writer;

import nuts.core.lang.ExceptionUtils;

import org.apache.commons.logging.LogFactory;
import org.apache.struts2.components.Component;
import org.apache.struts2.views.annotations.StrutsTag;
import org.apache.struts2.views.annotations.StrutsTagAttribute;

import com.opensymphony.xwork2.util.ValueStack;

/**
 * Render a log message.
 * 
 * Example:
 * 
 * <pre>
 * &lt;s:log level=&quot;debug&quot; exception=&quot;exception&quot;/&gt;
 * 
 * </pre>
 */
@StrutsTag(
		name = "log", 
		tldTagClass = "nuts.exts.struts2.views.jsp.LogTag", 
		description = "Render a log message")
public class Log extends Component {

	private static String LINE_SEPERATOR = System.getProperty("line.separator");
	
	protected String category;
	protected String exception;
	protected String level;

	/**
	 * Constructor
	 * 
	 * @param stack value stack
	 */
	public Log(ValueStack stack) {
		super(stack);
	}

	/**
	 * @return true to use body
	 */
	public boolean usesBody() {
		// overriding this to true such that EVAL_BODY_BUFFERED is return and
		// bodyContent will be valid hence, text between start & end tag will
		// be honoured as default message (WW-1268)
		return true;
	}

	private String addMsg(String msg, Object add) {
		if (msg == null) {
			msg = String.valueOf(add);
		}
		else {
			msg += LINE_SEPERATOR + add; 
		}
		return msg;
	}
	
	/**
	 * @see org.apache.struts2.components.Component#end(java.io.Writer, java.lang.String)
	 */
	public boolean end(Writer writer, String body) {
		String msg = body;

		Throwable ex = null;
		if (exception != null) {
			Object val = findValue(exception, Throwable.class);
			if (val instanceof Throwable) {
				ex = (Throwable)val;
				String trace = ExceptionUtils.getStackTrace(ex);
				msg = addMsg(msg, trace);
			}
			else {
				msg = addMsg(msg, val);
			}
		}

		if (msg != null) {
			String logLevel = null;
			if (level != null) {
				logLevel = findString(level);
			}
			
			org.apache.commons.logging.Log log = LogFactory.getLog(
				category == null ? Log.class.getName() : category);

			if ("fatal".equalsIgnoreCase(logLevel)) {
				log.fatal(msg);
			}
			else if ("error".equalsIgnoreCase(logLevel)) {
				log.error(msg);
			}
			else if ("warn".equalsIgnoreCase(logLevel)) {
				log.warn(msg);
			}
			else if ("info".equalsIgnoreCase(logLevel)) {
				log.debug(msg);
			}
			else if ("debug".equalsIgnoreCase(logLevel)) {
				log.debug(msg);
			}
			else if ("trace".equalsIgnoreCase(logLevel)) {
				log.debug(msg);
			}
			else if (ex != null) {
				log.error(msg);
			}
			else {
				log.info(msg);
			}
		}

		return super.end(writer, "");
	}

	/**
	 * @param category the category to set
	 */
	@StrutsTagAttribute(description = "log category")
	public void setCategory(String category) {
		this.category = category;
	}

	/**
	 * @param exception the exception to set
	 */
	@StrutsTagAttribute(description = "Exception property to fetch")
	public void setException(String exception) {
		this.exception = exception;
	}

	/**
	 * @param level the level to set
	 */
	@StrutsTagAttribute(description = "level to log")
	public void setLevel(String level) {
		this.level = level;
	}

}
