/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.tools.poi;

import nuts.core.io.Files;
import nuts.core.io.Streams;
import nuts.core.tool.AbstractFileTool;
import nuts.tools.poi.doc.DocTextReplacer;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;

import org.apache.commons.cli.CommandLine;
import org.apache.poi.hwpf.HWPFDocument;

/**
 * 
 */
public class WordReplacer extends AbstractFileTool {
	public static class Main extends AbstractFileTool.Main {
		/**
		 * @param args arguments
		 */
		public static void main(String[] args) {
			Main m = new Main();
			WordReplacer c = new WordReplacer();
			m.execute(c, args);
		}

		/**
		 * Constructor 
		 */
		public Main() {
			hasTarget = true;
		}

		@Override
		protected void addCommandLineOptions() throws Exception {
			super.addCommandLineOptions();

			addCommandLineOption("x", "suffix", "The suffix of target file name");
		}

		@Override
		protected void getCommandLineOptions(CommandLine cl) throws Exception {
			super.getCommandLineOptions(cl);

			if (cl.hasOption("x")) {
				setParameter("suffix", cl.getOptionValue("x"));
			}
		}
	}

	//---------------------------------------------------------------------------------------
	// properties
	//---------------------------------------------------------------------------------------
	protected String suffix = "";
	
	private int cntSucceed = 0;
	
	/**
	 * Constructor
	 */
	public WordReplacer() {
		includes = new String[] { "**/*.doc" };
	}

	/**
	 * @return the suffix
	 */
	public String getSuffix() {
		return suffix;
	}

	/**
	 * @param suffix the suffix to set
	 */
	public void setSuffix(String suffix) {
		this.suffix = suffix;
	}

	@Override
	protected void afterProcess() throws Exception {
		super.afterProcess();
		
		println0(cntSucceed + " of " + cntFile + " files replaced successfully");
	}

	@Override
	protected void processFile(File fdoc) throws Exception {
		File fxml = new File(Files.stripFileNameExtension(fdoc) + ".xml");
		if (!fxml.exists()) {
			println0("Skip " + fdoc.getPath());
			return;
		}

		println0("Replacing " + fdoc.getPath());
		File fdes = getTargetFile(fdoc, suffix + ".doc");
		FileInputStream fis = null;
		FileOutputStream fos = null;
		HWPFDocument doc = null;

		try {
			fis = new FileInputStream(fdoc);

			doc = new HWPFDocument(fis);

			DocTextReplacer dr = new DocTextReplacer();
			
			dr.replaceFromXml(doc, fxml);

			Streams.safeClose(fis);
			fis = null;
			
			fos = new FileOutputStream(fdes);
			doc.write(fos);
			
			cntSucceed++;
		} 
		finally {
			Streams.safeClose(fis);
			Streams.safeClose(fos);
		}
	}
}
