/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.xwork2.validator.validators;

import nuts.core.lang.Strings;

import com.opensymphony.xwork2.validator.ValidationException;

/**
 * <!-- START SNIPPET: javadoc -->
 * AlphaNumericStringFieldValidator checks that a String field is a alpha numeric string.
 * <!-- END SNIPPET: javadoc -->
 * <p/>
 *
 *
 * <!-- START SNIPPET: parameters -->
 * <ul>
 * 	  <li>fieldName - The field name this validator is validating. Required if using Plain-Validator Syntax otherwise not required</li>
 *    <li>trim - Trim the field value before evaluating its min/max length. ("space" or "blank" or "") Default is "blank"</li>
 * </ul>
 * <!-- END SNIPPET: parameters -->
 *
 *
 * <pre>
 * <!--START SNIPPET: example -->
 *		&lt;validators&gt;
 *          &lt;!-- Plain Validator Syntax --&gt;
 * 			&lt;validator type="alphanumeric string"&gt;
 * 				&lt;param name="fieldName"&gt;name&lt;/param&gt;
 *              &lt;param name="trim"&gt;blank&lt;/param&gt;
 *              &lt;message&gt;name needs to be a alpha numeric string&lt;/message&gt;
 *          &lt;/validator&gt;
 *
 * 			&lt;!-- Field Validator Syntax --&gt;
 * 			&lt;field name="name"&gt;
 *              &lt;param name="trim"&gt;space&lt;/param&gt;
 *              &lt;message&gt;name needs to be a alpha numeric string&lt;/message&gt;
 * 			&lt;/field-name&gt;
 *      &lt;/validators&gt;
 * <!-- END SNIPPET: example -->
 * </pre>
 *
 */
public class AlphaNumericStringFieldValidator extends AbstractStringFieldValidator {

	@Override
	protected void validateString(Object object, String value) throws ValidationException {
		if (!Strings.isAlphanumeric(value)) {
			addFieldError(getFieldName(), object);
		}
	}
}
