/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.struts2.interceptor;

import nuts.core.lang.Strings;
import nuts.core.servlet.HttpServletUtils;
import nuts.core.servlet.UserAgent;
import nuts.exts.struts2.util.StrutsContextUtils;

import java.util.Map;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import org.apache.struts2.ServletActionContext;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.AbstractInterceptor;

/**
 * <!-- START SNIPPET: description -->
 * 
 * An interceptor that handles setting the layout specified in a session as the layout for the
 * current action request. In addition, this interceptor will look for a specific HTTP request
 * parameter and set the layout to whatever value is provided. This means that this interceptor can
 * be used to allow for your application to dynamically change the layout for the user's session.
 * This is very useful for applications that require multi-lingual support and want the user to be
 * able to set his or her language preference at any point. The layout parameter is removed during
 * the execution of this interceptor, ensuring that properties aren't set on an action (such as
 * __layout) that have no typical corresponding setter in your action.
 * 
 * <p/>
 * For example, using the default parameter name, a request to
 * <b>foo.action?__layout=pc</b>, then the layout for PC is saved in the user's
 * session and will be used for all future requests.
 * 
 * <!-- END SNIPPET: description -->
 * 
 * <p/>
 * <u>Interceptor parameters:</u>
 * 
 * <!-- START SNIPPET: parameters -->
 * 
 * <ul>
 * 
 * <li>parameterName (optional) - the name of the HTTP request parameter that dictates the layout to
 * switch to and save in the session. By default this is <b>__layout</b></li>
 * 
 * <li>attributeName (optional) - the name of the session key to store the selected layout. By
 * default this is <b>WW_LAYOUT</b></li>
 * 
 * </ul>
 * 
 * <!-- END SNIPPET: parameters -->
 * 
 * <p/>
 * <u>Extending the interceptor:</u>
 * 
 * <p/>
 * 
 * <!-- START SNIPPET: extending -->
 * 
 * There are no known extensions points for this interceptor.
 * 
 * <!-- END SNIPPET: extending -->
 * 
 * <p/>
 * <u>Example code:</u>
 * 
 * <pre>
 * &lt;!-- START SNIPPET: example --&gt;
 * &lt;action name=&quot;someAction&quot; class=&quot;com.examples.SomeAction&quot;&gt;
 *     &lt;interceptor-ref name=&quot;layout&quot;/&gt;
 *     &lt;interceptor-ref name=&quot;basicStack&quot;/&gt;
 *     &lt;result name=&quot;success&quot;&gt;good_result.ftl&lt;/result&gt;
 * &lt;/action&gt;
 * &lt;!-- END SNIPPET: example --&gt;
 * </pre>
 */
@SuppressWarnings("serial")
public class LayoutInterceptor extends AbstractInterceptor {

	/**
	 * DEFAULT_CONTEXT_ATTRIBUTE = "WW_LAYOUT";
	 */
	public static final String DEFAULT_CONTEXT_ATTRIBUTE = "WW_LAYOUT";

	/**
	 * DEFAULT_SESSION_ATTRIBUTE = "WW_LAYOUT";
	 */
	public static final String DEFAULT_SESSION_ATTRIBUTE = "WW_LAYOUT";

	/**
	 * DEFAULT_PARAMETER = "__layout";
	 */
	public static final String DEFAULT_PARAMETER = "__layout";

	/**
	 * DEFAULT_REQUEST_PARAMETER = "__req_layout";
	 */
    public static final String DEFAULT_REQUEST_PARAMETER = "__req_layout";

    /**
	 * DEFAULT_COOKIE = DEFAULT_SESSION_ATTRIBUTE;
	 */
	public static final String DEFAULT_COOKIE = DEFAULT_SESSION_ATTRIBUTE;

	/**
	 * DEFAULT_COOKIE_MAXAGE = 60 * 60 * 24 * 30; //1M
	 */
	public static final Integer DEFAULT_COOKIE_MAXAGE = 60 * 60 * 24 * 30; //1M

	/**
	 * PC_LAYOUT = "pc";
	 */
	public static final String PC_LAYOUT = "pc";

	/**
	 * MOBILE_LAYOUT = "pc";
	 */
	public static final String MOBILE_LAYOUT = "mb";

	/**
	 * DEFAULT_LAYOUT = "pc";
	 */
	public static final String DEFAULT_LAYOUT = PC_LAYOUT;

	protected String attributeName = DEFAULT_SESSION_ATTRIBUTE;
	protected String parameterName = DEFAULT_PARAMETER;
	protected String requestName = DEFAULT_REQUEST_PARAMETER;
	protected String cookieName = DEFAULT_COOKIE;
	protected String cookieDomain;
	protected String cookiePath;
	protected Integer cookieMaxAge = DEFAULT_COOKIE_MAXAGE;
	protected String defaultLayout = PC_LAYOUT;

	protected boolean saveToSession = false;
	protected boolean saveToCookie = true;

	/**
	 * Constructor
	 */
	public LayoutInterceptor() {
	}

	/**
	 * @param attributeName the attributeName to set
	 */
	public void setAttributeName(String attributeName) {
		this.attributeName = attributeName;
	}

	/**
	 * @param parameterName the parameterName to set
	 */
	public void setParameterName(String parameterName) {
		this.parameterName = parameterName;
	}

	/**
	 * @param cookieName the cookieName to set
	 */
	public void setCookieName(String cookieName) {
		this.cookieName = cookieName;
	}

	/**
	 * @param cookieDomain the cookieDomain to set
	 */
	public void setCookieDomain(String cookieDomain) {
		this.cookieDomain = cookieDomain;
	}

	/**
	 * @param cookiePath the cookiePath to set
	 */
	public void setCookiePath(String cookiePath) {
		this.cookiePath = cookiePath;
	}

	/**
	 * @param cookieMaxAge the cookieMaxAge to set
	 */
	public void setCookieMaxAge(Integer cookieMaxAge) {
		this.cookieMaxAge = cookieMaxAge;
	}

	/**
	 * @param defaultLayout the defaultLayout to set
	 */
	public void setDefaultLayout(String defaultLayout) {
		this.defaultLayout = defaultLayout;
	}

	/**
	 * @param saveToSession the saveToSession to set
	 */
	public void setSaveToSession(boolean saveToSession) {
		this.saveToSession = saveToSession;
	}

	/**
	 * @param saveToCookie the saveToCookie to set
	 */
	public void setSaveToCookie(boolean saveToCookie) {
		this.saveToCookie = saveToCookie;
	}

	/**
	 * Allows the Interceptor to do some processing on the request before and/or after the rest of
	 * the processing of the request by the {@link ActionInvocation} or to short-circuit the
	 * processing and just return a String return code.
	 * 
	 * @param actionInvocation the action invocation
	 * @return the return code, either returned from {@link ActionInvocation#invoke()}, or from the
	 *         interceptor itself.
	 * @throws Exception any system-level error, as defined in
	 *             {@link com.opensymphony.xwork2.Action#execute()}.
	 */
	public String intercept(ActionInvocation actionInvocation) throws Exception {
		Map<String, Object> params = actionInvocation.getInvocationContext().getParameters();
		Map<String, Object> session = actionInvocation.getInvocationContext().getSession();

		String layout = null;

		layout = getLayoutFromParameter(params, parameterName);
		if (Strings.isEmpty(layout)) {
			layout = getLayoutFromParameter(params, requestName);
			if (Strings.isNotEmpty(layout)) {
				saveToSession = false;
				saveToCookie = false;
			}
		}

		if (Strings.isEmpty(layout) && session != null) {
			Object obj = session.get(attributeName);
			if (obj != null) {
				layout = (String)obj;
				saveToSession = false;
				saveToCookie = false;
			}
		}
		
		if (Strings.isEmpty(layout) && Strings.isNotEmpty(cookieName)) {
			layout = getLayoutFromCookie();
			saveToCookie = false;
		}
		
		if (Strings.isEmpty(layout)) {
			layout = getLayoutFromUserAgent();
			saveToCookie = false;
		}
		
		// save layout
		if (Strings.isNotEmpty(layout)) {
			saveLayout(actionInvocation, layout);
			
			if (saveToSession && session != null) {
				session.put(attributeName, layout);
			}
			if (saveToCookie && Strings.isNotEmpty(cookieName)) {
				saveLayoutToCookie(layout);
			}
		}
		
		return actionInvocation.invoke();
	}

	/**
	 * Save the given layout to the ActionInvocation.
	 * 
	 * @param invocation The ActionInvocation.
	 * @param layout The layout to save.
	 */
	protected void saveLayout(ActionInvocation invocation, String layout) {
		invocation.getInvocationContext().put(DEFAULT_CONTEXT_ATTRIBUTE, layout);
	}
	/**
	 * get context layout
	 * @return layout
	 */
	public static String getLayout() {
		String layout = null;

		ActionContext context = StrutsContextUtils.getActionContext();
		if (context != null) {
			layout = (String)context.get(DEFAULT_CONTEXT_ATTRIBUTE);
		}

		if (layout == null) {
			layout = DEFAULT_LAYOUT;
		}
		return layout;
	}

	protected String getLayoutFromCookie() {
		HttpServletRequest request = StrutsContextUtils.getServletRequest();
		String layout = HttpServletUtils.getCookieValue(request, cookieName);
		return layout;
    }

	protected String getLayoutFromParameter(Map<String, Object> params,
			String parameterName) {
		Object layout = params.remove(parameterName);
		if (layout != null && layout.getClass().isArray() && ((Object[])layout).length == 1) {
			layout = ((Object[])layout)[0];
		}

		return layout == null ? null : layout.toString();
	}

	protected String getLayoutFromUserAgent() {
		UserAgent ua = new UserAgent(ServletActionContext.getRequest());
		if (ua.isMobile() || ua.isRobot()) {
			return MOBILE_LAYOUT;
		}
		else {
			return defaultLayout;
		}
	}

    protected void saveLayoutToCookie(String layout) {
		Cookie c = new Cookie(cookieName, layout);
		if (Strings.isNotEmpty(cookieDomain)) {
			c.setDomain(cookieDomain);
		}
		
		if (Strings.isNotEmpty(cookiePath)) {
			c.setPath(cookiePath);
		}
		else {
			c.setPath(ServletActionContext.getRequest().getContextPath());
		}
		
		c.setMaxAge(cookieMaxAge);
		
		ServletActionContext.getResponse().addCookie(c);
    }
}
