/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.struts2.filter;

import nuts.exts.exception.UnknownHandlerException;
import nuts.exts.struts2.operations.NutsExecuteOperations;
import nuts.exts.struts2.operations.NutsPrepareOperations;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.struts2.dispatcher.Dispatcher;
import org.apache.struts2.dispatcher.mapper.ActionMapping;
import org.apache.struts2.dispatcher.ng.InitOperations;
import org.apache.struts2.dispatcher.ng.filter.FilterHostConfig;

/**
 * Executes the discovered request information. 
 * This filter requires the {@link NutsPrepareFilter} to have already
 * been executed in the current chain.
 */
public class NutsExecuteFilter implements Filter {
    protected NutsPrepareOperations prepare;
    protected NutsExecuteOperations execute;

    protected FilterConfig filterConfig;

    public void init(FilterConfig filterConfig) throws ServletException {
        this.filterConfig = filterConfig;
    }

    protected synchronized void lazyInit() {
        if (execute == null) {
            InitOperations init = new InitOperations();
            Dispatcher dispatcher = init.findDispatcherOnThread();
            init.initStaticContentLoader(new FilterHostConfig(filterConfig), dispatcher);

            prepare = new NutsPrepareOperations(filterConfig.getServletContext(), dispatcher);
            execute = new NutsExecuteOperations(filterConfig.getServletContext(), dispatcher);
        }

    }

    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException, ServletException {

        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;

		if (excludeUrl(request)) {
			chain.doFilter(request, response);
			return;
		}

        // This is necessary since we need the dispatcher instance, which was created by the prepare filter
		if (execute == null) {
			lazyInit();
		}

        ActionMapping mapping = prepare.findActionMapping(request, response);

        //if recursion counter is > 1, it means we are in a "forward", in that case a mapping will still be
        //in the request, if we handle it, it will lead to an infinte loop, see WW-3077
		Integer recursionCounter = (Integer)request.getAttribute(NutsPrepareOperations.CLEANUP_RECURSION_COUNTER);

		if (mapping != null && recursionCounter == 1) {
			try {
				execute.executeAction(request, response, mapping);
				return;
			}
			catch (UnknownHandlerException e) {
				//skip;
			}
			catch (Exception e) {
		        // send a http error response to use the servlet defined error handler
		        // make the exception availible to the web.xml defined error page
		        request.setAttribute("javax.servlet.error.exception", e);
	
		        // for compatibility
		        request.setAttribute("javax.servlet.jsp.jspException", e);
				
				response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, e.getMessage());
			}
		}

		boolean handled = execute.executeStaticResourceRequest(request, response);
		if (handled) {
			return;
		}

		chain.doFilter(request, response);
	}

	private boolean excludeUrl(HttpServletRequest request) {
		return request.getAttribute(NutsPrepareFilter.REQUEST_EXCLUDED_FROM_ACTION_MAPPING) != null;
	}

    public void destroy() {
        prepare = null;
        execute = null;
        filterConfig = null;
    }
}
