/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.struts2.dispatcher;

import nuts.core.bind.xml.XmlSerializer;
import nuts.core.log.Log;
import nuts.core.log.Logs;
import nuts.exts.struts2.NutsStrutsConstants;

import java.io.PrintWriter;

import com.opensymphony.xwork2.inject.Inject;

/**
 * <!-- START SNIPPET: description -->
 * 
 * A result that send the content out as plain json text.
 * 
 * <!-- END SNIPPET: description -->
 * 
 * 
 * <!-- START SNIPPET: params -->
 * 
 * <ul>
 * <li>circleStrategy (optional) = circleStrategy to be used. [strict, lenient,
 * noprop] (default: strict)</li>
 * <li>dateFormat (optional) = date format to be used. (default: long -
 * Date.getTime())</li>
 * <li>filterTypes (optional) = filter specified property types when serialize.
 * (default: java.io.*, java.awt.*, [B, [Ljava.lang.Byte;)</li>
 * <li>noCache (optional) = no cache to set. (default: true)</li>
 * <li>contentType (optional) = content type to be used. (default:
 * text/javascript)</li>
 * <li>encoding (optional) = character encoding to be used. This character
 * encoding will be used to set. (default: UTF-8)</li>
 * <li>properties (default) = extra properties to be output. { "user": ... }</li>
 * </ul>
 * 
 * <!-- END SNIPPET: params -->
 * 
 * 
 * <pre>
 * <!-- START SNIPPET: example -->
 * 
 * &lt;action name="displayXmlContent" &gt;
 *   &lt;result type="xml"&gt;username, password, other&lt;/result&gt;
 * &lt;/action&gt;
 * 
 * &lt;action name="displayXmlContent" &gt;
 *   &lt;result type="xml"&gt;
 *   	&lt;param name="properties"&gt;username, password, other&lt;/param&gt;
 *   &lt;/result&gt;
 * &lt;/action&gt;
 * 
 * <!-- END SNIPPET: example -->
 * </pre>
 * 
 */
@SuppressWarnings("serial")
public class XmlResult extends AbstractOMResult {

	protected static final Log log = Logs.getLog(XmlResult.class);

	private String rootName = "result";
	
	/**
	 * Constructor.
	 */
	public XmlResult() {
		super();
		setContentType("text/xml");
	}

	/**
	 * @param shortName the shortName to set
	 */
    @Inject(value=NutsStrutsConstants.NUTS_XML_SHORT_NAME, required=false)
	public void setBeanShortName(String shortName) {
		this.shortName = Boolean.parseBoolean(shortName);
	}

	/**
	 * @return the rootName
	 */
	public String getRootName() {
		return rootName;
	}

	/**
	 * @param rootName the rootName to set
	 */
	public void setRootName(String rootName) {
		this.rootName = rootName;
	}


	/**
	 * write result
	 * @param writer response writer
	 * @param result result object
	 * @throws Exception
	 */
	@Override
	protected void writeResult(PrintWriter writer, Object result) throws Exception {
		if (result != null) {
			XmlSerializer xs = new XmlSerializer();
			setSerializerOptions(xs);

			if (log.isDebugEnabled()) {
				if (xs.isPrettyPrint()) {
					log.debug(xs.serialize(result));
				}
				else {
					xs.setPrettyPrint(true);
					log.debug(xs.serialize(result));
					xs.setPrettyPrint(false);
				}
			}
			xs.serialize(result, writer);
		}
	}
}
