/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.xwork2.validator.validators;

import java.math.BigDecimal;

import com.opensymphony.xwork2.validator.ValidationException;


/**
 * <!-- START SNIPPET: javadoc -->
 * Field Validator that checks if the integer specified is within a certain range.
 * <!-- END SNIPPET: javadoc -->
 * 
 * 
 * <!-- START SNIPPET: parameters -->
 * <ul>
 * 		<li>fieldName - The field name this validator is validating. Required if using Plain-Validator Syntax otherwise not required</li>
 * 		<li>precision - the precision of the decimal value. Required. </li>
 * 		<li>scale - the scale of the decimal value. default is 0.</li>
 * 		<li>type - the type of validate. default is "db".
 *                 java: error on (value.precision &gt; precision || value.scale &gt; scale)
 *                 db: error on ((value.precision - value.scale &gt; precision - scale) || value.scale &gt; scale)
 *      </li>
 * </ul>
 * <!-- END SNIPPET: parameters -->
 * 
 * 
 * <pre>
 * <!-- START SNIPPET: examples -->
 * 		&lt;validators>
 *           &lt;!-- Plain Validator Syntax --&gt;
 *           &lt;validator type="precision">
 *               &lt;param name="fieldName"&gt;price&lt;/param&gt;
 *               &lt;param name="precision"&gt;10&lt;/param&gt;
 *               &lt;param name="scale"&gt;2&lt;/param&gt;
 *               &lt;message&gt;decimal value should be (${precision}, ${scale})&lt;/message&gt;
 *           &lt;/validator&gt;
 *           
 *           &lt;!-- Field Validator Syntax --&gt;
 *           &lt;field name="price"&gt;
 *               &lt;field-validator type="precision"&gt;
 *                   &lt;param name="precision"&gt;10&lt;/param&gt;
 *                   &lt;param name="scale"&gt;2&lt;/param&gt;
 *                   &lt;message&gt;decimal value should be (${precision}, ${scale})&lt;/message&gt;
 *               &lt;/field-validator&gt;
 *           &lt;/field&gt;
 *      &lt;/validators&gt;
 * <!-- END SNIPPET: examples -->
 * </pre>
 * 
 */
public class DecimalPrecisionFieldValidator extends AbstractFieldValidator {

	protected static final String TYPE_DB = "db";
	protected static final String TYPE_JAVA = "java";
	
    private Integer precision = null;
    private Integer scale = 0;
    private String type = "db";
    
	/**
	 * @return precision
	 */
	public Integer getPrecision() {
		return precision;
	}
	/**
	 * @param precision the precision to set
	 */
	public void setPrecision(Integer precision) {
		this.precision = precision;
	}
	/**
	 * @return scale
	 */
	public Integer getScale() {
		return scale;
	}
	/**
	 * @param scale the scale to set
	 */
	public void setScale(Integer scale) {
		this.scale = scale;
	}
	/**
	 * @return type
	 */
	public String getType() {
		return type;
	}
	/**
	 * @param type the type to set
	 */
	public void setType(String type) {
		if (type != null) {
			type = type.toLowerCase();
		}
		this.type = type;
	}
	
	/**
	 * @see com.opensymphony.xwork2.validator.Validator#validate(java.lang.Object)
	 */
	public void validate(Object object) throws ValidationException {
        Object value = this.getFieldValue(getFieldName(), object);

        if (precision == null || precision < 0) {
        	throw new ValidationException("the precision of [" + getFieldName() + "] can not be null or nagative");
        }
        if (scale != null) {
        	if (scale < 0 || scale >= precision) {
        		throw new ValidationException("the scale of [" + getFieldName() + "] can not nagative or >= precision");
        	}
        }

        if (value == null) {
        	return;
        }
        
        if (!(value instanceof BigDecimal)) {
        	throw new ValidationException("filed[" + getFieldName() + "] (" + value.getClass() + ") is not a instance of " + BigDecimal.class);
        }

        BigDecimal bd = (BigDecimal)value;
        
        if (TYPE_JAVA.equals(type)) {
	        if (bd.precision() > precision || bd.scale() > scale) {
	        	addFieldError(getFieldName(), object);
	        }
        }
        else {
	        if (bd.precision() - bd.scale() > precision - scale
	        		|| bd.scale() > scale) {
	        	addFieldError(getFieldName(), object);
	        }
        }
    }
}
