/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.xwork2;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;

import nuts.exts.xwork2.util.ContextUtils;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.opensymphony.xwork2.Action;
import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.LocaleProvider;
import com.opensymphony.xwork2.Preparable;
import com.opensymphony.xwork2.Validateable;
import com.opensymphony.xwork2.interceptor.ValidationWorkflowAware;
import com.opensymphony.xwork2.util.ValueStack;


/**
 * Provides a default implementation for the most common actions.
 * See the documentation for all the interfaces this class implements for more detailed information.
 */
@SuppressWarnings("serial")
public class ActionSupport implements Action, Preparable, Validateable, ValidationWorkflowAware, TextProvider,
		ActionValidationAware, ApplicationValidationAware, SessionValidationAware, PermissionValidator,
		LocaleProvider, StateProvider, Serializable {

	protected static Log log = LogFactory.getLog(ActionSupport.class);

    private transient TextProvider textProvider;
    
    private transient StateProvider stateProvider;

    private final ActionValidationAwareSupport actionValidationAware;

    private final ApplicationValidationAwareSupport applicationValidationAware;

    private final SessionValidationAwareSupport sessionValidationAware;

    /**
	 * Constructor
	 */
	public ActionSupport() {
		textProvider = new TextProviderFactory().createInstance(getClass(), this);
		stateProvider = new SessionStateProvider(getClass().getName() + '.');

		actionValidationAware = new ActionValidationAwareSupport();
		applicationValidationAware = new ApplicationValidationAwareSupport();
		sessionValidationAware = new SessionValidationAwareSupport();
	}

	/**
	 * @return the textProvider
	 */
	public TextProvider getTextProvider() {
		return textProvider;
	}

	/**
	 * @param textProvider the textProvider to set
	 */
	public void setTextProvider(TextProvider textProvider) {
		this.textProvider = textProvider;
	}

	/**
	 * @return the stateProvider
	 */
	public StateProvider getStateProvider() {
		return stateProvider;
	}

	/**
	 * @param stateProvider the stateProvider to set
	 */
	public void setStateProvider(StateProvider stateProvider) {
		this.stateProvider = stateProvider;
	}

	/**
     * @return "input"
     */
    public String getInputResultName() {
    	return INPUT;
    }
    
    /**
     * prepare
     * @throws Exception if an error occurs
     */
    public void prepare() throws Exception {
    	
    }
    
    /**
     * @return "input"
     * @throws Exception if an error occurs
     */
    public String input() throws Exception {
        return INPUT;
    }
    
    /**
     * @return "success"
     * @throws Exception if an error occurs
     */
    public String doDefault() throws Exception {
        return SUCCESS;
    }

    /**
     * A default implementation that does nothing an returns "success".
     * <p/>
     * Subclasses should override this method to provide their business logic.
     * <p/>
     * See also {@link com.opensymphony.xwork2.Action#execute()}.
     *
     * @return returns {@link #SUCCESS}
     * @throws Exception  can be thrown by subclasses.
     */
    public String execute() throws Exception {
        return SUCCESS;
    }

    /**
     * A default implementation that validates nothing.
     * Subclasses should override this method to provide validations.
     */
    public void validate() {
    }

    /**
     * <!-- START SNIPPET: pause-method -->
     * Stops the action invocation immediately (by throwing a PauseException) and causes the action invocation to return
     * the specified result, such as {@link #SUCCESS}, {@link #INPUT}, etc.
     * <p/>
     *
     * The next time this action is invoked (and using the same continuation ID), the method will resume immediately
     * after where this method was called, with the entire call stack in the execute method restored.
     * <p/>
     *
     * Note: this method can <b>only</b> be called within the {@link #execute()} method.
     * <!-- END SNIPPET: pause-method -->
     *
     * @param result the result to return - the same type of return value in the {@link #execute()} method.
     */
    public void pause(String result) {
    }

    /**
     * @see com.opensymphony.xwork2.LocaleProvider#getLocale()
     */
    public Locale getLocale() {
        ActionContext ctx = ActionContext.getContext();
        if (ctx != null) {
            return ctx.getLocale();
        } else {
            log.debug("Action context not initialized");
            return null;
        }
    }

    /**
	 * @see nuts.exts.xwork2.StateProvider#loadState(java.lang.String)
	 */
	public Object loadState(String name) {
		return stateProvider.loadState(name);
	}

	/**
	 * @see nuts.exts.xwork2.StateProvider#saveState(java.lang.String, java.lang.Object)
	 */
	public void saveState(String name, Object value) {
		stateProvider.saveState(name, value);
	}

	/**
     * @see com.opensymphony.xwork2.TextProvider#hasKey(java.lang.String)
     */
    public boolean hasKey(String key) {
        return textProvider.hasKey(key);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String)
     */
    public String getText(String aTextName) {
        return textProvider.getText(aTextName);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String)
     */
    public String getText(String aTextName, String defaultValue) {
        return textProvider.getText(aTextName, defaultValue);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String, java.lang.String)
     */
    public String getText(String aTextName, String defaultValue, String obj) {
        return textProvider.getText(aTextName, defaultValue, obj);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.util.List)
     */
    public String getText(String aTextName, List<?> args) {
        return textProvider.getText(aTextName, args);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String[])
     */
    public String getText(String key, String[] args) {
        return textProvider.getText(key, args);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String, java.util.List)
     */
    public String getText(String aTextName, String defaultValue, List<?> args) {
        return textProvider.getText(aTextName, defaultValue, args);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String, java.lang.String[])
     */
    public String getText(String key, String defaultValue, String[] args) {
        return textProvider.getText(key, defaultValue, args);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String, java.util.List, com.opensymphony.xwork2.util.ValueStack)
     */
    public String getText(String key, String defaultValue, List<?> args, ValueStack stack) {
        return textProvider.getText(key, defaultValue, args, stack);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getText(java.lang.String, java.lang.String, java.lang.String[], com.opensymphony.xwork2.util.ValueStack)
     */
    public String getText(String key, String defaultValue, String[] args, ValueStack stack) {
        return textProvider.getText(key, defaultValue, args, stack);
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getTexts()
     */
    public ResourceBundle getTexts() {
        return textProvider.getTexts();
    }

    /**
     * @see com.opensymphony.xwork2.TextProvider#getTexts(java.lang.String)
     */
    public ResourceBundle getTexts(String aBundleName) {
        return textProvider.getTexts(aBundleName);
    }

	/**
	 * getTextAsBoolean
	 * @param name resource name
	 * @return boolean value
	 */
	public Boolean getTextAsBoolean(String name) {
		return textProvider.getTextAsBoolean(name);
	}

	/**
	 * getTextAsBoolean
	 * @param name resource name
	 * @param defaultValue default value
	 * @return boolean value
	 */
	public Boolean getTextAsBoolean(String name, Boolean defaultValue) {
		return textProvider.getTextAsBoolean(name, defaultValue);
	}

	/**
	 * getTextAsInt
	 * @param name resource name
	 * @return integer value
	 */
	public Integer getTextAsInt(String name) {
		return textProvider.getTextAsInt(name);
	}

	/**
	 * getTextAsInt
	 * @param name resource name
	 * @param defaultValue default value
	 * @return integer value
	 */
	public Integer getTextAsInt(String name, Integer defaultValue) {
		return textProvider.getTextAsInt(name, defaultValue);
	}

	/**
	 * getTextAsLong
	 * @param name resource name
	 * @return long value
	 */
	public Long getTextAsLong(String name) {
		return textProvider.getTextAsLong(name);
	}

	/**
	 * getTextAsLong
	 * @param name resource name
	 * @param defaultValue default value
	 * @return long value
	 */
	public Long getTextAsLong(String name, Long defaultValue) {
		return textProvider.getTextAsLong(name, defaultValue);
	}

	/**
	 * getTextAsList
	 * @param name resource name
	 * @return List value
	 */
	public List getTextAsList(String name) {
		return textProvider.getTextAsList(name);
	}
	
	/**
	 * getTextAsList
	 * @param name resource name
	 * @param defaultValue default value
	 * @return list value
	 */
	public List getTextAsList(String name, List defaultValue) {
		return textProvider.getTextAsList(name, defaultValue);
	}

	/**
	 * getTextAsMap
	 * @param name resource name
	 * @return map value
	 */
	public Map getTextAsMap(String name) {
		return textProvider.getTextAsMap(name);
	}
	
	/**
	 * getTextAsMap
	 * @param name resource name
	 * @param defaultValue default value
	 * @return map value
	 */
	public Map getTextAsMap(String name, Map defaultValue) {
		return textProvider.getTextAsMap(name, defaultValue);
	}
	
	/**
	 * @see com.opensymphony.xwork2.ValidationAware#setFieldErrors(java.util.Map)
	 */
	public void setFieldErrors(Map<String, List<String>> errorMap) {
        actionValidationAware.setFieldErrors(errorMap);
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#getFieldErrors()
     */
    public Map<String, List<String>> getFieldErrors() {
        return actionValidationAware.getFieldErrors();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#setActionErrors(java.util.Collection)
     */
    public void setActionErrors(Collection<String> errorMessages) {
        actionValidationAware.setActionErrors(errorMessages);
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#getActionErrors()
     */
    public Collection<String> getActionErrors() {
        return actionValidationAware.getActionErrors();
    }

    /**
     * @see nuts.exts.xwork2.ActionValidationAware#setActionWarnings(java.util.Collection)
     */
    public void setActionWarnings(Collection<String> warnMessages) {
        actionValidationAware.setActionWarnings(warnMessages);
    }

    /**
     * @see nuts.exts.xwork2.ActionValidationAware#getActionWarnings()
     */
    public Collection<String> getActionWarnings() {
        return actionValidationAware.getActionWarnings();
    }

    /**
     * @see nuts.exts.xwork2.ActionValidationAware#setActionConfirms(java.util.Collection)
     */
    public void setActionConfirms(Collection<String> confirmMessages) {
        actionValidationAware.setActionConfirms(confirmMessages);
    }

    /**
     * @see nuts.exts.xwork2.ActionValidationAware#getActionConfirms()
     */
    public Collection<String> getActionConfirms() {
        return actionValidationAware.getActionConfirms();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#setActionMessages(java.util.Collection)
     */
    public void setActionMessages(Collection<String> messages) {
        actionValidationAware.setActionMessages(messages);
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#getActionMessages()
     */
    public Collection<String> getActionMessages() {
        return actionValidationAware.getActionMessages();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#addFieldError(java.lang.String, java.lang.String)
     */
    public void addFieldError(String fieldName, String errorMessage) {
        actionValidationAware.addFieldError(fieldName, errorMessage);
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#addActionError(java.lang.String)
     */
    public void addActionError(String anErrorMessage) {
        actionValidationAware.addActionError(anErrorMessage);
    }

    /**
     * @see nuts.exts.xwork2.ActionValidationAware#addActionWarning(java.lang.String)
     */
    public void addActionWarning(String anWarnMessage) {
        actionValidationAware.addActionWarning(anWarnMessage);
    }

    /**
     * @see nuts.exts.xwork2.ActionValidationAware#addActionConfirm(java.lang.String)
     */
    public void addActionConfirm(String anConfirmMessage) {
        actionValidationAware.addActionConfirm(anConfirmMessage);
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#addActionMessage(java.lang.String)
     */
    public void addActionMessage(String aMessage) {
        actionValidationAware.addActionMessage(aMessage);
    }
    /**
     * @see com.opensymphony.xwork2.ValidationAware#hasFieldErrors()
     */
    public boolean hasFieldErrors() {
        return actionValidationAware.hasFieldErrors();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#hasActionErrors()
     */
    public boolean hasActionErrors() {
        return actionValidationAware.hasActionErrors();
    }

    /**
     * @see nuts.exts.xwork2.ActionValidationAware#hasActionWarnings()
     */
    public boolean hasActionWarnings() {
        return actionValidationAware.hasActionWarnings();
    }

    /**
     * @see nuts.exts.xwork2.ActionValidationAware#hasActionConfirms()
     */
    public boolean hasActionConfirms() {
        return actionValidationAware.hasActionConfirms();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#hasActionMessages()
     */
    public boolean hasActionMessages() {
        return actionValidationAware.hasActionMessages();
    }

    /**
     * @see com.opensymphony.xwork2.ValidationAware#hasErrors()
     */
    public boolean hasErrors() {
        return actionValidationAware.hasErrors();
    }

    /**
     * @return true if action has messages
     */
    public boolean hasMessages() {
        return actionValidationAware.hasMessages();
    }

    /**
     * Clears field errors. Useful for Continuations and other situations
     * where you might want to clear parts of the state on the same action.
     */
    public void clearFieldErrors() {
        actionValidationAware.clearFieldErrors();
    }

    /**
     * Clears action errors. Useful for Continuations and other situations
     * where you might want to clear parts of the state on the same action.
     */
    public void clearActionErrors() {
        actionValidationAware.clearActionErrors();
    }

    /**
     * Clears all errors. Useful for Continuations and other situations
     * where you might want to clear parts of the state on the same action.
     */
    public void clearErrors() {
        actionValidationAware.clearErrors();
    }

	/**
	 * Clears action warning list.
	 * <p/>
	 * Will clear the list that contains action warning.
	 */
	public void clearActionWarnings() {
		actionValidationAware.clearActionWarnings();
	}

	/**
	 * Clears action confirm list.
	 * <p/>
	 * Will clear the list that contains action confirm.
	 */
	public void clearActionConfirms() {
		actionValidationAware.clearActionConfirms();
	}

	/**
	 * Clears action messages list.
	 * <p/>
	 * Will clear the list that contains action messages.
	 */
	public void clearActionMessages() {
		actionValidationAware.clearActionMessages();
	}

	/**
     * Clears messages. Useful for Continuations and other situations
     * where you might want to clear parts of the state on the same action.
     */
    public void clearMessages() {
        actionValidationAware.clearMessages();
    }

    /**
     * Clears all errors and messages. Useful for Continuations and other situations
     * where you might want to clear parts of the state on the same action.
     */
    public void clearErrorsAndMessages() {
        actionValidationAware.clearErrorsAndMessages();
    }

	/**
	 * @see ApplicationValidationAware#setApplicationErrors(java.util.Map)
	 */
	public void setApplicationErrors(Map<String, String> errorMessages) {
		applicationValidationAware.setApplicationErrors(errorMessages);
	}

	/**
	 * @see ApplicationValidationAware#getApplicationErrors()
	 */
	public Map<String, String> getApplicationErrors() {
		return applicationValidationAware.getApplicationErrors();
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#setApplicationWarnings(java.util.Map)
	 */
	public void setApplicationWarnings(Map<String, String> applicationWarnings) {
		applicationValidationAware.setApplicationWarnings(applicationWarnings);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#getApplicationWarnings()
	 */
	public Map<String, String> getApplicationWarnings() {
		return applicationValidationAware.getApplicationWarnings();
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#setApplicationConfirms(java.util.Map)
	 */
	public void setApplicationConfirms(Map<String, String> applicationConfirms) {
		applicationValidationAware.setApplicationConfirms(applicationConfirms);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#getApplicationConfirms()
	 */
	public Map<String, String> getApplicationConfirms() {
		return applicationValidationAware.getApplicationConfirms();
	}

	/**
	 * @see ApplicationValidationAware#setApplicationMessages(java.util.Map)
	 */
	public void setApplicationMessages(Map<String, String> messages) {
		applicationValidationAware.setApplicationMessages(messages);
	}

	/**
	 * @see ApplicationValidationAware#getApplicationMessages()
	 */
	public Map<String, String> getApplicationMessages() {
		return applicationValidationAware.getApplicationMessages();
	}

	/**
	 * @see ApplicationValidationAware#addApplicationError(java.lang.String, java.lang.String)
	 */
	public void addApplicationError(String id, String anErrorMessage) {
		applicationValidationAware.addApplicationError(id, anErrorMessage);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#addApplicationWarning(java.lang.String,
	 *      java.lang.String)
	 */
	public void addApplicationWarning(String id, String aWarningMessage) {
		applicationValidationAware.addApplicationWarning(id, aWarningMessage);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#addApplicationConfirm(java.lang.String,
	 *      java.lang.String)
	 */
	public void addApplicationConfirm(String id, String aConfirmMessage) {
		applicationValidationAware.addApplicationConfirm(id, aConfirmMessage);
	}

	/**
	 * @see ApplicationValidationAware#addApplicationMessage(java.lang.String, java.lang.String)
	 */
	public void addApplicationMessage(String id, String aMessage) {
		applicationValidationAware.addApplicationMessage(id, aMessage);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#removeApplicationError(java.lang.String)
	 */
	public String removeApplicationError(String id) {
		return applicationValidationAware.removeApplicationError(id);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#removeApplicationWarning(java.lang.String)
	 */
	public String removeApplicationWarning(String id) {
		return applicationValidationAware.removeApplicationWarning(id);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#removeApplicationConfirm(java.lang.String)
	 */
	public String removeApplicationConfirm(String id) {
		return applicationValidationAware.removeApplicationConfirm(id);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#removeApplicationMessage(java.lang.String)
	 */
	public String removeApplicationMessage(String id) {
		return applicationValidationAware.removeApplicationMessage(id);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#hasApplicationError(java.lang.String)
	 */
	public boolean hasApplicationError(String id) {
		return applicationValidationAware.hasApplicationError(id);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#hasApplicationWarning(java.lang.String)
	 */
	public boolean hasApplicationWarning(String id) {
		return applicationValidationAware.hasApplicationWarning(id);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#hasApplicationConfirm(java.lang.String)
	 */
	public boolean hasApplicationConfirm(String id) {
		return applicationValidationAware.hasApplicationConfirm(id);
	}

	/**
	 * @see ApplicationValidationAware#hasApplicationMessage(java.lang.String)
	 */
	public boolean hasApplicationMessage(String id) {
		return applicationValidationAware.hasApplicationMessage(id);
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#hasApplicationErrors()
	 */
	public boolean hasApplicationErrors() {
		return applicationValidationAware.hasApplicationErrors();
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#hasApplicationWarnings()
	 */
	public boolean hasApplicationWarnings() {
		return applicationValidationAware.hasApplicationWarnings();
	}

	/**
	 * @see nuts.exts.xwork2.ApplicationValidationAware#hasApplicationConfirms()
	 */
	public boolean hasApplicationConfirms() {
		return applicationValidationAware.hasApplicationConfirms();
	}

	/**
	 * @see ApplicationValidationAware#hasApplicationMessages()
	 */
	public boolean hasApplicationMessages() {
		return applicationValidationAware.hasApplicationMessages();
	}

	/**
	 * Clears application errors map.
	 * <p/>
	 * Will clear the map that contains application errors.
	 */
	public void clearApplicationErrors() {
		applicationValidationAware.clearApplicationErrors();
	}

	/**
	 * Clears application warning map.
	 * <p/>
	 * Will clear the map that contains application warning.
	 */
	public void clearApplicationWarnings() {
		applicationValidationAware.clearApplicationWarnings();
	}

	/**
	 * Clears application confirm map.
	 * <p/>
	 * Will clear the map that contains application confirm.
	 */
	public void clearApplicationConfirms() {
		applicationValidationAware.clearApplicationConfirms();
	}

	/**
	 * Clears application messages map.
	 * <p/>
	 * Will clear the map that contains application messages.
	 */
	public void clearApplicationMessages() {
		applicationValidationAware.clearApplicationMessages();
	}

	/**
	 * Clears all application error and messages.
	 * <p/>
	 * Will clear all the maps.
	 */
	public void clearApplicationErrorsAndMessages() {
		applicationValidationAware.clearErrorsAndMessages();
	}

	/**
	 * @see SessionValidationAware#setSessionErrors(java.util.Map)
	 */
	public void setSessionErrors(Map<String, String> errorMessages) {
		sessionValidationAware.setSessionErrors(errorMessages);
	}

	/**
	 * @see SessionValidationAware#getSessionErrors()
	 */
	public Map<String, String> getSessionErrors() {
		return sessionValidationAware.getSessionErrors();
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#setSessionWarnings(java.util.Map)
	 */
	public void setSessionWarnings(Map<String, String> sessionWarnings) {
		sessionValidationAware.setSessionWarnings(sessionWarnings);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#getSessionWarnings()
	 */
	public Map<String, String> getSessionWarnings() {
		return sessionValidationAware.getSessionWarnings();
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#setSessionConfirms(java.util.Map)
	 */
	public void setSessionConfirms(Map<String, String> sessionConfirms) {
		sessionValidationAware.setSessionConfirms(sessionConfirms);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#getSessionConfirms()
	 */
	public Map<String, String> getSessionConfirms() {
		return sessionValidationAware.getSessionConfirms();
	}

	/**
	 * @see SessionValidationAware#setSessionMessages(java.util.Map)
	 */
	public void setSessionMessages(Map<String, String> messages) {
		sessionValidationAware.setSessionMessages(messages);
	}

	/**
	 * @see SessionValidationAware#getSessionMessages()
	 */
	public Map<String, String> getSessionMessages() {
		return sessionValidationAware.getSessionMessages();
	}

	/**
	 * @see SessionValidationAware#addSessionError(java.lang.String, java.lang.String)
	 */
	public void addSessionError(String id, String anErrorMessage) {
		sessionValidationAware.addSessionError(id, anErrorMessage);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#addSessionWarning(java.lang.String,
	 *      java.lang.String)
	 */
	public void addSessionWarning(String id, String aWarningMessage) {
		sessionValidationAware.addSessionWarning(id, aWarningMessage);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#addSessionConfirm(java.lang.String,
	 *      java.lang.String)
	 */
	public void addSessionConfirm(String id, String aConfirmMessage) {
		sessionValidationAware.addSessionConfirm(id, aConfirmMessage);
	}

	/**
	 * @see SessionValidationAware#addSessionMessage(java.lang.String, java.lang.String)
	 */
	public void addSessionMessage(String id, String aMessage) {
		sessionValidationAware.addSessionMessage(id, aMessage);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#removeSessionError(java.lang.String)
	 */
	public String removeSessionError(String id) {
		return sessionValidationAware.removeSessionError(id);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#removeSessionWarning(java.lang.String)
	 */
	public String removeSessionWarning(String id) {
		return sessionValidationAware.removeSessionWarning(id);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#removeSessionConfirm(java.lang.String)
	 */
	public String removeSessionConfirm(String id) {
		return sessionValidationAware.removeSessionConfirm(id);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#removeSessionMessage(java.lang.String)
	 */
	public String removeSessionMessage(String id) {
		return sessionValidationAware.removeSessionMessage(id);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#hasSessionError(java.lang.String)
	 */
	public boolean hasSessionError(String id) {
		return sessionValidationAware.hasSessionError(id);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#hasSessionWarning(java.lang.String)
	 */
	public boolean hasSessionWarning(String id) {
		return sessionValidationAware.hasSessionWarning(id);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#hasSessionConfirm(java.lang.String)
	 */
	public boolean hasSessionConfirm(String id) {
		return sessionValidationAware.hasSessionConfirm(id);
	}

	/**
	 * @see SessionValidationAware#hasSessionMessage(java.lang.String)
	 */
	public boolean hasSessionMessage(String id) {
		return sessionValidationAware.hasSessionMessage(id);
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#hasSessionErrors()
	 */
	public boolean hasSessionErrors() {
		return sessionValidationAware.hasSessionErrors();
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#hasSessionWarnings()
	 */
	public boolean hasSessionWarnings() {
		return sessionValidationAware.hasSessionWarnings();
	}

	/**
	 * @see nuts.exts.xwork2.SessionValidationAware#hasSessionConfirms()
	 */
	public boolean hasSessionConfirms() {
		return sessionValidationAware.hasSessionConfirms();
	}

	/**
	 * @see SessionValidationAware#hasSessionMessages()
	 */
	public boolean hasSessionMessages() {
		return sessionValidationAware.hasSessionMessages();
	}

	/**
	 * Clears session errors map.
	 * <p/>
	 * Will clear the map that contains session errors.
	 */
	public void clearSessionErrors() {
		sessionValidationAware.clearSessionErrors();
	}

	/**
	 * Clears session warning map.
	 * <p/>
	 * Will clear the map that contains session warning.
	 */
	public void clearSessionWarnings() {
		sessionValidationAware.clearSessionWarnings();
	}

	/**
	 * Clears session confirm map.
	 * <p/>
	 * Will clear the map that contains session confirm.
	 */
	public void clearSessionConfirms() {
		sessionValidationAware.clearSessionConfirms();
	}

	/**
	 * Clears session messages map.
	 * <p/>
	 * Will clear the map that contains session messages.
	 */
	public void clearSessionMessages() {
		sessionValidationAware.clearSessionMessages();
	}

	/**
	 * Clears all session error and messages.
	 * <p/>
	 * Will clear all the maps.
	 */
	public void clearSessionErrorsAndMessages() {
		sessionValidationAware.clearErrorsAndMessages();
	}

	/**
	 * @see nuts.exts.xwork2.PermissionValidator#hasPermission(java.lang.String)
	 */
	public boolean hasPermission(String action) {
		return hasPermission(ContextUtils.getNamespace(), action);
	}

	/**
	 * @see nuts.exts.xwork2.PermissionValidator#hasPermission(java.lang.String, java.lang.String)
	 */
	public boolean hasPermission(String namespace, String action) {
		return true;
	}

	/**
	 * @see nuts.exts.xwork2.PermissionValidator#hasDataPermission(java.lang.Object, java.lang.String)
	 */
	public boolean hasDataPermission(Object data, String action) {
		return hasDataPermission(data, ContextUtils.getNamespace(), action);
	}

	/**
	 * @see nuts.exts.xwork2.PermissionValidator#hasDataPermission(java.lang.Object, java.lang.String, java.lang.String)
	 */
	public boolean hasDataPermission(Object data, String namespace, String action) {
		return hasPermission(namespace, action);
	}

	/**
     * Clone action
     * @return cloned object
     * @throws CloneNotSupportedException if clone failed
     */
    public Object clone() throws CloneNotSupportedException {
        return super.clone();
    }

}
