/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.struts2.components;

import java.io.Writer;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.struts2.components.UIBean;
import org.apache.struts2.views.annotations.StrutsTag;
import org.apache.struts2.views.annotations.StrutsTagAttribute;

import com.opensymphony.xwork2.util.ValueStack;
import com.opensymphony.xwork2.util.logging.Logger;
import com.opensymphony.xwork2.util.logging.LoggerFactory;

/**
 * <!-- START SNIPPET: javadoc -->
 * Render a reset button. The reset tag is used together with the form tag to provide form resetting.
 * The reset can have two different types of rendering:
 * <ul>
 * <li>input: renders as html &lt;input type="reset"...&gt;</li>
 * <li>button: renders as html &lt;button type="reset"...&gt;</li>
 * </ul>
 * Please note that the button type has advantages by adding the possibility to seperate the submitted value from the
 * text shown on the button face, but has issues with Microsoft Internet Explorer at least up to 6.0
 * <!-- END SNIPPET: javadoc -->
 *
 * <p/> <b>Examples</b>
 *
 * <pre>
 * <!-- START SNIPPET: example -->
 * &lt;s:reset value="Reset" /&gt;
 * <!-- END SNIPPET: example -->
 * </pre>
 *
 * <pre>
 * <!-- START SNIPPET: example2 -->
 * Render an button reset:
 * &lt;s:reset type="button" key="reset"/&gt;
 * <!-- END SNIPPET: example2 -->
 * </pre>
 *
 */
@StrutsTag(
	name="reset",
	tldTagClass="org.apache.struts2.views.jsp.ui.ResetTag",
	description="Render a reset button",
	allowDynamicAttributes=true)
public class Reset extends FormButton {
	private static final Logger LOG = LoggerFactory.getLogger(Reset.class);

	/**
	 * OPEN_TEMPLATE = "n-reset";
	 */
	final public static String OPEN_TEMPLATE = "n-reset";

	/**
	 * TEMPLATE = "n-reset-close";
	 */
	final public static String TEMPLATE = "n-reset-close";

	protected String icon;
	protected String sicon;

	/**
	 * @see UIBean
	 * @param stack value stack
	 * @param request http servlet request
	 * @param response http servlet response
	 */
	public Reset(ValueStack stack, HttpServletRequest request, HttpServletResponse response) {
		super(stack, request, response);
	}

	/**
	 * @return default open template
	 */
	public String getDefaultOpenTemplate() {
		return OPEN_TEMPLATE;
	}

	protected String getDefaultTemplate() {
		return TEMPLATE;
	}

	/**
	 * @see org.apache.struts2.components.FormButton#evaluateExtraParams()
	 */
	public void evaluateParams() {

		if ((key == null) && (value == null)) {
			value = "Reset";
		}

		if (((key != null)) && (value == null)) {
			this.value = "%{getText('"+key +"')}";
		}

		super.evaluateParams();

	}

    public void evaluateExtraParams() {
        super.evaluateExtraParams();

        if (icon != null) {
            addParameter("icon", findString(icon));
        }

        if (sicon != null) {
            addParameter("sicon", findString(sicon));
        }
    }
    
	/**
	 * Indicate whether the concrete button supports the type "image".
	 *
	 * @return <tt>false</tt> to indicate type image is supported.
	 */
	protected boolean supportsImageType() {
		return false;
	}

	/**
	 * @return true
	 */
	@Override
	public boolean usesBody() {
		return true;
	}

	/**
	 * Overrides to be able to render body in a template rather than always before the template
	 * @see org.apache.struts2.components.UIBean#end(java.io.Writer, java.lang.String)
	 */
	public boolean end(Writer writer, String body) {
		evaluateParams();
		try {
			addParameter("body", body);

			mergeTemplate(writer, buildTemplateName(template, getDefaultTemplate()));
		} catch (Exception e) {
			LOG.error("error when rendering", e);
		}
		finally {
			popComponentStack();
		}

		return false;
	}

	/**
	 * @param icon the icon to set
	 */
    @StrutsTagAttribute(description="icon attribute",  type="String")
	public void setIcon(String icon) {
		this.icon = icon;
	}

	/**
	 * @param sicon the sicon to set
	 */
    @StrutsTagAttribute(description="second icon attribute",  type="String")
	public void setSicon(String sicon) {
		this.sicon = sicon;
	}

}
