/*
 * This file is part of Nuts Framework.
 * Copyright (C) 2009 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.exts.struts2.components;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import nuts.exts.struts2.NutsStrutsConstants;

import org.apache.struts2.components.UIBean;
import org.apache.struts2.views.annotations.StrutsTag;
import org.apache.struts2.views.annotations.StrutsTagAttribute;

import com.opensymphony.xwork2.inject.Inject;
import com.opensymphony.xwork2.util.ValueStack;

/**
 * <!-- START SNIPPET: javadoc --> Renders an HTML input element of type
 * checkbox, populated by the specified property from the ValueStack. <!-- END
 * SNIPPET: javadoc -->
 * 
 * <p/>
 * <b>Examples</b>
 * 
 * <pre>
 * <!-- START SNIPPET: example -->
 * JSP:
 * &lt;s:checkbox label="checkbox test" name="checkboxField1" value="aBoolean" fieldValue="true"/&gt;
 * 
 * Velocity:
 * #tag( Checkbox "label=checkbox test" "name=checkboxField1" "value=aBoolean" )
 * 
 * Resulting HTML (simple template, aBoolean == true):
 * &lt;input type="checkbox" name="checkboxField1" value="true" checked="checked" /&gt;
 * 
 * <!-- END SNIPPET: example -->
 * </pre>
 * 
 */
@StrutsTag(name = "checkbox", tldTagClass = "nuts.exts.struts2.views.jsp.ui.CheckboxTag", description = "Render a checkbox input field", allowDynamicAttributes = true)
public class Checkbox extends UIBean {
	/**
	 * TEMPLATE = "n-checkbox";
	 */
	public final static String TEMPLATE = "n-checkbox";

	protected Boolean checkboxIntercept;
	protected String fieldValue;
	protected String readonly;

	/**
	 * Constructor
	 * 
	 * @param stack value stack
	 * @param request request
	 * @param response response
	 */
	public Checkbox(ValueStack stack, HttpServletRequest request,
			HttpServletResponse response) {
		super(stack, request, response);
	}

	protected String getDefaultTemplate() {
		return TEMPLATE;
	}

	protected void evaluateExtraParams() {
		super.evaluateExtraParams();

		if (fieldValue != null) {
			addParameter("fieldValue", findString(fieldValue));
		}
		else {
			addParameter("fieldValue", "true");
		}

		if (readonly != null) {
			addParameter("readonly", findValue(readonly, Boolean.class));
		}
		
		if (checkboxIntercept != null) {
			addParameter("checkboxIntercept", checkboxIntercept);
		}
	}

	protected Class getValueClassType() {
		return Boolean.class; // for checkboxes, everything needs to end up as a
								// Boolean
	}

    /**
     * @param intercept intercept
     */
    @Inject(value=NutsStrutsConstants.NUTS_CHECKBOX_INTERCEPT, required=false)
    public void setCheckboxIntercept(String intercept) {
    	checkboxIntercept = Boolean.parseBoolean(intercept);
    }

	/**
	 * @param fieldValue the fieldValue to set
	 */
	@StrutsTagAttribute(description = "The actual HTML value attribute of the checkbox.", defaultValue = "true")
	public void setFieldValue(String fieldValue) {
		this.fieldValue = fieldValue;
	}

    /**
     * @param readonly the readonly to set
     */
    @StrutsTagAttribute(description="Whether the input is readonly", type="Boolean", defaultValue="false")
    public void setReadonly(String readonly) {
        this.readonly = readonly;
    }
}
