/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.lang.mutable;

import org.junit.Test;
import static org.junit.Assert.*;

/**
 * JUnit tests.
 * @see MutableLong
 */
public class MutableLongTest {

	// ----------------------------------------------------------------
	@Test
	public void testConstructors() {
		assertEquals(0, new MutableLong().longValue());

		assertEquals(1, new MutableLong(1).longValue());

		assertEquals(2, new MutableLong(Long.valueOf(2)).longValue());
		assertEquals(3, new MutableLong(new MutableLong(3)).longValue());

		assertEquals(2, new MutableLong("2").longValue());

	}

	@Test(expected = NullPointerException.class)
	public void testConstructorNull() {
		new MutableLong((Number)null);
	}

	@Test
	public void testGetSet() {
		final MutableLong mutNum = new MutableLong(0);
		assertEquals(0, new MutableLong().longValue());
		assertEquals(Long.valueOf(0), new MutableLong().getValue());

		mutNum.setValue(1);
		assertEquals(1, mutNum.longValue());
		assertEquals(Long.valueOf(1), mutNum.getValue());

		mutNum.setValue(Long.valueOf(2));
		assertEquals(2, mutNum.longValue());
		assertEquals(Long.valueOf(2), mutNum.getValue());

		mutNum.setValue(new MutableLong(3));
		assertEquals(3, mutNum.longValue());
		assertEquals(Long.valueOf(3), mutNum.getValue());
	}

	@Test(expected = NullPointerException.class)
	public void testSetNull() {
		final MutableLong mutNum = new MutableLong(0);
		mutNum.setValue(null);
	}

	@Test
	public void testEquals() {
		final MutableLong mutNumA = new MutableLong(0);
		final MutableLong mutNumB = new MutableLong(0);
		final MutableLong mutNumC = new MutableLong(1);

		assertTrue(mutNumA.equals(mutNumA));
		assertTrue(mutNumA.equals(mutNumB));
		assertTrue(mutNumB.equals(mutNumA));
		assertTrue(mutNumB.equals(mutNumB));
		assertFalse(mutNumA.equals(mutNumC));
		assertFalse(mutNumB.equals(mutNumC));
		assertTrue(mutNumC.equals(mutNumC));
		assertFalse(mutNumA.equals(null));
		assertFalse(mutNumA.equals(Long.valueOf(0)));
		assertFalse(mutNumA.equals("0"));
	}

	@Test
	public void testHashCode() {
		final MutableLong mutNumA = new MutableLong(0);
		final MutableLong mutNumB = new MutableLong(0);
		final MutableLong mutNumC = new MutableLong(1);

		assertTrue(mutNumA.hashCode() == mutNumA.hashCode());
		assertTrue(mutNumA.hashCode() == mutNumB.hashCode());
		assertFalse(mutNumA.hashCode() == mutNumC.hashCode());
		assertTrue(mutNumA.hashCode() == Long.valueOf(0).hashCode());
	}

	@Test
	public void testCompareTo() {
		final MutableLong mutNum = new MutableLong(0);

		assertEquals(0, mutNum.compareTo(new MutableLong(0)));
		assertEquals(+1, mutNum.compareTo(new MutableLong(-1)));
		assertEquals(-1, mutNum.compareTo(new MutableLong(1)));
	}

	@Test(expected = NullPointerException.class)
	public void testCompareToNull() {
		final MutableLong mutNum = new MutableLong(0);
		mutNum.compareTo(null);
	}

	@Test
	public void testPrimitiveValues() {
		final MutableLong mutNum = new MutableLong(1L);

		assertEquals(1.0F, mutNum.floatValue(), 0);
		assertEquals(1.0, mutNum.doubleValue(), 0);
		assertEquals((byte)1, mutNum.byteValue());
		assertEquals((short)1, mutNum.shortValue());
		assertEquals(1, mutNum.intValue());
		assertEquals(1L, mutNum.longValue());
	}

	@Test
	public void testToLong() {
		assertEquals(Long.valueOf(0L), new MutableLong(0L).toLong());
		assertEquals(Long.valueOf(123L), new MutableLong(123L).toLong());
	}

	@Test
	public void testIncrement() {
		final MutableLong mutNum = new MutableLong(1);
		mutNum.increment();

		assertEquals(2, mutNum.intValue());
		assertEquals(2L, mutNum.longValue());
	}

	@Test
	public void testDecrement() {
		final MutableLong mutNum = new MutableLong(1);
		mutNum.decrement();

		assertEquals(0, mutNum.intValue());
		assertEquals(0L, mutNum.longValue());
	}

	@Test
	public void testAddValuePrimitive() {
		final MutableLong mutNum = new MutableLong(1);
		mutNum.add(1);

		assertEquals(2, mutNum.intValue());
		assertEquals(2L, mutNum.longValue());
	}

	@Test
	public void testAddValueObject() {
		final MutableLong mutNum = new MutableLong(1);
		mutNum.add(Long.valueOf(1));

		assertEquals(2, mutNum.intValue());
		assertEquals(2L, mutNum.longValue());
	}

	@Test
	public void testSubtractValuePrimitive() {
		final MutableLong mutNum = new MutableLong(1);
		mutNum.subtract(1);

		assertEquals(0, mutNum.intValue());
		assertEquals(0L, mutNum.longValue());
	}

	@Test
	public void testSubtractValueObject() {
		final MutableLong mutNum = new MutableLong(1);
		mutNum.subtract(Long.valueOf(1));

		assertEquals(0, mutNum.intValue());
		assertEquals(0L, mutNum.longValue());
	}

	@Test
	public void testToString() {
		assertEquals("0", new MutableLong(0).toString());
		assertEquals("10", new MutableLong(10).toString());
		assertEquals("-123", new MutableLong(-123).toString());
	}

}
