/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.lang.escape;

import java.io.IOException;

/**
 * Translates codepoints to their Unicode escaped value.
 *
 */
public class UnicodeEscaper extends CodePointTranslator {

	private final int below;
	private final int above;
	private final boolean between;

	/**
	 * <p>
	 * Constructs a <code>UnicodeEscaper</code> for all characters.
	 * </p>
	 */
	public UnicodeEscaper() {
		this(0, Integer.MAX_VALUE, true);
	}

    /**
	 * <p>
	 * Constructs a <code>UnicodeEscaper</code> for the specified range. This is the underlying
	 * method for the other constructors/builders. The <code>below</code> and <code>above</code>
	 * boundaries are inclusive when <code>between</code> is <code>true</code> and exclusive when it
	 * is <code>false</code>.
	 * </p>
	 * 
	 * @param below int value representing the lowest codepoint boundary
	 * @param above int value representing the highest codepoint boundary
	 * @param between whether to escape between the boundaries or outside them
	 */
	protected UnicodeEscaper(final int below, final int above, final boolean between) {
		this.below = below;
		this.above = above;
		this.between = between;
	}

	/**
	 * <p>
	 * Constructs a <code>UnicodeEscaper</code> below the specified value (exclusive).
	 * </p>
	 * 
	 * @param codepoint below which to escape
	 * @return the newly created {@code UnicodeEscaper} instance
	 */
	public static UnicodeEscaper below(final int codepoint) {
		return outsideOf(codepoint, Integer.MAX_VALUE);
	}

	/**
	 * <p>
	 * Constructs a <code>UnicodeEscaper</code> above the specified value (exclusive).
	 * </p>
	 * 
	 * @param codepoint above which to escape
	 * @return the newly created {@code UnicodeEscaper} instance
	 */
	public static UnicodeEscaper above(final int codepoint) {
		return outsideOf(0, codepoint);
	}

	/**
	 * <p>
	 * Constructs a <code>UnicodeEscaper</code> outside of the specified values (exclusive).
	 * </p>
	 * 
	 * @param codepointLow below which to escape
	 * @param codepointHigh above which to escape
	 * @return the newly created {@code UnicodeEscaper} instance
	 */
	public static UnicodeEscaper outsideOf(final int codepointLow, final int codepointHigh) {
		return new UnicodeEscaper(codepointLow, codepointHigh, false);
	}

	/**
	 * <p>
	 * Constructs a <code>UnicodeEscaper</code> between the specified values (inclusive).
	 * </p>
	 * 
	 * @param codepointLow above which to escape
	 * @param codepointHigh below which to escape
	 * @return the newly created {@code UnicodeEscaper} instance
	 */
	public static UnicodeEscaper between(final int codepointLow, final int codepointHigh) {
		return new UnicodeEscaper(codepointLow, codepointHigh, true);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean translate(final int codepoint, final Appendable out) throws IOException {
		if (between) {
			if (codepoint < below || codepoint > above) {
				return false;
			}
		}
		else {
			if (codepoint >= below && codepoint <= above) {
				return false;
			}
		}

		// TODO: Handle potential + sign per various Unicode escape implementations
		if (codepoint > 0xffff) {
			out.append(toUtf16Escape(codepoint));
		}
		else if (codepoint > 0xfff) {
			out.append("\\u" + hex(codepoint));
		}
		else if (codepoint > 0xff) {
			out.append("\\u0" + hex(codepoint));
		}
		else if (codepoint > 0xf) {
			out.append("\\u00" + hex(codepoint));
		}
		else {
			out.append("\\u000" + hex(codepoint));
		}
		return true;
	}

	/**
	 * Converts the given codepoint to a hex string of the form {@code "\\uXXXX"}
	 * 
	 * @param codepoint a Unicode code point
	 * @return the hex string for the given codepoint
	 */
	protected String toUtf16Escape(final int codepoint) {
		return "\\u" + hex(codepoint);
	}
}
