/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.lang.escape;

import java.io.IOException;

/**
 * Translate escaped octal Strings back to their octal values.
 *
 * For example, "\45" should go back to being the specific value (a %).
 *
 * Note that this currently only supports the viable range of octal for Java; namely 
 * 1 to 377. This is both because parsing Java is the main use case and Integer.parseInt
 * throws an exception when values are larger than octal 377.
 * 
 */
public class OctalUnescaper extends CharSequenceTranslator {

	private static int OCTAL_MAX = 377;

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int translate(final CharSequence input, final int index, final Appendable out) throws IOException {
		if (input.charAt(index) == '\\' && index < (input.length() - 1) && Character.isDigit(input.charAt(index + 1))) {
			final int start = index + 1;

			int end = index + 2;
			while (end < input.length() && Character.isDigit(input.charAt(end))) {
				end++;
				if (Integer.parseInt(input.subSequence(start, end).toString(), 10) > OCTAL_MAX) {
					end--; // rollback
					break;
				}
			}

			out.append((char)(Integer.parseInt(input.subSequence(start, end).toString(), 8)));
			return 1 + end - start;
		}
		return 0;
	}
}
