/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.lang.escape;

import nuts.core.lang.Arrays;

import java.io.IOException;
import java.util.EnumSet;

/**
 * Translate XML numeric entities of the form &#[xX]?\d+;? to 
 * the specific codepoint.
 *
 * Note that the semi-colon is optional.
 * 
 */
public class NumericEntityUnescaper extends CharSequenceTranslator {

	public static enum OPTION {
		semiColonRequired, semiColonOptional, errorIfNoSemiColon
	}

	// TODO?: Create an OptionsSet class to hide some of the conditional logic below
	private final EnumSet<OPTION> options;

	/**
	 * Create a UnicodeUnescaper. The constructor takes a list of options, only one type of which is
	 * currently available (whether to allow, error or ignore the semi-colon on the end of a numeric
	 * entity to being missing). For example, to support numeric entities without a ';': new
	 * NumericEntityUnescaper(NumericEntityUnescaper.OPTION.semiColonOptional) and to throw an
	 * IllegalArgumentException when they're missing: new
	 * NumericEntityUnescaper(NumericEntityUnescaper.OPTION.errorIfNoSemiColon) Note that the
	 * default behaviour is to ignore them.
	 * 
	 * @param options to apply to this unescaper
	 */
	public NumericEntityUnescaper(final OPTION... options) {
		if (options.length > 0) {
			this.options = EnumSet.copyOf(Arrays.asList(options));
		}
		else {
			this.options = EnumSet.copyOf(Arrays.asList(new OPTION[] { OPTION.semiColonRequired }));
		}
	}

	/**
	 * Whether the passed in option is currently set.
	 * 
	 * @param option to check state of
	 * @return whether the option is set
	 */
	public boolean isSet(final OPTION option) {
		return options == null ? false : options.contains(option);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int translate(final CharSequence input, final int index, final Appendable out) throws IOException {
		final int seqEnd = input.length();
		// Uses -2 to ensure there is something after the &#
		if (input.charAt(index) == '&' && index < seqEnd - 2 && input.charAt(index + 1) == '#') {
			int start = index + 2;
			boolean isHex = false;

			final char firstChar = input.charAt(start);
			if (firstChar == 'x' || firstChar == 'X') {
				start++;
				isHex = true;

				// Check there's more than just an x after the &#
				if (start == seqEnd) {
					return 0;
				}
			}

			int end = start;
			// Note that this supports character codes without a ; on the end
			while (end < seqEnd
					&& (input.charAt(end) >= '0' && input.charAt(end) <= '9' || input.charAt(end) >= 'a'
							&& input.charAt(end) <= 'f' || input.charAt(end) >= 'A' && input.charAt(end) <= 'F')) {
				end++;
			}

			final boolean semiNext = end != seqEnd && input.charAt(end) == ';';

			if (!semiNext) {
				if (isSet(OPTION.semiColonRequired)) {
					return 0;
				}
				else if (isSet(OPTION.errorIfNoSemiColon)) {
					throw new IllegalArgumentException("Semi-colon required at end of numeric entity");
				}
			}

			int entityValue;
			try {
				if (isHex) {
					entityValue = Integer.parseInt(input.subSequence(start, end).toString(), 16);
				}
				else {
					entityValue = Integer.parseInt(input.subSequence(start, end).toString(), 10);
				}
			}
			catch (final NumberFormatException nfe) {
				return 0;
			}

			if (entityValue > 0xFFFF) {
				final char[] chrs = Character.toChars(entityValue);
				out.append(chrs[0]);
				out.append(chrs[1]);
			}
			else {
				out.append((char)entityValue);
			}

			return 2 + end - start + (isHex ? 1 : 0) + (semiNext ? 1 : 0);
		}
		return 0;
	}
}
