/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.dao.sql.criteria;

import nuts.core.dao.sql.SqlUtils;
import nuts.core.lang.Strings;

/**
 * OrderByExpression
 */
@SuppressWarnings("serial")
public class OrderExpression extends AbstractExpression {

	protected String column;
	protected String direction = "";

	/**
	 * Constructor
	 * 
	 * @param column column
	 */
	public OrderExpression(String column) {
		setColumn(column);
	}
	
	/**
	 * Constructor
	 * 
	 * @param column column
	 * @param direction direction
	 */
	public OrderExpression(String column, String direction) {
		setColumn(column);
		setDirection(direction);
	}
	
	/**
	 * @return the direction
	 */
	public String getDirection() {
		return direction;
	}

	/**
	 * @param direction the direction to set
	 */
	public void setDirection(String direction) {
		if (Strings.isNotBlank(direction)) {
			if (direction.compareToIgnoreCase(ASC) != 0 
					&& direction.compareToIgnoreCase(DESC) != 0) {
				throw new IllegalArgumentException("direction for [" + column + "] is not [" + ASC
						+ "] or [" + DESC + "]");
			}
			this.direction = direction;
		}
	}

	/**
	 * @return the column
	 */
	public String getColumn() {
		return column;
	}

	/**
	 * @param column the column to set
	 */
	public void setColumn(String column) {
		if (Strings.isBlank(column)) {
			throw new IllegalArgumentException("column cannot be blank string");
		}
		if (SqlUtils.isIllegalFieldName(column)) {
			throw new IllegalArgumentException("column [" + column + "] cannot contain any non-word character");
		}
		this.column = column;
	}

	/**
	 * Clone
	 * @throws CloneNotSupportedException if clone not supported
	 * @return Clone Object
	 */
	public Object clone() throws CloneNotSupportedException {
		return super.clone();
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((column == null) ? 0 : column.hashCode());
		result = prime * result + ((direction == null) ? 0 : direction.hashCode());
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		OrderExpression other = (OrderExpression) obj;
		if (column == null) {
			if (other.column != null)
				return false;
		}
		else if (!column.equals(other.column))
			return false;
		if (direction == null) {
			if (other.direction != null)
				return false;
		}
		else if (!direction.equals(other.direction))
			return false;
		return true;
	}
	
	/**
     * @return  a string representation of the object.
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		sb.append("{ ");
		sb.append("column: ").append(column);
		sb.append(", ");
		sb.append("direction: ").append(direction);
		sb.append(" }");

		return sb.toString();
	}
}
