/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.dao.sql;

import nuts.core.dao.DaoException;
import nuts.core.dao.DaoSession;
import nuts.core.dao.DataHandler;

import java.util.List;


/**
 */
public interface SqlDataAccessSession extends DaoSession {
	/**
	 * Insert is a bit different from other update methods, as it provides
	 * facilities for returning the primary key of the newly inserted row
	 * (rather than the effected rows). This functionality is of course
	 * optional.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * INSERT values.
	 * 
	 * @param statement The name of the statement to execute.
	 * @param parameter The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @return The primary key of the newly inserted row. This might be
	 *         automatically generated by the RDBMS, or selected from a sequence
	 *         table or other source.
	 * @throws DaoException If an error occurs.
	 */
	Object insert(String statement, Object parameter) throws DaoException;

	/**
	 * Update can also be used for any other update statement type, such as
	 * inserts and deletes. Update returns the number of rows effected.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * UPDATE values as well as the WHERE clause parameter(s).
	 * 
	 * @param statement The name of the statement to execute.
	 * @param parameter The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @return The number of rows effected.
	 * @throws DaoException If an error occurs.
	 */
	int update(String statement, Object parameter) throws DaoException;

	/**
	 * Delete returns the number of rows effected.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the DELETE statement.
	 * 
	 * @param statement The name of the statement to execute.
	 * @param parameter The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @return The number of rows effected.
	 * @throws DaoException If an error occurs.
	 */
	int delete(String statement, Object parameter) throws DaoException;
	
	/**
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the SELECT statement.
	 * 
	 * @param statement The name of the statement to execute.
	 * @param parameter The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @return The single result object populated with the result set data, or
	 *         null if no result was found
	 * @throws DaoException If more than one result was found, or if
	 *             any other error occurs.
	 */
	Object selectOne(String statement, Object parameter) throws DaoException;

	/**
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the SELECT statement.
	 * 
	 * @param statement The name of the statement to execute.
	 * @param parameter The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @return A List of result objects.
	 * @throws DaoException If an error occurs.
	 */
	List selectList(String statement, Object parameter) throws DaoException;

	/**
	 * Executes a mapped SQL SELECT statement that returns data to populate a
	 * number of result objects within a certain range.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the SELECT statement.
	 * 
	 * @param statement The name of the statement to execute.
	 * @param parameter The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @param offset The number of results to ignore.
	 * @param limit The maximum number of results to return.
	 * @return A List of result objects.
	 * @throws DaoException If an error occurs.
	 */
	List selectList(String statement, Object parameter, int offset, int limit)
			throws DaoException;
	
	/**
	 * This is generally a good approach to take when dealing with large sets of
	 * records (i.e. hundreds, thousands...) that need to be processed without
	 * eating up all of the system resources.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the SELECT statement.
	 * 
	 * @param statement The name of the statement to execute.
	 * @param parameter The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @param dataHandler A DataHandler instance
	 * @return data count
	 * @throws DaoException If an error occurs.
	 */
	int selectWithDataHandler(String statement, Object parameter,
			DataHandler dataHandler) throws DaoException;

	/**
	 * This is generally a good approach to take when dealing with large sets of
	 * records (i.e. hundreds, thousands...) that need to be processed without
	 * eating up all of the system resources.
	 * <p/>
	 * The parameter object is generally used to supply the input data for the
	 * WHERE clause parameter(s) of the SELECT statement.
	 * 
	 * @param statement The name of the statement to execute.
	 * @param parameter The parameter object (e.g. JavaBean, Map, XML
	 *            etc.).
	 * @param offset The number of results to ignore.
	 * @param limit The maximum number of results to return.
	 * @param dataHandler A DataHandler instance
	 * @return data count
	 * @throws DaoException If an error occurs.
	 */
	int selectWithDataHandler(String statement, Object parameter, 
			int offset, int limit,
			DataHandler dataHandler) throws DaoException;
}
