/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.castor.castors;

import nuts.core.bean.BeanHandler;
import nuts.core.castor.AbstractCastor;
import nuts.core.castor.CastContext;
import nuts.core.castor.Castor;
import nuts.core.castor.Castors;
import nuts.core.lang.Types;

import java.lang.reflect.Array;
import java.lang.reflect.Type;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;


public class MapCastor<S, T extends Map<?,?>> extends AbstractCastor<S, T> {
	private Castors castors;
	private Type keyType;
	private Type valType;
	private Castor keyCastor;
	private Castor valCastor;
	
	public MapCastor(Type fromType, Type toType, Castors castors) {
		super(fromType, toType);
		
		if (!Types.isAssignable(toType, Map.class)) {
			throw new IllegalArgumentException("The argument is not a map type: " + toType);
		}

		this.castors = castors;
		
		Type[] ts = Types.getMapKeyAndValueTypes(toType);
		this.keyType = ts[0];
		this.valType = ts[1];
	}

	@Override
	@SuppressWarnings("unchecked")
	protected boolean isAssignable(Object value) {
		if (super.isAssignable(value)) {
			if (isObjectType(keyType) && isObjectType(valType)) {
				return true;
			}

			Map m = (Map)value;
			if (m.isEmpty()) {
				return true;
			}

			boolean assignable = true;
			for (Entry en : (Set<Entry>)m.entrySet()) {
				Object key = en.getKey();
				if (key != null && !Types.isAssignable(key.getClass(), keyType)) {
					assignable = false;
					break;
				}

				Object val = en.getValue();
				if (val != null && !Types.isAssignable(val.getClass(), valType)) {
					assignable = false;
					break;
				}
			}
			return assignable;
		}
		return false;
	}

	@Override
	@SuppressWarnings("unchecked")
	protected T convertValue(Object value, CastContext context) {
		if (keyCastor == null) {
			keyCastor = castors.getCastor(keyType);
		}
		if (valCastor == null) {
			valCastor = castors.getCastor(valType);
		}
		
		Map map = createTarget();
		if (value.getClass().isArray()) {
			int size = Array.getLength(value);

			for (int i = 0; i < size; i++) {
				Object v = Array.get(value, i);
				if (v instanceof Entry) {
					Map.Entry kv = castChild(context, keyCastor, valCastor, String.valueOf(i), ((Entry)v).getKey(), ((Entry)v).getValue());
					if (kv != null) {
						map.put(kv.getKey(), kv.getValue());
					}
				}
				else {
					Map.Entry kv = castChild(context, keyCastor, valCastor, String.valueOf(i), v, null);
					if (kv != null) {
						map.put(kv.getKey(), kv.getValue());
					}
				}
			}
		}
		else if (value instanceof Map) {
			int i = 0;
			for (Entry en : (Set<Entry>)((Map)value).entrySet()) {
				Map.Entry kv = castChild(context, keyCastor, valCastor, String.valueOf(i++), en.getKey(), en.getValue());
				if (kv != null) {
					map.put(kv.getKey(), kv.getValue());
				}
			}
		}
		else if (value instanceof Iterable) {
			Iterator it = ((Iterable)value).iterator();
			for (int i = 0; it.hasNext(); i++) {
				Object v = it.next();
				if (v instanceof Entry) {
					Map.Entry kv = castChild(context, keyCastor, valCastor, String.valueOf(i), ((Entry)v).getKey(), ((Entry)v).getValue());
					if (kv != null) {
						map.put(kv.getKey(), kv.getValue());
					}
				}
				else {
					Map.Entry kv = castChild(context, keyCastor, valCastor, String.valueOf(i), v, null);
					if (kv != null) {
						map.put(kv.getKey(), kv.getValue());
					}
				}
			}
		}
		else {
			BeanHandler bh = castors.getBeanHandler(value.getClass());
			for (String key : bh.getReadPropertyNames()) {
				Object val = bh.getPropertyValue(value, key);
				Map.Entry kv = castChild(context, keyCastor, valCastor, key, key, val);
				if (kv != null) {
					map.put(kv.getKey(), kv.getValue());
				}
			}
		}
		return (T)map; 
	}
}
