/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.castor.castors;

import nuts.core.castor.AbstractCastor;
import nuts.core.castor.CastContext;
import nuts.core.castor.Castor;
import nuts.core.castor.Castors;
import nuts.core.lang.Types;

import java.lang.reflect.Array;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.Iterator;


public class CollectionCastor<T extends Collection<?>> extends AbstractCastor<Object, T> {
	protected Castors castors;
	protected Type toElementType;
	
	public CollectionCastor(Type fromType, Type toType, Castors castors) {
		super(fromType, toType);
		
		if (!Types.isAssignable(toType, Collection.class)) {
			throw new IllegalArgumentException("The argument is not a collection type: " + toType);
		}

		this.castors = castors;
		this.toElementType = Types.getCollectionElementType(toType);
	}

	@Override
	protected boolean isAssignable(Object value) {
		if (super.isAssignable(value)) {
			if (isObjectType(toElementType)) {
				return true;
			}

			Collection c = (Collection)value;
			if (c.isEmpty()) {
				return true;
			}

			boolean assignable = true;
			Iterator it = c.iterator();
			while (it.hasNext()) {
				Object v = it.next();
				if (v != null && !Types.isAssignable(v.getClass(), toElementType)) {
					assignable = false;
					break;
				}
			}
			return assignable;
		}
		return false;
	}

	private Type getFromComponentType() {
		if (Types.isArrayType(fromType)) {
			return Types.getArrayComponentType(fromType);
		}
		else if (Types.isAssignable(fromType, Collection.class)) {
			return Types.getCollectionElementType(fromType);
		}
		return Object.class;
	}

	@Override
	@SuppressWarnings("unchecked")
	protected T convertValue(Object value, CastContext context) {
		Collection coll = createTarget();
		if (value.getClass().isArray()) {
			Type fType = value.getClass().getComponentType();
			Castor castor = castors.getCastor(fType, toElementType);
			int size = Array.getLength(value);

			for (int i = 0; i < size; i++) {
				Object v = Array.get(value, i);
				v = castChild(context, castor, i, v);
				coll.add(v);
			}
		}
		else if (value instanceof Iterable) {
			Type fType = getFromComponentType();
			Castor castor = castors.getCastor(fType, toElementType);

			int i = 0;
			Iterator it = ((Iterable)value).iterator();
			while (it.hasNext()) {
				Object v = it.next();
				v = castChild(context, castor, i++, v);
				coll.add(v);
			}
		}
		else {
			Castor castor = castors.getCastor(value.getClass(), toElementType);
			value = castChild(context, castor, 0, value);
			coll.add(value);
		}
		return (T)coll; 
	}
}
