/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.bind.xml;

import nuts.core.bind.AbstractSerializer;
import nuts.core.bind.json.JsonException;
import nuts.core.lang.StringEscapes;
import nuts.core.lang.Strings;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;


public class XmlSerializer extends AbstractSerializer {
	private String rootName = "doc";
	private String itemName = "i";
	
	private List<String> nodes = new ArrayList<String>();
	
	/** current indent */
	private int indent;
	
	public XmlSerializer() {
	}

	/**
	 * @return the rootName
	 */
	public String getRootName() {
		return rootName;
	}

	/**
	 * @param rootName the rootName to set
	 */
	public void setRootName(String rootName) {
		this.rootName = rootName;
	}

	@Override
	protected void startDocument(Object src) {
		indent = 0;
		nodes.clear();
		writeBeginTag(rootName);
	}

	private void validateTagName(String tagName) {
		
	}
	
	private void writeBeginTag(String tagName) {
		validateTagName(tagName);
		nodes.add(tagName);

		try {
			writer.append('<');
			StringEscapes.escapeXml(tagName, writer);
			writer.append('>');
		}
		catch (IOException e) {
			throw new XmlException(e);
		}
	}

	private void writeEndTag() {
		String tagName = nodes.remove(nodes.size() - 1);
		try {
			writer.append("</");
			writer.append(tagName);
			writer.append('>');
		}
		catch (IOException e) {
			throw new XmlException(e);
		}
	}

	@Override
	protected void endDocument(Object src) {
		writeEndTag();
	}

	@Override
	protected void startArray(Object src) {
		indent += indentFactor;
	}

	@Override
	protected void endArray(Object src, int len) {
		indent -= indentFactor;
		if (len > 0) {
			writeIndent();
		}
	}

	@Override
	protected void startArrayElement(Object src, int index) {
		writeIndent();
		writeBeginTag(itemName);
	}
	
	@Override
	protected void endArrayElement(Object src, int index) {
		writeEndTag();
	}

	@Override
	protected void startObject(Object src) {
		indent += indentFactor;
	}
	
	@Override
	protected void endObject(Object src, int len) {
		indent -= indentFactor;
		if (len > 0) {
			writeIndent();
		}
	}

	@Override
	protected void startObjectProperty(String key, Object val, int index) {
		writeIndent();
		writeBeginTag(key);
	}
	
	@Override
	protected void endObjectProperty(String key, Object val, int index) {
		writeEndTag();
	}
	
	private void writeIndent() {
		try {
			writeIndent(indent);
		}
		catch (IOException e) {
			throw new JsonException(e);
		}
	}
	
	@Override
	protected void writeString(String str) {
		try {
			if (Strings.isEmpty(str)) {
				return;
			}
			
			StringEscapes.escapeXml(str, writer);
		}
		catch (IOException e) {
			throw new JsonException(e);
		}
	}

	@Override
	protected void writeNull() {
	}
	
	@Override
	protected void writeNumber(Number num) {
		write(num.toString());
	}
	
	@Override
	protected void writeBoolean(Boolean boo) {
		write(boo.toString());
	}
	
	private void write(String str) {
		try {
			writer.append(str);
		}
		catch (IOException e) {
			throw new JsonException(e);
		}
	}
}
