/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.bean.handlers;

import nuts.core.bean.Beans;
import nuts.core.bean.IncorrectAccessException;
import nuts.core.bean.NoSuchPropertyException;
import nuts.core.lang.Arrays;
import nuts.core.lang.Types;

import java.lang.reflect.Type;


/**
 * AbstractJavaBeanHandler
 */
public abstract class AbstractJavaBeanHandler<T> extends AbstractBeanHandler<T> {

	protected Type type;

	/**
	 * Constructor
	 * @param factory bean handler factory
	 * @param type bean type
	 */
	public AbstractJavaBeanHandler(Beans factory, Type type) {
		super(factory);
		this.type = type;
	}

	/**
	 * create bean object
	 * @return bean instance 
	 */
	public T createObject() {
		try {
			return Types.newInstance(Types.getDefaultImplType(type));
		}
		catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * is the property readable
	 * @param beanObject bean object (can be null)
	 * @param propertyName property name
	 * @return property type
	 */
	public boolean canReadProperty(T beanObject, String propertyName) {
		return Arrays.contains(getReadPropertyNames(beanObject), propertyName);
	}

	/**
	 * is the property writable
	 * @param beanObject bean object (can be null)
	 * @param propertyName property name
	 * @return property writable
	 */
	public boolean canWriteProperty(T beanObject, String propertyName) {
		return Arrays.contains(getWritePropertyNames(beanObject), propertyName);
	}
	
	/**
	 * @param propertyName property name
	 * @return NoSuchPropertyException
	 */
	protected RuntimeException noSuchPropertyException(String propertyName) {
		return new NoSuchPropertyException("Unknown property: " + propertyName + " [" + type + "].");
	}
	
	/**
	 * @param propertyName property name
	 * @return No getter method Exception
	 */
	protected RuntimeException noGetterMethodException(String propertyName) {
		return new IncorrectAccessException("No getter method for property: " 
				+ propertyName + " [" + type + "].");
	}
	
	/**
	 * @param propertyName property name
	 * @return No setter method Exception
	 */
	protected RuntimeException noSetterMethodException(String propertyName) {
		return new IncorrectAccessException("No setter method for property: " 
				+ propertyName + " [" + type + "].");
	}
	
	/**
	 * @return a string representation of the object.
	 */
	public String toString() {
		return getClass().getName() + ": " + type;
	}
	
}
