/*
 * This file is part of Nuts Framework.
 * Copyright(C) 2009-2012 Nuts Develop Team.
 *
 * Nuts Framework is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Nuts Framework is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts Framework. If not, see <http://www.gnu.org/licenses/>.
 */
package nuts.core.bean;

import java.lang.reflect.Type;

/**
 * PropertyHandler
 */
public interface PropertyHandler<T> {
	/**
	 * get read property names
	 * @param beanObject bean object (can be null)
	 * @return property names
	 */
	String[] getReadPropertyNames(T beanObject);

	/**
	 * get read property names
	 * @return property names
	 */
	String[] getReadPropertyNames();

	/**
	 * get write property names
	 * @return property names
	 */
	String[] getWritePropertyNames();

	/**
	 * get write property names
	 * @param beanObject bean object (can be null)
	 * @return property names
	 */
	String[] getWritePropertyNames(T beanObject);
	
	/**
	 * get property type
	 * @param propertyName property name
	 * @return property type
	 */
	Type getPropertyType(String propertyName);

	/**
	 * get property type
	 * @param beanObject bean object (can be null)
	 * @param propertyName property name
	 * @return property type
	 */
	Type getPropertyType(T beanObject, String propertyName);

	/**
	 * is the property readable
	 * @param propertyName property name
	 * @return true if property is readable
	 */
	boolean canReadProperty(String propertyName);

	/**
	 * is the property readable
	 * @param beanObject bean object (can be null)
	 * @param propertyName property name
	 * @return true if property is readable
	 */
	boolean canReadProperty(T beanObject, String propertyName);
	
	/**
	 * is the property writable
	 * @param beanName bean name
	 * @return true if property is writable
	 */
	boolean canWriteProperty(String propertyName);
	
	/**
	 * is the property writable
	 * @param beanObject bean object (can be null)
	 * @param beanName bean name
	 * @return true if property is writable
	 */
	boolean canWriteProperty(T beanObject, String propertyName);
	
	/**
	 * get property value 
	 * @param beanObject bean object
	 * @param propertyName property name
	 * @return property value
	 */
	Object getPropertyValue(T beanObject, String propertyName);
	
	/**
	 * set property value 
	 * @param beanObject bean object
	 * @param propertyName property name
	 * @param value value
	 */
	void setPropertyValue(T beanObject, String propertyName, Object value);
}
